from adcsng.utils.files import set_path
from adcsng.utils.files import dirlist
import os

class MissionDir(object):
    def __init__(self, config, mission='', root_dir='', source='', output='',
                 kernels='', processed='', temp='', templates='', log='',
                 exec_log = '', executables='', reports=''):
        '''
        Class that provides all the relevant directories for the adcsng
        execution

        :param mission: Mission for which adcsng is executed
        :type mission: str
        :type root_dir: The adcsng execution directory
        :param source: str
        :type source: Directory where the source files are -incoming directory-
        :param output: Directory where the generated kernels are to be placed
        :type output: str
        :param kernels: Mission Kernel dataset directory for inputs
        :type kernels: str
        :param processed: Directory where the already processed input is stored
        :type processed: str
        :param temp: Temporary directory for temporary files created during execution
        :type temp: str
        :param templates: Directory where the templates for NAIF utilities are stored
        :type templates: str
        '''
        self.mission = mission
        self.root_dir = root_dir
        self.source = source
        self.output = output
        self.kernels =kernels
        self.processed = processed
        self.temp = temp
        self.templates = templates
        self.executables = executables
        self.reports = reports
        self.log = log
        self.exec_log = exec_log


        kernels_sub_dirs = ['ck','spk','lsk','sclk','pck','ik','fk', 'mk']
        output_sub_dirs = ['ck','sclk','mk','spk']
        execution_dirs = [source, output, kernels, temp, processed, log]

        #
        # If the string is empty, it means that we do not want to move the
        # source files into the processed directory
        #
        if processed:
            processed_sub_dirs = dirlist(config)
            for subdir in processed_sub_dirs:
                execution_dirs.append(os.path.join(processed, subdir))

        for subdir in kernels_sub_dirs:
            execution_dirs.append(os.path.join(kernels, subdir))
        for subdir in output_sub_dirs:
            execution_dirs.append(os.path.join(output, subdir))

        try:
            execution_dirs.remove('')
        except:
            pass
        sdir_flag = False
        log_message = ''
        try:
            for dir in execution_dirs:
                if not os.path.exists(dir):
                    for sdir in processed_sub_dirs:
                        if sdir in dir:
                            try:
                                os.mkdir(dir)
                                log_message += '   Directory {} did not exist but has been generated.\n'.format(sdir)
                            except:
                                pass
                            sdir_flag = True
                    if not sdir_flag:
                        if not os.path.isdir(dir):
                            raise ValueError('ADCSng directory {} specified in the configuration does not exist.'.format(dir))

                    sdir_flag = False
        except:
            pass

        if log_message:
            print('+ + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + ')
            print('ADCSng run preparation:')
            for line in log_message.split('\n')[:-1]:
                print(line)
            print('+ + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + \n')



    def __setattr__(self, key, value):
        if key in ['source', 'output', 'kernels', 'processed', 'temp', 'templates', 'executables', 'log', 'exec_log','reports']:
            value = set_path(self.root_dir, value)
            super(MissionDir, self).__setattr__(key, value)
        elif key in ['mission', 'root_dir']:
            super(MissionDir, self).__setattr__(key, value)
        else:
            raise AttributeError('Invalid attribute: ' + key)

    def __getattr__(self, item):
        if item in ['mission', 'root_dir','source', 'output', 'kernels', 'processed', 'temp', 'templates','log','exec_log','reports']:
            return self.__dict__[item]
        else:
            raise AttributeError('Invalid attribute: ' + item)

    def __str__(self):
        mission_dir = "ADCSng directory structure for " + self.mission + '\n'
        mission_dir += "   source: " + self.source + '\n'
        mission_dir += "   kernel dataset dir: " + self.kernels + '\n'
        mission_dir += "   kernel destination: " + self.output + '\n'
        mission_dir += "   source destination: " + self.processed + '\n'
        mission_dir += "   working dir: " + self.temp + '\n'
        mission_dir += "   templates dir: " + self.templates
        mission_dir += "   executables dir: " + self.executables
        mission_dir += "   reports dir: " + self.reports

        return mission_dir