import os
import re
import glob
import shutil
import filecmp
import tarfile
import subprocess
import numpy.testing as npt

from adcsng.utils.files import print_html
from adcsng.utils.files import get_exe_dir
from adcsng.command_line import main as adcsng


def runADCSng(version, debug=False, log=False, untar=True):

    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    if untar:
        tar = tarfile.open('incoming.tar.gz')
        tar.extractall()
        tar.close()

    shutil.copy('ker_dir/mk/former_versions/MEX_OPS_V100_20200623_000.TM', 'ker_gen/mk/')

    print_html(f"Executing ADCSng v{version} for Mars-Express")
    adcsng('mars_express.json', debug=debug, log=log)

def cleanup(initial=False):

    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    try:
        os.remove('mex_test.lock')
    except:
        pass

    try:
        shutil.rmtree('incoming')
        os.remove('._incoming')
    except:
        pass

    try:
        shutil.rmtree('ancdr/soc')
    except:
        pass

    if not initial:
        for CleanUp in glob.glob('ker_gen/**/*.*', recursive=True):
            if 'ATNM_T6_P170331235152_01566.BC' in CleanUp or 'MEX_OPS_V100_20200623_000.TM' in CleanUp:
                os.remove(CleanUp)
            elif not CleanUp.endswith('aareadme.txt') and not 'former_versions' in CleanUp:
                    os.remove(CleanUp)
    else:
        for CleanUp in glob.glob('ker_gen/**/*.*', recursive=True):
            if not CleanUp.endswith('aareadme.txt') and not 'MEX_200422_STEP.TSC' in CleanUp and not 'ATNM_P170331235152_01580.BC' in CleanUp:
                    os.remove(CleanUp)

    for CleanUp in glob.glob('ker_gen/**/*.*', recursive=True):
        if not 'former_versions' in CleanUp and not CleanUp.endswith('aareadme.txt'):
                shutil.move(CleanUp, ''.join(CleanUp.split('/former_versions')))

    for CleanUp in glob.glob('ancdr/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt') and not 'TCP' in CleanUp:
            os.remove(CleanUp)

    for CleanUp in glob.glob('log/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('temp/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt') or not CleanUp.endswith('temp.log'):
            try:
                shutil.rmtree(CleanUp)
            except:
                if 'aareadme.txt' not in CleanUp:
                    os.remove(CleanUp)


def SPKcompare():

    spks = glob.glob('ker_gen/spk/*.BSP')
    for spk in spks:

        exe_dir = get_exe_dir()
        spk_1 = spk
        spk_2 = f'ker_val/spk/{spk.split(os.sep)[-1]}'

        if not os.path.exists(spk_1): raise NameError(f"SPK Kernel {spk_1} does not exist")
        if not os.path.exists(spk_2): raise NameError(f"SPK Kernel {spk_2} does not exist")

        command = f'../..{exe_dir}/spkdiff  {spk_1}  {spk_2}'
        print_html(command)


        command_process = subprocess.Popen(command, shell=True,
                                           stdout=subprocess.PIPE,stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)
        if 'Toolkit version:' in text:
            raise NameError(f"SPKDIFF did not work with SPK Kernels {spk}")

        try:
            text = text.split('\n')
            for line in text:
                if 'Position:' in line or 'Velocity:' in line:
                    npt.assert_almost_equal(float(line.split()[1]), 0.0, decimal=10)
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=10)
                if 'Position (km)' in line or 'Velocity (km/s)' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=10)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=10)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"SPK Kernels {spk} are different")


def CKcompare():

    kernels =  'ker_dir/fk/MEX_V14.TF ' + \
               'ker_dir/lsk/NAIF0012.TLS ' + \
               'ker_dir/sclk/MEX_200422_STEP.TSC'

    cks = {'1': {'ck': 'ATNM_MEASURED_180313_180313_V01.BC',
              'fframe': 'MEX_SC_REF',
              'tframe': 'J2000',
              'start':  '2018-MAR-13 00:10:15 TDB',
              'finish': '2018-MAR-13 07:50:31 TDB'},
           '2': {
               'ck':     'ATNM_P170331235152_01588.BC',
               'fframe': 'MEX_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2017-MAR-31 23:55:51 TDB',
               'finish': '2017-APR-08 21:50:00 TDB'},
           '3': {
               'ck':     'ATNM_T6_P170331235152_01588.BC',
               'fframe': 'MEX_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2017-MAR-31 23:55:51 TDB',
               'finish': '2017-APR-08 21:50:00 TDB'},
           '4': {
               'ck': 'ATNM_SGS_P170331235152_01588.BC',
               'fframe': 'MEX_SPACECRAFT',
               'tframe': 'J2000',
               'start': '2017-MAR-31 23:55:51 TDB',
               'finish': '2017-APR-06 21:50:00 TDB'},
           '5': {
               'ck':     'former_versions/ATNM_T6_P170331235152_01566.BC',
               'fframe': 'MEX_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2017-MAR-31 23:55:51 TDB',
               'finish': '2017-APR-08 21:50:00 TDB'},
           '6': {
               'ck':     'ATNM_PTR01298_200321_001.BC',
               'fframe': 'MEX_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2020-MAR-21 16:29:44 TDB',
               'finish': '2020-MAR-21 21:21:06 TDB'},
           '7': {
               'ck': 'ATNM_PTR01300_200418_002.BC',
               'fframe': 'MEX_SPACECRAFT',
               'tframe': 'J2000',
               'start': '2020-APR-18 00:59:44 TDB',
               'finish': '2020-APR-18 06:59:06 TDB'},
           '8': {
               'ck': 'ATNM_PTR01302_200516_001.BC',
               'fframe': 'MEX_SPACECRAFT',
               'tframe': 'J2000',
               'start': '2020-MAY-16 13:20:44 TDB',
               'finish': '2020-MAY-16 19:20:06 TDB'},
           '9': {
               'ck':     'MEX_SA_2018_V0011.BC',
               'fframe': 'MEX_SA+Y_ZERO',
               'tframe': 'MEX_SA+Y',
               'start':  '2018-APR-14 00:05:27 TDB',
               'finish': '2018-APR-15 00:02:07 TDB'},
           '10': {
               'ck':     'MEX_PLANNED_MAPPS_201127_201127_S200604_V01.BC',
               'fframe': 'MEX_SC_REF',
               'tframe': 'J2000',
               'start':  '2020-NOV-27 19:59:30 TDB',
               'finish': '2020-NOV-27 20:01:30 TDB'}
           }

    if len(cks) < len(os.listdir("ker_gen/ck/")) - 2:
        raise NameError("WARNING: some kernels in ker_gen are not being tested")

    for ck in cks.items():
        ck = ck[1]
        exe_dir = get_exe_dir()
        ck_1 = f"ker_val/ck/{ck['ck']}"
        ck_2 = f"ker_gen/ck/{ck['ck']}"

        if not os.path.exists(ck_1): raise NameError(f"CK Kernel {ck_1} does not exist")
        if not os.path.exists(ck_2): raise NameError(f"CK Kernel {ck_2} does not exist")

        command = f"../..{exe_dir}/frmdiff -k {kernels} -f1 {ck['fframe']} " \
                  f"-t1 {ck['tframe']} -b {ck['start']} -e {ck['finish']} " \
                  f" {ck_1} {ck_2}"
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                                stdout=subprocess.PIPE,
                                                stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)
        if 'version: N0066' in text:
            raise NameError(f"FRMDIFF did not work with CK Kernels {ck['ck']}: {text}")

        #
        # Whilst with the SPKs and SCLKs it is enough to compare the DIFF
        # files, for PREDICKT the mantissa and the precision differ from
        # Linux to MacOS:
        #
        #   OS:    Rotation (rad):             +4.8013978590743E-16      +1.4316667159571E-16
        #   LINUX: Rotation (rad):             +0.0000000000000E+00      +0.0000000000000E+00
        #
        # Therefore we will use an assertion instead of a file comparison
        #
        try:
            text = text.split('\n')
            for line in text:
                if 'Rotation (rad):' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=4)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=4)
        except AssertionError as error:
            print(error)
            raise NameError(f"CK Kernels {ck['ck']} are different")

    return


def ORBNUMcompare():

    orbnums = glob.glob('ker_gen/orbnum/*.ORB')
    for orb in orbnums:

        reldir = f"/orbnum/{orb.split(os.sep)[-1]}"

        if not filecmp.cmp('ker_gen'+reldir, 'ker_val'+reldir):
            raise NameError(f"ORBNUM Files {orb} are different")
        else:
            with open('ker_gen' + reldir, 'r') as fil:
                print_html(fil.read())

    return


def SCLKcompare():

    begin = '2003-09-01'
    finish = '2020-01-01'
    id =  '-41'

    lsk = 'ker_dir/lsk/naif0012.tls'
    sclks = glob.glob('ker_gen/sclk/*.tsc')
    for sclk in sclks:

        sclk = sclk.split(os.sep)[-1]
        exe_dir = get_exe_dir()
        command = f"../..{exe_dir}/diffsclk {lsk} {begin} {finish} " \
                  f"864000 {id} ker_val/sclk/{sclk} ker_gen/sclk/{sclk}"
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                           stdout=subprocess.PIPE,
                                           stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)

        try:
            text = text.split('\n')
            for line in text:
                if line.strip():
                        npt.assert_almost_equal(float(line.split()[1]), 0.0,
                                            decimal=5)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"SCLK Kernels {sclk} are different")

    return


def MKcompare():

    mks = [f for f in os.listdir('ker_gen/mk/') if
           re.search(r'.*[^0-9].TM', f)]

    if not mks:
        raise NameError("No MK Kernels available")

    for mk in mks:
        data_gen = []
        record = False
        with open(f'ker_gen/mk/{mk}', 'r') as m:
            for line in m:
                if record: data_gen.append(line)
                if '\\begindata' in line: record = True
                if '\\begintext' in line: break

        print_html(''.join(data_gen))

        with open(f'ker_val/mk/{mk}', 'r') as m: data_val = m.readlines()

        if not set(data_gen).issubset(data_val):
            raise NameError("MK Kernels are different")

    return


def GAPcompare():

    gaps = [f for f in os.listdir('ker_gen/reports/') if re.search(r'.*gap', f)]
    if not gaps: gaps = [f for f in os.listdir('ker_gen/reports/') if re.search(r'.*GAP', f)]

    if not gaps:
        raise NameError("No GAP files available")

    for gap in gaps:
        with open(f'ker_gen/reports/{gap}', 'r') as m: gap_gen = m.readlines()[2:]
        with open(f'misc/reports/{gap}', 'r') as m: gap_val = m.readlines()[2:]
        for line in gap_gen: print_html(line)

        if not gap_gen == gap_val:
            raise NameError("GAP files are different")

    return



if __name__ == '__main__':
    runADCSng('1.0.0', debug=True, log=True, untar=True)
    #os.chdir(os.getcwd() + '/..')
    #SPKcompare()
    #CKcompare()
    #GAPcompare()