import os
import re
import glob
import shutil
import tarfile
import subprocess
import numpy.testing as npt

from adcsng.utils.files import print_html
from adcsng.utils.files import get_exe_dir
from adcsng.command_line import main as adcsng


def runADCSng(version, debug=False, log=False, untar=True):

    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    if untar:
        tar = tarfile.open('incoming.tar.gz')
        tar.extractall()
        tar.close()

    #
    # To test meta-kernel update in former versions
    #
    for file in glob.glob(r'ker_dir/mk/former_versions/*.tm'):
        shutil.copy(file, 'ker_gen/mk/former_versions')

    print_html(f"Executing ADCSng v{version} for JUICE")
    adcsng('juice.json', debug=debug, log=log)


def cleanup():
    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    try:
        os.remove('*.lock')
    except:
        pass

    try:
        shutil.rmtree('incoming')
        os.remove('._incoming')
    except:
        pass

    for CleanUp in glob.glob('ker_gen/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('ker_gen/**/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('ancdr/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('log/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('temp/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt') or not CleanUp.endswith('temp.log'):
            try:
                shutil.rmtree(CleanUp)
            except:
                if 'aareadme.txt' not in CleanUp:
                    os.remove(CleanUp)


def SPKcompare():

    spks = glob.glob('ker_gen/spk/*.bsp')
    for spk in spks:

        exe_dir = get_exe_dir()
        spk_1 = spk
        spk_2 = f'ker_val/spk/{spk.split(os.sep)[-1]}'

        if not os.path.exists(spk_1): raise NameError(f"SPK Kernel {spk_1} does not exist")
        if not os.path.exists(spk_2): raise NameError(f"SPK Kernel {spk_2} does not exist")

        command = f'../..{exe_dir}/spkdiff  {spk_1}  {spk_2}'
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                           stdout=subprocess.PIPE, stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)
        if 'Toolkit version:' in text:
            raise NameError(f"SPKDIFF did not work with SPK Kernels {spk}")

        try:
            text = text.split('\n')
            for line in text:
                if 'Position:' in line or 'Velocity:' in line:
                    npt.assert_almost_equal(float(line.split()[1]), 0.0, decimal=10)
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=10)
                if 'Position (km)' in line or 'Velocity (km/s)' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=10)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=10)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"SPK Kernels {spk} are different")


def CKcompare():

    kernels =  'ker_dir/fk/juice_v20.tf ' + \
               'ker_dir/lsk/naif0012.tls '

    cks = {'1':{'ck': 'juice_sa_default_s20160326_v01.bc',
                'fframe': 'JUICE_SA+Y',
                'tframe': 'JUICE_SA+Y_ZERO',
                'start':  '2032-JUL-02 00:01:10 TDB',
                'finish': '2032-JUL-04 00:01:08 TDB',
                'sclk': 'ker_dir/sclk/juice_step_20160326.tsc ' },
           '2': {'ck': 'juice_sa_default_s20160326_v01.bc',
                 'fframe': 'JUICE_SA-Y',
                 'tframe': 'JUICE_SA-Y_ZERO',
                 'start': '2032-JUL-02 00:01:10 TDB',
                 'finish': '2032-JUL-04 00:01:08 TDB',
                 'sclk': 'ker_dir/sclk/juice_step_20160326.tsc '}
           }

    if len(cks) < len(os.listdir("ker_gen/ck/")):
        raise NameError("WARNING: some kernels in ker_gen are not being tested")


    for ck in cks.items():
        ck = ck[1]
        exe_dir = get_exe_dir()

        ck_1 = f"ker_gen/ck/{ck['ck']}"
        ck_2 = f"ker_val/ck/{ck['ck']}"

        if not os.path.exists(ck_1): raise NameError(f"CK Kernel {ck_1} does not exist")
        if not os.path.exists(ck_2): raise NameError(f"CK Kernel {ck_2} does not exist")

        command = f"../..{exe_dir}/frmdiff -k {kernels}{ck['sclk']} -f1 {ck['fframe']} " \
                  f"-t1 {ck['tframe']} -b {ck['start']} -e {ck['finish']} " \
                  f" {ck_1} {ck_2}"
        print_html(command)
        command_process = subprocess.Popen(command, shell=True, stdout=subprocess.PIPE,
                                                stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)
        if 'version: N0066' in text:
            raise NameError(f"FRMDIFF did not work with CK Kernels {ck['ck']}")

        #
        # Whilst with the SPKs and SCLKs it is enough to compare the DIFF
        # files, for PREDICKT the mantissa and the precision differ from
        # Linux to MacOS:
        #
        #   OS:    Rotation (rad):             +4.8013978590743E-16      +1.4316667159571E-16
        #   LINUX: Rotation (rad):             +0.0000000000000E+00      +0.0000000000000E+00
        #
        # Therefore we will use an assertion instead of a file comparison
        #
        try:
            text = text.split('\n')
            for line in text:
                if 'Rotation (rad):' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=10)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=10)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"CK Kernels {ck['ck']} are different")

    return


def MKcompare():

    mks = [f for f in os.listdir('ker_gen/mk/') if
           re.search(r'.*[^0-9].tm', f)]

    if not mks:
        raise NameError("No MK Kernels available")

    for mk in mks:
        data_gen = []
        record = False
        with open(f'ker_gen/mk/{mk}', 'r') as m:
            for line in m:
                if record: data_gen.append(line)
                if '\\begindata' in line: record = True
                if '\\begintext' in line: break

        print_html(''.join(data_gen))

        with open(f'ker_val/mk/{mk}', 'r') as m: data_val = m.readlines()

        if not set(data_gen).issubset(data_val):
            raise NameError("MK Kernels are different")

    return


def GAPcompare():

    gaps = [f for f in os.listdir('ker_gen/reports/') if
           re.search(r'.*gap', f)]

    if not gaps:
        raise NameError("No GAP files available")

    for gap in gaps:
        with open(f'ker_gen/reports/{gap}', 'r') as m: gap_gen = m.readlines()[2:]
        with open(f'misc/reports/{gap}', 'r') as m: gap_val = m.readlines()[2:]
        for line in gap_gen: print_html(line)

        if not gap_gen == gap_val:
            raise NameError("GAP files are different")

    return


if __name__ == '__main__':
    cleanup()
    runADCSng('1.0.0', debug=True, log=True, untar=True)
    os.chdir(os.getcwd() + '/..')
    SPKcompare()
    CKcompare()
    #MKcompare()