C$Procedure      FITCK3 ( Fit a orientations with a type 3 C-kernel )
 
      SUBROUTINE FITCK3 ( CAVSRC, REPORT, SPICE, TOL,   SCID,
     .                    HANDLE, BEGET,  ENDET, PLTFRM,  REF, SEGID )
 
C$ Abstract
C
C     Given a routine that will provide quaternions and angular
C     velocity of some structure with respect to some reference
C     frame, construct one or more type 3 C-kernel segmenst that
C     agrees with the source data to a user specified tolerance.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C      None.
C
C$ Keywords
C
C      CK
C
C$ Declarations
 
      IMPLICIT NONE
      EXTERNAL              CAVSRC
      EXTERNAL              REPORT
      LOGICAL               SPICE
      DOUBLE PRECISION      TOL
      INTEGER               SCID
      INTEGER               HANDLE
      DOUBLE PRECISION      BEGET
      DOUBLE PRECISION      ENDET
      INTEGER               PLTFRM
      CHARACTER*(*)         REF
      CHARACTER*(*)         SEGID
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     CAVSRC     I   is a routine that given an ET will produce Q and AV
C     REPORT     I   is a routine that reports progress of the work.
C     SPICE      I   TRUE if CAVSRC has proper exception handling
C     TOL        I   is the accuracy required of the output C-kernel
C     SCID       I   is the SCLK id-code to use.
C     HANDLE     I   is the handle of a C-kernel open for writing
C     BEGET      I   is the beginning ET for the C-kernel
C     ENDET      I   is the ending ET for the C-kernel
C     PLTFRM     I   is the id-code for the platform being modelled
C     REF        I   is the reference frame to associate with the CK
C     SEGID      I   is the string to attach to each segment created
C
C$ Detailed_Input
C
C     CAVSRC     is a routine that given an ET will produce Q and AV.
C                The calling sequence should be:
C
C                   SUBROUTINE CAVSRC ( ET, CMATRX, AV )
C
C                the only input is ET.  The outputs are the CMATRX
C                and AV (C-matrix and Angular Velocity)
C                to associate with the structure specified by PLTFRM
C                relative to the reference frame specified by REF.
C
C                It is probably a good design practice to create an
C                umbrella routine that contains an initialization
C                entry point and the entry point that performs the
C                function of CAVSRC.  In this way, you can set up any
C                particular parameters needed by CAVSRC prior to calling
C                this routine.  When CAVSRC is called by FITCK3, it
C                can simply look up these parameters from the variables
C                shared by the initialization entry point and the
C                entry point corresponding to CAVSRC.
C
C     REPORT     is a routine that given the last ET processed by FITCK3
C                will produce a suitable progress report.  The calling
C                sequence to REPORT should be
C
C                    SUBROUTINE REPORT ( ET )
C
C                As with CAVSRC, this assumes that REPORT is an entry
C                point in a routine that contains another initialization
C                entry point that sets up the parameters needed for
C                producing a meaningful report.
C
C                If you don't care about progress reports in your
C                application, just make a stub routine that simply
C                returns without taking any action.
C
C
C     SPICE      is a flag indicating that CAVSRC make proper use of
C                SPICE exeption handling facilities and does not modify
C                its input value of ET.
C
C     TOL        is the accuracy required of the output C-kernel.  The
C                output C-kernel will match CAVSRC at every ET to the
C                accuracy specified by TOL.  TOL is an angle in
C                radians.  The orientation of the C-kernel will match
C                the outputs of CAVSRC to within and angle of TOL.
C                The angular velocity vectors will also be within
C                TOL radians of being parallel.
C
C     SCID       is the SCLK id to use when converting ET to ticks.
C                Normally this should be the id-code of the spacecraft
C                that the structure is mounted to.  However, in some
C                cases, another clock id may be approrpriate.
C
C     HANDLE     is the handle of a C-kernel that has been opened
C                for writing in which FITCK3 will insert new C-kernel
C                segments.
C
C     BEGET      is the initial epoch for the C-kernel segment(s) that
C                will be produced by this routine.
C
C     ENDET      is the last epoch for the C-kernel segment(s) that
C                will be produced by this routine.
C
C     PLTFRM     is the ID-code of the object whose orientation is to
C                be captured in the C-kernel.
C
C     REF        is the reference frame to associate with the C-kernel
C                segments created by this routine.
C
C     SEGID      is the segment identifier to be used for segments
C                created by this routine.  Note that the same segment
C                identifier will be used for all segments that are
C                created by this routine.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     1)  If the routine CAVSRC modifies the input value of ET that
C         it receives without changing resetting it before returning,
C         the error  'SPICE(BADPASSEDROUTINE)' is signalled.
C
C     2)  Any problems that arise the construction of the new
C         C-kernel data are diagnosed by routines called by this
C         routine.
C
C$ Particulars
C
C     This routine is intended to handle the normal logic needed
C     when creating a C-kernel that fits some continuous source
C     of orientation data.
C
C$ Examples
C
C     Later.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 4.0.0, 02-AUG-2000 (WLT)
C
C
C        The calling sequence was modified to take in information about
C        the confidence in the routine CAVSRC complying with SPICE
C        expectations.
C
C        In addition the mechanism for choosing a step size was greatly
C        simplified.  A fixed increment is now used and the stepsize is
C        adjusted by either just adding multiples (including negative
C        multiples) of the increment or by dividing the step size by the
C        square root of 2 (in rare cases where the default increment is
C        just too big).
C
C        Finally, since a finer test criterion is used, matrices and
C        quaternions are buffered so as to avoid lookups whenever
C        possible.
C
C-    SPICELIB Version 3.0.0, 25-FEB-1998 (WLT)
C
C        Added code to short circuit SPICELIB error handling
C        so that extra information can be added to the long
C        message if one is signalled in the call to CAVSRC.
C
C        In addition if the value of ET supplied to CAVSRC
C        comes back changed, and error is signalled indicating
C        that CAVSRC is not performing to specification.
C
C-    SPICELIB Version 2.1.0, 25-FEB-1998 (WLT)
C
C        Added aliasing checks and checks to make sure the
C        initial STEP away from BEGET is not too large.
C
C-    SPICELIB Version 2.0.0, 30-JAN-1998 (WLT)
C
C        Reworked the basic logic of the routine with the intent
C        of making the meaning of various branches clearer
C
C-    SPICELIB Version 1.0.0, 24-JUL-1997 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Fit type 3 C-kernel to a continuous source of orientation
C
C-&
C
C     SPICELIB Functions
C
      DOUBLE PRECISION      VNORM
      DOUBLE PRECISION      PI
 
      INTEGER               BSRCHD
      INTEGER               LSTLED
 
      LOGICAL               RETURN
      LOGICAL               FAILED
 
C
C     Parameters
C
C
C     NCHECK is the number of minimum number of interpolation checks we
C     perform per step.
C
      INTEGER               NCHECK
      PARAMETER           ( NCHECK = 7 )
 
      DOUBLE PRECISION      MNPTS
      PARAMETER           ( MNPTS = 20.0D0 )
C
C     STPMIN and STPMAX are the adjustable parameters (setable in the
C     entry points SETMIN and SETMAX that adjust how small and how
C     large a step can be made when sampling matrices.  The constant
C     DSIZE is used to determine that there is a discontinuity in the
C     computation of the orientation.
C
      DOUBLE PRECISION      STPMAX
      DOUBLE PRECISION      STPMIN
      DOUBLE PRECISION      DSIZE
      SAVE                  DSIZE
      SAVE                  MAXSTP
      SAVE                  MINSTP
 
      INTEGER               WDSIZE
      PARAMETER           ( WDSIZE = 32 )

      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 80 )
C
C     The variables below make up the local time tagged buffer of
C     matrices and angular velocity vectors.  These are used to
C     assist in efficiency when testing the quality of the interpolation
C     across a mesh of test points.
C
 
      INTEGER               BUFMAX
      PARAMETER           ( BUFMAX = 1000 )
 
      DOUBLE PRECISION      AVLBUF ( 3, BUFMAX )
      DOUBLE PRECISION      CMLBUF ( 9, BUFMAX )
      DOUBLE PRECISION      ETLBUF (    BUFMAX )
      INTEGER               LBPNTR (    BUFMAX )
      INTEGER               LBUFFD
 
C
C     QBUFSZ is the number of quaternians we buffer before writing
C     a segment.
C
      INTEGER               QBUFSZ
      PARAMETER           ( QBUFSZ = 10000 )
 
      DOUBLE PRECISION      AVEL   ( 3, QBUFSZ )
      DOUBLE PRECISION      Q      ( 4, QBUFSZ )
      DOUBLE PRECISION      TICKS  (    QBUFSZ )
 
      INTEGER               QCOUNT
      INTEGER               TQ
 
C
C     Local Utility Variables
C
      CHARACTER*(WDSIZE)    TIME1
      CHARACTER*(WDSIZE)    TIME2
      CHARACTER*(LNSIZE)    LINE
 
      DOUBLE PRECISION      ANGLE
 
      DOUBLE PRECISION      AV     ( 3 )
      DOUBLE PRECISION      AVAPRX ( 3 )
      DOUBLE PRECISION      AVFRAC ( 3 )
      DOUBLE PRECISION      AVLAST ( 3 )
      DOUBLE PRECISION      AVSTEP ( 3 )
      DOUBLE PRECISION      AVTMP  ( 3 )
      DOUBLE PRECISION      AXIS   ( 3 )
      DOUBLE PRECISION      BEGTIK
      DOUBLE PRECISION      CMAPRX ( 3, 3 )
      DOUBLE PRECISION      CMATRX ( 3, 3 )
      DOUBLE PRECISION      CMFRAC ( 3, 3 )
      DOUBLE PRECISION      CMLAST ( 3, 3 )
      DOUBLE PRECISION      CMSTEP ( 3, 3 )
      DOUBLE PRECISION      DELTA
      DOUBLE PRECISION      ENDTIK
      DOUBLE PRECISION      ENUF
      DOUBLE PRECISION      ETFRAC
      DOUBLE PRECISION      ETFRST
      DOUBLE PRECISION      ETLAST
      DOUBLE PRECISION      ETSTEP
      DOUBLE PRECISION      FRAC
      DOUBLE PRECISION      INCR
      DOUBLE PRECISION      MAXSTP
      DOUBLE PRECISION      MESH
      DOUBLE PRECISION      MINSTP
      DOUBLE PRECISION      RDIFF  ( 3, 3 )
      DOUBLE PRECISION      REMAIN
      DOUBLE PRECISION      STARTS (2)
      DOUBLE PRECISION      STEP
 
 
      INTEGER               GET
      INTEGER               LESIZE
      INTEGER               LPSIZE
      INTEGER               NINTS
      INTEGER               POINTS
      INTEGER               PUT
      INTEGER               USE
 
      INTEGER               HITS
      INTEGER               LOOKUP
 
 
      LOGICAL               AVFLAG
      LOGICAL               BAD
      LOGICAL               GOTONE
      LOGICAL               OK
      LOGICAL               TOOBIG
 
      DATA                  STPMIN / 2.0D1 /
      DATA                  STPMAX / 1.0D4 /
      DATA                  DSIZE  / 1.0D0 /
 
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'FITCK3' )
C
C     Initialize information about what is currently buffered.
C
 
      LBUFFD = 0
      QCOUNT = 0
      TQ     = 0
      INCR   = STPMIN
 
      HITS   = 0
      LOOKUP = 0
C
C     We don't have a bracketing C-matrix yet.
C
      GOTONE = .FALSE.
 
C
C     Get the first ET (ETFRST) used for sampling the matrix
C     routine CAVSRC.
C
      ETFRST = BEGET
 
C
C     ENUF is used to ensure that the STEP size is always greater enough
C     so that there are at least MNPTS quaternions in the segment we
C     create.
C
      MINSTP = STPMIN
      MESH   = MINSTP
      ENUF   = MAX( MINSTP, (ENDET - BEGET)/MNPTS )
      MAXSTP = MIN( STPMAX, ENUF )
 
C
C     Initialize the routine ZZGETCAV.
C
      CALL ZZRESET()
 
C
C     Fetch the first pointing instance.
C
      CALL ZZGETCAV ( 'CAVSRC-0',  SPICE, CAVSRC, ETFRST,
     .                 CMATRX, AV, BAD )
 
      IF ( BAD ) THEN
         CALL CHKOUT ( 'FITCK3' )
         RETURN
      END IF
 
      QCOUNT = QCOUNT + 1
      TQ     = TQ     + 1
 
 
C
C     Copy the first quaternion and angular velocity into
C     the holding area.
C
      CALL M2Q    ( CMATRX,       Q    (1,QCOUNT) )
      CALL VEQU   ( AV,           AVEL (1,QCOUNT) )
      CALL SCE2T  ( SCID, ETFRST, TICKS(  QCOUNT) )
 
C
C     Here's how we are going to handle this.
C
C
C           |<----STEP---->|
C        ---*--------------*-----------
C           ETFRST         ETSTEP
C
C     We will compute the rotation and angular velocity at ETFRST
C     (we've already done that) and STEP seconds later.  Given this
C     pair of rotations, we'll interpolate several orientations between
C     ETFRST and ETSTEP.  To ensure that we interpolate at enough
C     points between ETFRST and ETSTEP we will either subdivide the
C     interval into NCHECK points OR check the interpolation on a set
C     whose points are MESH seconds apart. (Which ever gives us a finer
C     sampling.)
C
C     If the interpolated orientations match to
C     within the specified tolerance the actual orientations at
C     these points, we'll
C
C           |<----------STEP-------->|
C        ---*--------------*---------*-
C           ETFRST         ETLAST    ETSTEP
C
C       1) store our last successful interval endpoint in ETLAST;
C       2) increase STEP
C       3) Compute the actual orientation at the new ETSTEP
C       4) perform the checks all overagain.
C
C     This loop continues until we don't match the specified
C     tolerance. At this point we:
C
C
C       1) buffer the time and orientation information at ETLAST.
C       2) advance the reference ETFRST to be where we had ETLAST
C       3) decrease STEP by a small amount
C       4) start the above process all over agin.
C
C
C        Transition from this ...
C
C           |<----------STEP-------->|
C        ---*--------------*---------*---------------
C           ETFRST         ETLAST    ETSTEP
C
C
C        to this.   Note ETFRST moves and STEP shrinks.
C
C                          |<----STEP---->|
C        ---*--------------*--------------*---------
C                          ETFRST         ETSTEP
C
C
 
 
      STEP   = MIN ( MINSTP, MAXSTP )
      ETSTEP = MIN ( ETFRST + STEP, ENDET )
      ETLAST = ETFRST
      STEP   = MIN ( STEP, ENDET - ETLAST )
 
C
C     Since we'll always have a good C-matrix at ETLAST, we
C     compare ETLAST to ENDET so that one loop can handle all
C     of the tests for goodness of fit.  (As will be evident below,
C     the loop will terminate when ETLAST is equal to ENDET.)
C
      DO WHILE ( ETLAST .LT. ENDET )
 
C
C        Perform one idiot check just to make sure we haven't got
C        a bug somewhere below.
C
         IF ( ETSTEP .GT. ENDET .OR. ETSTEP .LT. BEGET ) THEN
            CALL SIGERR ( 'SPICE(SPICEBUG)'  )
            CALL CHKOUT ( 'FITCK3'           )
            RETURN
         END IF
 
C
C        Get the C-matrix at the next step point.
C
         CALL ZZGETCAV ( 'CAVSRC-1', SPICE,  CAVSRC, ETSTEP,
     .                    CMSTEP,    AVSTEP, BAD )
         IF ( BAD ) THEN
            CALL CHKOUT ( 'FITCK3' )
            RETURN
         END IF
C
C        Compare the interpolated C-matrix with the actual C-matrix at
C        several points in the interval between ETFRST and ETSTEP.  We
C        check first at the midpoint based on the assumption that we
C        are likely to have the worst error at the midpoint.
C
         OK     = .TRUE.
         POINTS =  1
         DELTA  =  MIN ( STEP/DBLE(NCHECK), MESH )
         ETFRAC =  ETFRST + DELTA
 
         DO WHILE ( ETFRAC .LT. ETSTEP .AND.  OK  )
 
 
C
C           Get the interpolated C-matrix and the actual C-matrix.
C
            CALL MXMT     ( CMSTEP,CMATRX, RDIFF )
            CALL RAXISA   ( RDIFF, AXIS,   ANGLE )
 
            TOOBIG = ANGLE .GT. 0.75D0*PI()
 
            FRAC   = (ETFRAC - ETFRST) / STEP
            REMAIN = 1.0D0 - FRAC
 
            CALL LINROT_M ( CMATRX, CMSTEP, FRAC,   CMAPRX, AVTMP  )
            CALL VLCOM    ( REMAIN, AV,     FRAC,   AVSTEP, AVAPRX )
 
 
            USE = BSRCHD ( ETFRAC, LBUFFD, ETLBUF )
 
C
C           If we have buffered this epoch (and corresponding
C           c-matrix and av) we jut get them instead of computing
C           them. We keep track of how many successful hits we have
C           so that we can decide whent to empty the local buffer.
C
            IF ( USE .GT. 0 ) THEN
 
               HITS = HITS + 1
               GET  = LBPNTR(USE)
               CALL MOVED ( CMLBUF(1,GET), 9, CMFRAC )
               CALL MOVED ( AVLBUF(1,GET), 3, AVFRAC  )
 
            ELSE
 
               LOOKUP = LOOKUP + 1
               CALL ZZGETCAV ( 'CAVSRC-2', SPICE,  CAVSRC, ETFRAC,
     .                          CMFRAC,    AVFRAC, BAD )
 
               IF ( BAD ) THEN
                  CALL CHKOUT ( 'FITCK3' )
                  RETURN
               END IF
 
C
C              If we have room, buffer the time, matrix and angular
C              velocity
C
               IF ( LBUFFD .LT. BUFMAX ) THEN
 
 
                  LESIZE  = LBUFFD
                  LPSIZE  = LBUFFD
                  LBUFFD  = LBUFFD + 1
 
C
C                 Copy the angular velocity and matrix into the
C                 storage buffer.
C
                  CALL MOVED ( AVFRAC, 3, AVLBUF(1,LBUFFD) )
                  CALL MOVED ( CMFRAC, 9, CMLBUF(1,LBUFFD) )
 
C
C                 Locate the slot where to store the time so that the
C                 times stay ordered.  And store a pointer to the
C                 last slot in the matrix and vector arrays.
C
                  PUT     = LSTLED ( ETFRAC, LESIZE, ETLBUF ) + 1
 
                  CALL INSLAD ( ETFRAC, 1, PUT, ETLBUF, LESIZE )
                  CALL INSLAI ( LBUFFD, 1, PUT, LBPNTR, LPSIZE )
 
               END IF
 
            END IF
 
C
C           Find the size of the rotation from the interpolated
C           matrix to the real matrix.
C
            CALL MXMT     ( CMFRAC, CMAPRX, RDIFF )
            CALL RAXISA   ( RDIFF,  AXIS,   ANGLE )
 
C
C           As long as ANGLE remains below TOL we will not need
C           to store anything.
C
            OK =        TOL .GE. ANGLE
     .            .AND. TOL .GE. DABS (VNORM(AVAPRX) - VNORM(AVFRAC))
     .            .AND. .NOT. TOOBIG
 
            ETFRAC = ETFRAC + DELTA
 
         END DO
C
C        How did the above loop terminate?
C
         IF ( OK ) THEN
C
C           All the interpolations match to the specified tolerance.
C           Make the stepsize a bit bigger and store the value of the
C           C-matrix and angular velocity that were computed at ETSTEP.
C
C           Note:  In the DO WHILE ( ETLAST .LT. ENDET ) loop that
C           encloses the IF block we're in now, this
C           is the only location where ETLAST has its value modified.
C
            GOTONE = .TRUE.
            ETLAST = ETSTEP
            STEP   = MIN ( MAXSTP, STEP  + INCR )
            CALL     VEQUG ( CMSTEP, 9, CMLAST )
            CALL     VEQU  ( AVSTEP,    AVLAST )
 
         ELSE IF ( .NOT. GOTONE ) THEN
C
C           There's only one way to get here.  The initial step
C           away from ETFRST was too big.  We need to shrink things
C           down a bit.
C
            IF ( STEP .LE. MINSTP ) THEN
C
C              We allow step sizes below the minimum, but
C              this requires some additional refinements
C
               STEP = STEP / SQRT(2.0D0 )
               MESH = STEP / 8.0D0
C
C              In addition, there is  no point in saving any currently
C              buffered times, matrices and quaternions.  Set the
C              number of buffered items to zero.
C
               LBUFFD = 0
 
             ELSE
 
               STEP = MAX( MINSTP, STEP - 4.0*INCR )
 
            END IF
C
C           If the step gets too small, we are going to declare
C           that we have a discontinuity and go on from here.
C
            IF ( STEP .LT. DSIZE ) THEN
 
               GOTONE = .TRUE.
               ETLAST = ETSTEP
               STEP   = MIN ( MAXSTP, STEP  + INCR )
               CALL     VEQUG ( CMSTEP, 9, CMLAST )
               CALL     VEQU  ( AVSTEP,    AVLAST )
 
               OK     = .FALSE.
 
               CALL ETCAL ( ETFRST, TIME1 )
               CALL ETCAL ( ETSTEP, TIME2 )

               CALL REMARK ( ' ' )
               CALL REMARK ('   A discontinuity occurs in the '
     .         //           'definition for the orientation')
               LINE = '   of object # between the ephemeris epochs:'
               CALL REPMI ( LINE, '#', PLTFRM, LINE )
 
               CALL REMARK ( LINE )
               CALL REMARK ( '   ' // TIME1 )
               CALL REMARK ( '   ' // TIME2 )
               CALL REMARK ( ' ' )

               CALL TOSTDO (' ' )
               CALL TOSTDO ('   A discontinuity occurs in the '
     .         //           'definition for the orientation')
               CALL TOSTDO ( LINE )
               CALL TOSTDO ('   ' // TIME1 )
               CALL TOSTDO ('   ' // TIME2 )
               CALL TOSTDO (' ' )

            END IF
 
         END IF
 
 
         IF (     ( GOTONE  .AND.  .NOT. OK)
     .       .OR. ( STEP .GE. MAXSTP ) ) THEN
C
C           There are two ways  to get to this point.
C
C           1) The current C-matrix and angular velocity are too far
C           from the last buffered matrix and angular velocity for
C           the simple interpolation method we are using to work.
C           However, the stuff in CMLAST and AVLAST were ok, so we
C           put these into the buffer and get set to start with
C           a new ETFRST and STEP.
C
C           2) The step size has gotten as big as we plan to let
C              it get.
C
C           In either case it's time to store a quaternion and
C           angular velocity.
C
            QCOUNT = QCOUNT + 1
            TQ     = TQ     + 1
 
            CALL M2Q  ( CMLAST,          Q    (1,QCOUNT) )
            CALL VEQU ( AVLAST,          AVEL (1,QCOUNT) )
            CALL SCE2T( SCID,   ETLAST,  TICKS(  QCOUNT) )
C
C           Move the last ending C-matrix and angular velocity
C           into the slots used for the beginning items in
C           interpolation.
C
            CALL VEQUG ( CMLAST, 9,      CMATRX )
            CALL VEQU  ( AVLAST,         AV     )
C
C           Call the report facility to note the last time for
C           which we have an interpolation set.
C
            CALL REPORT ( ETLAST, TQ )
C
C           Finally adjust the stepsize back a small amount
C           so that we have a good chance of having the
C           interpolation method work next time.
C
            STEP   = MAX( MINSTP, STEP   - 2.0D0*INCR )
            ETFRST = ETLAST
            GOTONE = .FALSE.
C
C           There is no point in keeping any of the buffered
C           times, matrices or angular velocity vectors.
C
            LBUFFD = 0
 
 
         END IF
C
C        If the quaternion buffer has filled up, we need to
C        write it out into the C-Kernel we are building.
C
C        Note that you can only exercise the block inside the IF
C        statement if you also exercised the ELSE block in the IF-THEN
C        etc. block above.
C
         IF ( QCOUNT .EQ. QBUFSZ ) THEN
 
            BEGTIK =  TICKS(1)
            ENDTIK =  TICKS(QCOUNT)
 
            AVFLAG = .TRUE.
 
            NINTS     = 1
            STARTS(1) = TICKS(1)
 
            CALL CKW03 ( HANDLE, BEGTIK, ENDTIK, PLTFRM,  REF,  AVFLAG,
     .                   SEGID,  QCOUNT, TICKS,  Q,       AVEL, NINTS,
     .                   STARTS  )
 
 
            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'FITCK3' )
               RETURN
            END IF
C
C           Move the last quaternion, angular velocity and tick into
C           the first slot of the buffer so there will be no gaps
C           between the next segment and the one just written.
C
            CALL VEQUG ( Q   (1,QCOUNT), 4, Q   (1,1) )
            CALL VEQU  ( AVEL(1,QCOUNT),    AVEL(1,1) )
 
            TICKS(1) = TICKS(QCOUNT)
            QCOUNT   = 1
 
         END IF
C
C        Note that ETSTEP is modified in the enclosing DO WHILE
C        loop only in the next statement.
C
         ETSTEP   = MIN ( ETFRST + STEP, ENDET )
 
 
         IF ( STEP .GT. MINSTP ) THEN
            MESH = MINSTP
         END IF
 
      END DO
 
C
C     Since ETLAST is modified only by setting it equal to ETSTEP
C     in the loop above, and ETSTEP is set only by the assignment
C     ETSTEP = MIN ( ETFRST + STEP, ENDET ) it follows that once
C     you reach this point ETLAST = ENDET.  Also there is only
C     one place where ETLAST is set.  At that location we set
C     CMLAST and AVLAST but we have not buffered them.  We do that
C     now.
C
      QCOUNT = QCOUNT + 1
      TQ     = TQ     + 1
 
      CALL M2Q  ( CMLAST,          Q    (1,QCOUNT) )
      CALL VEQU ( AVLAST,          AVEL (1,QCOUNT) )
      CALL SCE2T( SCID,   ETLAST,  TICKS(  QCOUNT) )
C
C     There is a remote possibility that the last two TICKS values
C     are the same due to granularity of the SCLK we are using.
C
      IF ( TICKS(QCOUNT) .EQ. TICKS(QCOUNT-1) ) THEN
         QCOUNT = QCOUNT - 1
      END IF
C
C     As long as we have more than 1 orientation set, we need to
C     write a CK segment.
C
      IF ( QCOUNT .GT. 1 ) THEN
 
         AVFLAG    = .TRUE.
         NINTS     = 1
         STARTS(1) = TICKS(1)
 
         BEGTIK =  TICKS(1)
         ENDTIK =  TICKS(QCOUNT)
 
         CALL CKW03 ( HANDLE, BEGTIK, ENDTIK, PLTFRM,  REF,  AVFLAG,
     .                SEGID,  QCOUNT, TICKS,  Q,       AVEL, NINTS,
     .                STARTS  )
 
      END IF
 
 
 
      CALL CHKOUT ( 'FITCK3' )
      RETURN
 
 
C$Procedure      SETMIN ( Set the minimum step size )
 
      ENTRY SETMIN ( BEGET )
 
C$ Abstract
C
C     Set the minimum step size used when sampling the C-matrix
C     generation function.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     KEYWORD
C
C$ Declarations
C
C     IMPLICIT NONE
C     DOUBLE PRECISION      BEGET
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     BEGET      I   is the minimum step size to use
C
C$ Detailed_Input
C
C     BEGET      is the minimum stepsize to use when performing
C                a normal traversal through the interval of
C                interest
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C$ Particulars
C
C     This entry point allows you to set the minimum step size that
C     will be used when sampling c-matrices and angular velocities
C     when constructing C-kernel segments.
C
C$ Examples
C
C     Sorry.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 02-AUG-2000 (WLT)
C
C
C-&
 
 
      STPMIN = MAX ( 0.00001D0, BEGET )
      RETURN
 
 
C$Procedure      SETMAX ( Set the maximum allowed step size )
 
      ENTRY SETMAX ( BEGET )
 
C$ Abstract
C
C     Set the maximum step size that can be used when sampling
C     matrices in the process of constructing a C-kernel.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     UTILITY
C
C$ Declarations
C
C     IMPLICIT NONE
C     DOUBLE PRECISION      BEGET
C
C$ Brief_I/O
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     BEGET      I   New maximum step size to use when sampling
C
C     $ Detailed_Input
C
C     BEGET      is the maximum step size allowed when sampling
C                matrices in the construction of a C-kernel.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C$ Particulars
C
C     This routine allows you to set the maximum step that will be used
C     when sampling matrices for the construction of a C-kernel
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 02-AUG-2000 (WLT)
C
C
C-&
 
      STPMAX = MAX( MINSTP, BEGET )
      RETURN
 
 
C$Procedure      SETMAX ( Set the dicontinuity threshold )
 
      ENTRY SETDSC ( BEGET )
 
C$ Abstract
C
C    Set the minimum step size that can be used between points in the
C    output C-kernel.  Inability to fit over intervals smaller than
C    this are regarded as discontinuities that cannot be fit.  (Yes
C    they do occur).
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     UTILITY
C
C$ Declarations
C
C     IMPLICIT NONE
C     DOUBLE PRECISION      BEGET
C
C$ Brief_I/O
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     BEGET      I   Discontinuity threshold.
C
C     $ Detailed_Input
C
C     BEGET      smallest step size that will be allowed between
C                output quaternions.  If the step size needs to
C                be less than this to fit the orientation to
C                tolerance, the orientation is declared to have
C                a discontinuity.  (Note that the C-kernel is
C                continuous anyway.)
C
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C$ Particulars
C
C     This routine allows you to set the discontinuity threshold for
C                an orientation fit.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 02-AUG-2000 (WLT)
C
C
C-&
 
      DSIZE = MAX( MINSTP/1000.0D0, BEGET )
      RETURN
 
 
 
 
 
 
      END
