C$Procedure PKTRAN ( Packet transformation )
 
      SUBROUTINE PKTRAN ( SUBTYP, PACKT1, PACKT2 )
      IMPLICIT NONE
 
C$ Abstract
C
C     Transform attitude packets from AEM-style to SPICE-style.
C     
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     CK
C     SCLK
C
C$ Keywords
C
C     AEM
C
C$ Declarations
 
      INCLUDE 'ck06.inc'

      INTEGER               SUBTYP
      DOUBLE PRECISION      PACKT1 ( * )
      DOUBLE PRECISION      PACKT2 ( * )
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     SUBTYP     I   CK type 6 subtype.
C     PACKT1     I   Input attitude packet.
C     PACKT2     O   Output attitude packet.
C
C$ Detailed_Input
C           
C     SUBTYP         is the subtype of CK data type 6 corresponding to
C                    the input data representation.
C          
C     PACKT1         is an attitude data packet. The packet may
C                    consist of any of the following, depending on
C                    the subtype:
C
C                       Subtype   Packet contents
C                       =======   ===============
C                       0         Quaternion and quaternion derivative
C
C                       1         Quaternion only
C
C                       2         [Not supported]
C
C                       3         Quaternion and angular velocity
C
C
C                    For all of the above subtypes, quaternions
C                    are "engineering style" and angular velocities
C                    are expressed relative to the instrument
C                    reference frame.
C
C$ Detailed_Output
C
C     PACKT2         is the result of transforming the input
C                    packet to "SPICE style." For quaternions,
C                    this means the quaternion is cycled to the
C                    right, so the scalar part of the quaternion
C                    becomes the first element. The vector part
C                    of the quaternion is negated. For angular
C                    velocity vectors, this means that the vector
C                    is transformed so that it expresses angular
C                    velocity in the base frame. Derivatives
C                    of quaternions and angular velocities are
C                    transformed analogously.
C
C$ Parameters
C
C     See the SPICELIB INCLUDE file ck06.inc.
C
C$ Exceptions
C
C     1) If the input subtype is not recognized, the error
C        SPICE(NOTSUPPORTED) is signaled.
C
C$ Files
C
C     The input packet normally will be provided by an CCSDS format
C     version 1 AEM file.
C      
C$ Particulars
C
C     None.
C
C$ Examples
C
C     See usage in AEM2CK routine MKCK06.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C     AEM2CK Version 1.0.0, 31-JUL-2015 (NJB)
C
C        Based on MEX2KER Version 1.1.0, 28-JUN-2012 (NJB)
C
C           Corrected duplicate argument bug in VMINUS calls.
C           Removed declaration of unused variable
C           TVSIZ. Added header.
C
C        MEX2KER Version 1.0.0, 13-MAY-2002 (NJB)
C
C-&

C
C     SPICELIB functions
C
      LOGICAL               RETURN

C
C     Local variables
C
      DOUBLE PRECISION      TMPBUF ( 3 )

      DOUBLE PRECISION      CMAT   ( 3, 3 )
      DOUBLE PRECISION      V      ( 3 )


      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'PKTRAN' )


      IF ( SUBTYP .EQ. C06TP0 ) THEN
C
C        The packet contains a quaternion, followed by the time
C        derivative of the quaternion. Conjugate and cycle forward 
C        both 4-vectors.
C
         CALL VMINUS ( PACKT1,               TMPBUF    )
         CALL VEQU   ( TMPBUF,               PACKT1    )
         CALL CYCLAD ( PACKT1,    4, 'F', 1, PACKT2    )
        
         CALL VMINUS ( PACKT1(5),            TMPBUF    )
         CALL VEQU   ( TMPBUF,               PACKT1(5) )
         CALL CYCLAD ( PACKT1(5), 4, 'F', 1, PACKT2(5) )


      ELSE IF ( SUBTYP .EQ. C06TP1 ) THEN
C
C        We have only a quaternion in this packet.
C
         CALL VMINUS ( PACKT1,               TMPBUF    )
         CALL VEQU   ( TMPBUF,               PACKT1    )
         CALL CYCLAD ( PACKT1,    4, 'F', 1, PACKT2    )
         

      ELSE IF ( SUBTYP .EQ. C06TP3 ) THEN
C
C        We have a quaternion and angular velocity.
C        
         CALL VMINUS ( PACKT1,               TMPBUF )
         CALL VEQU   ( TMPBUF,               PACKT1 )
         CALL CYCLAD ( PACKT1,    4, 'F', 1, PACKT2 )

         CALL Q2M  ( PACKT2, CMAT )

         CALL MTXV ( CMAT,   PACKT2(5),   V )
         CALL VEQU ( V,      PACKT2(5)      )

      ELSE

         CALL SETMSG ( 'Requested segment subtype <#> is not '//
     .                 'supported.'                           )
         CALL ERRINT ( '#',  SUBTYP                           )
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                  )
         CALL CHKOUT ( 'PKTRAN'                               )
         RETURN

      END IF


      CALL CHKOUT ( 'PKTRAN' )
      END

