C$Procedure CHKTAG ( Check time tags )
 
      SUBROUTINE CHKTAG ( SCLKID, SPCLIM, QUOLIM, NMINI, NPKTS, SCLKDP )
 
C$ Abstract
C
C     Check spacing of time tags in data to be written to a CK type 6
C     segment.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     CK
C
C$ Keywords
C
C     AEM2CK
C
C$ Declarations
 
      IMPLICIT NONE

      INTEGER               SCLKID
      DOUBLE PRECISION      SPCLIM
      DOUBLE PRECISION      QUOLIM
      INTEGER               NMINI
      INTEGER               NPKTS  ( * )
      DOUBLE PRECISION      SCLKDP ( * )

C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     SCLKID     I   ID of SCLK associated with a CK frame.
C     SPCLIM     O   Minimum time tag spacing.
C     QUOLIM     O   Maximum quotient of successive tag separations.
C     NMINI      I   Number of mini-segments.
C     NPKTS      I   Packet counts for each mini-segment.
C     SCLKDP     I   Array of time tags for a type 6 CK segment.
C
C$ Detailed_Input
C          
C     SCLKID         is the ID of a spacecraft clock associated with
C                    the input time tags.
C
C     SPCLIM         is the minimum spacing, measured in ephemeris
C                    seconds, between successive states in one
C                    input data block (the set of states following
C                    a metadata block and preceding the next
C                    metadata block or EOF). If not specified in the
C                    setup file, SPCLIM defaults to MINSPC (declared in
C                    aem2ck.inc).
C     
C     QUOLIM         is the maximum quotient of successive separations
C                    between states (considered in either order). If
C                    not specified in the setup file, QUOLIM defaults to
C                    MAXQUO (declared in aem2ck.inc).
C
C     NMINI          is the number of mini-segments with which the input
C                    time tags are associated.
C
C     NPKTS          is an array of packet counts. The Ith element of
C                    NPKTS is the packet count of the Ith mini-segment.
C                   
C                    NPKTS has dimension NMINI.
C
C     SCLKDP         an array of time tags (epochs) associated with the
C                    spacecraft clock identified by SCLKID. The time
C                    tags are expressed as encoded SCLK ("ticks"). The
C                    sets of time tags associated with the
C                    mini-segments are ordered by the indices of the
C                    mini-segments to which they belong. That is, the
C                    time tags of the first mini-segment are at the
C                    beginning of the array, followed by the time tags
C                    of the second mini-segment, and so on.
C
C                    The epochs for a given mini-segment are stored
C                    contiguously in increasing order.
C
C                    For each mini-segment, "padding" is allowed: the
C                    sequence of epochs for that mini-segment may start
C                    before the corresponding mini-segment interval
C                    start time and end after the corresponding
C                    mini-segment interval stop time. Padding is used
C                    to control behavior of interpolating polynomials
C                    near mini-segment interval boundaries.
C
C                    Due to possible use of padding, the elements of
C                    SCLKDP, taken as a whole, might not be in

C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C 
C     1)  If a consecutive pair of time tags is separated by less
C         than SPCLIM seconds, the error SPICE(TIMETAGSTOOCLOSE)
C         is signaled.
C
C     2)  If two consecutive time tags match, and if this condition
C         is not detected by the SPCLIM check, the error 
C         SPICE(ZEROSPACING) is signaled.
C
C     3)  If the ratio of separations between two adjacent pairs of
C         time tags exceeds QUOLIM, the error SPICE(BADSPACINGRATIO) is
C         signaled. The maximum of the ratio and its inverse is
C         compared against QUOLIM.
C
C$ Files
C
C     1) A leapseconds kernel and an SCLK kernel for the spacecraft
C        clock associated with SCLKID must be loaded before this
C        routine is called.
C     
C     The inputs are normally parsed from an AEM metadata block.
C     See [1] for details.
C
C$ Particulars
C
C     This routine checks the spacing of time tags within sequences of
C     time tags to be written to CK type 6 mini-segments. The spacing
C     of consecutive time tags is checked against a lower bound. The
C     spacing of consecutive pairs of time tags is checked a maximum
C     spacing ratio.
C
C     The spacing limits are expressed as TDB seconds, so the input
C     time tags are converted from SCLK to TDB for the comparisons.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     [1]  CCSDS Attitude Data Messages Blue Book, version CCSDS 
C          504.0-B-1, May, 2008.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    AEM2CK Version 1.0.0, 19-AUG-2015 (NJB)
C
C-&



C
C     SPICELIB functions
C
      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C
      INTEGER               TIMLEN
      PARAMETER           ( TIMLEN = 50 )

C
C     Local variables
C
      CHARACTER*(TIMLEN)    TIMSTR ( 3 )
      
      DOUBLE PRECISION      ET     ( 3 )
      DOUBLE PRECISION      Q
      DOUBLE PRECISION      R

      INTEGER               BASE
      INTEGER               I
      INTEGER               J
      INTEGER               MINIDX
      INTEGER               NREC
      INTEGER               W

      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'CHKTAG' )


C
C     Check data for each mini-segment individually.
C
      BASE = 0

      DO MINIDX = 1, NMINI

         NREC = NPKTS(MINIDX)

C
C        Convert the initial SCLK time tag to TDB.
C
         CALL SCT2E ( SCLKID, SCLKDP(BASE+1), ET(1) )

         IF ( FAILED() ) THEN
            CALL CHKOUT( 'CHKTAG' )
            RETURN
         END IF


         DO I = 2, NREC
C
C           First, make sure all consecutive time tags are 
C           separated by at least the required interval SPCLIM.
C                  
C
C           Convert the current SCLK time tag to TDB.
C
            W = BASE + I

            CALL SCT2E ( SCLKID, SCLKDP(W), ET(2) )
            
            IF ( FAILED() ) THEN
               CALL CHKOUT( 'CHKTAG' )
               RETURN
            END IF


            IF (  ( ET(2) - ET(1) )  .LT.  SPCLIM  ) THEN

               DO J = 1, 2

                  CALL TIMOUT ( ET(J),
     .                          'YYYY-MON-DD HR:MN:SC.######::TDB TDB',
     .                          TIMSTR(J)                              )
               END DO

               CALL SETMSG ( 'In mini-segment #, Consecutive '
     .         //            'time tags # and # at indices # '
     .         //            'and # respectively '
     .         //            'have spacing less than the limit '
     .         //            '# seconds.'                       )
               CALL ERRINT ( '#', MINIDX                        )

               DO J = 1, 2
                  CALL ERRCH  ( '#', TIMSTR(J)                  )
               END DO

               CALL ERRINT ( '#', W-1                           ) 
               CALL ERRINT ( '#', W                             )
               CALL ERRDP  ( '#',  SPCLIM                       )
               CALL SIGERR ( 'SPICE(TIMETAGSTOOCLOSE)'          )
               CALL CHKOUT ( 'CHKTAG'                           )
               RETURN

            END IF

C
C           Update ET(1) for the next loop iteration.
C
            ET(1) = ET(2)

         END DO



C
C        Now make sure that for any three consecutive time tags
C        T1, T2, T3 the spacing ratio R defined by 
C
C           R  =  MAX ( (T3-T2)/(T2-T1), (T2-T1)/(T3-T2) )
C
C        is no greater than the limit QUOLIM. Convert the time
C        tags to TDB for this computation, since the input tags
C        are expressed in the TDB system.
C
         DO I = 1, 2

            W = BASE + I

            CALL SCT2E ( SCLKID, SCLKDP(W), ET(I) )

            IF ( FAILED() ) THEN
               CALL CHKOUT( 'CHKTAG' )
               RETURN
            END IF

         END DO


         DO I = 3, NREC
C
C           Convert the time tag for the current record to TDB.
C                  
            W = BASE + I

            CALL SCT2E ( SCLKID, SCLKDP(W), ET(3) )

            IF ( FAILED() ) THEN
               CALL CHKOUT( 'CHKTAG' )
               RETURN
            END IF

C
C           The spacing check should rule out a zero denominator, but
C           we'll check anyway.
C
            IF ( ( ET(2) .EQ. ET(1) ) .OR. ( ET(3) .EQ. ET(2) ) ) THEN

               DO J = 1, 3
                  
                  CALL TIMOUT ( ET(J),
     .                          'YYYY-MON-DD HR:MN:SC.######::TDB TDB',
     .                          TIMSTR(J)                              )
               END DO

               CALL SETMSG ( 'Consecutive time tags #, #, #, include '
     .         //            'two adjacent, equal values; '
     .         //            'these tags should be strictly '
     .         //            'increasing.'                        )

               DO J = 1, 3            
                  CALL ERRCH  ( '#', TIMSTR(J) )
               END DO

               CALL SIGERR ( 'SPICE(ZEROSPACING)'  )
               CALL CHKOUT ( 'CHKTAG'              )
               RETURN
 
            END IF

C
C           We've ensured that Q and 1/Q are non-zero.
C
            Q = ( ET(3) - ET(2) ) / ( ET(2) - ET(1) )

            R = MAX ( Q, 1.D0/Q )

            IF ( R .GT. QUOLIM ) THEN

               DO J = 1, 3
                  
                  CALL TIMOUT ( ET(J),
     .                          'YYYY-MON-DD HR:MN:SC.######::TDB TDB',
     .                          TIMSTR(J)                              )
               END DO

               CALL SETMSG ( 'Consecutive time tags #, #, #, '   
     .         //            'have spacing ratio #; the limit ' 
     .         //            'is #.'                            )
               DO J = 1, 3            
                  CALL ERRCH  ( '#', TIMSTR(J)                  )
               END DO
                     
               CALL ERRDP  ( '#',  R                            )
               CALL ERRDP  ( '#',  QUOLIM                       )
               CALL SIGERR ( 'SPICE(BADSPACINGRATIO)'           )
               CALL CHKOUT ( 'CHKTAG'                           )
               RETURN

            END IF

C
C           Update the trailing ET values for the next pass.
C
            DO J = 1, 2
               ET(J) = ET(J+1)
            END DO

         END DO


         BASE = BASE + NPKTS(MINIDX)

      END DO


      CALL CHKOUT ( 'CHKTAG' )
      RETURN
      END
