import zipfile
import subprocess
import datetime
import logging
import sys
import os
import math

from datetime import datetime, timedelta

from .em16 import quaternions_filter
from .bepic import phebus_fov_reader

def tcp_reader(kernel, tcp_file, config):

    #
    # We convert the binary files into text files. The binary files
    # are then removed
    #
    if config['input_extension'] == 'zip' in tcp_file:

        zip_ref = zipfile.ZipFile(tcp_file, 'r')
        zip_ref.extractall()

        tcp_file = zip_ref.namelist()[0]

        zip_ref.close()

    lsk_file = kernel.config['kernels_dir']+'/'+kernel.config['lsk']

    #TODO: Remove this dirty workarodund for BepiColombo
    if config['mission_accr'] == 'MPO':
        mission = '_bc'
    elif config['mission_accr'] == 'SOLO':
        mission = '_bc'
    elif config['mission_accr'] == 'RM':
        mission = '_emrsp'
    else:
        mission = ''

    command_line_process = subprocess.Popen(
                    [kernel.directories.executables + '/' +
                     'tcpescan'+mission,
                     tcp_file, lsk_file],
                    stdout=subprocess.PIPE,
                    stderr=subprocess.STDOUT)

    process_output, _ = command_line_process.communicate()
    log = process_output.decode('utf-8')

    tcp_list = log.split('\n')

    #
    # The following lines of the output are generated by TCPESCAN and FORTRAN
    #
    tcp_list_iter = list(tcp_list)
    for element in tcp_list_iter:
        if 'IEEE_DENORMAL' in element:
            tcp_list.remove(element)

        if 'DELTET=' in element:
            tcp_list.remove(element)

        if 'STOP' in element:
            if 'Unexpected end of File' in element:
                sys.exit("TCP2SCLK: Unexpected end of File in {}".format(tcp_file))
            else:
                tcp_list.remove(element)

        if element == '':
                tcp_list.remove(element)

    return(tcp_list)


def aem_reader(aem_file, config):

    #
    # We init the variables to log the processing
    #
    info_ratio = 0

    with open(aem_file, 'r') as t:

        data_start = False
        aem_list = []
        index = 0
        previous_dt = ''

        for line in t:
            index += 1

            if 'START_TIME' in line:
                try:
                    date = line.split('=')[1].strip()[:-2]
                    segment_start_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S.%f")
                except:
                    date = line.split('=')[1].strip()
                    segment_start_dt = datetime.strptime(date,"%Y-%m-%dT%H:%M:%S")

            if 'STOP_TIME' in line:
                try:
                    date = line.split('=')[1].strip()[:-2]
                    segment_finish_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S.%f")
                except:
                    date = line.split('=')[1].strip()
                    segment_finish_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S")


            # ================================================================================
            # Toolkit version: N0065
            # SPICE(BADSPACINGRATIO) --
            # Consecutive time tags 2017-JAN-11 05:41:09.184225 TDB, 2017-JAN-11
            # 05:46:06.784225 TDB, 2017-JAN-11 05:46:09.184225 TDB, have spacing ratio
            # 1.2400000124176E+02; the limit is 1.0000000000000E+02.
            # A traceback follows.  The name of the highest level module is first.
            # AEM2CK --> CVTAEM --> CVT01 --> MKCK06 --> CHKTAG
            # ================================================================================
            if 'DATA_START' in line:
                data_start = True
                data_start_index = 0
            if 'DATA_STOP' in line:
                data_start = False


                #
                # We need to reset it in between segments otherwise:
                #
                #  ================================================================================
                #  Toolkit version: N0065
                #  SPICE(BOUNDSDISAGREE) --
                #  Mini-segment interval 1 start time 1.7161649282063E+12 precedes mini-segment's
                #  first epoch 1.7162033060708E+12.
                #  A traceback follows.  The name of the highest level module is first.
                #  AEM2CK --> CVTAEM --> CVT01 --> MKCK06 --> CKW06
                #  ================================================================================
                previous_dt = ''

            #
            # We identify the line as data line (after the META_STOP tag and
            # not an empty line)
            #
            if data_start and line.strip() != '\n' and 'DATA_START' not in line:

                date = line.split()[0][:-2]

                if previous_dt:
                    try:
                        current_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S.%f")
                    except:
                        date = line.split()[0]
                        current_dt = datetime.strptime(date,"%Y-%m-%dT%H:%M:%S")

                    deltat = current_dt - previous_dt
                    deltat = abs(deltat.seconds)

                    #
                    # We need to include the data_start_index condition to avoid removing the first entry of
                    # a data block.
                    #
                    if deltat < 5.0:
                        if previous_dt != segment_start_dt:
                            #logging.warning('   AEM Processing: Found spacing ratio less than 5 seconds in {index}:')
                            #logging.warning('   ' + line[:-1])
                            info_ratio += 1

                            #
                            # We need to remove the previous line, not the current line, otherwise we will remove a
                            # segment boundary.
                            #
                            aem_list.remove(previous_line)

                    aem_list.append(line)
                else:
                    aem_list.append(line)

                try:
                    previous_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S.%f")
                except:
                    date = line.split()[0]
                    previous_dt = datetime.strptime(date,"%Y-%m-%dT%H:%M:%S")

                data_start_index += 1
            else:
                aem_list.append(line)

            previous_line = line

    #
    # We report the number of exceptions here
    #
    if info_ratio:
        logging.warning(f'   AEM Processing: Found {info_ratio} intervals with spacing ratio less than 5 seconds')

    return(aem_list)


def oem_reader(oem_file, config):

    #
    # We init the variables to log the processing
    #
    info_incomplete = 0

    with open(oem_file, 'r') as t:

        data_start = False
        oem_list = []
        index = 0

        for line in t:

            index += 1

            # ==============================================================================
            #
            # Toolkit version: N0066
            #
            # SPICE(TOOFEWTOKENS) --
            #
            # Found only 1 numeric tokens while attempting to read a state vector. OEM file
            # is <OEMS_DA_059_01______00105.EXM>. Last line read was #68269. Data record
            # started on line #68269. Concatenated data line was
            # <2016-06-24T03:25:34.09215033>
            #
            # A traceback follows.  The name of the highest level module is first.
            # OEM2SPK --> CVTOEM --> CVT02 --> MAKSEG --> PRSDR
            #
            # ==============================================================================
            if 'META_START' in line:
                data_start = False

            # We need to remove the end of line character:
            line_splitted = line.split()

            #
            # We identify the line as data line (after the META_STOP tag and
            # not an empty line)
            #

            #
            # Implemented the possibility of having acceleration or not as an
            # input.
            #
            if ',' in config['input_fields']:
                input_fields = config['input_fields'].split(',')
            else:
                input_fields = [config['input_fields']]

            incomplete_line = False

            if data_start and line.strip() != '':

                incomplete_line = True

                for input_field in input_fields:
                    if len(line_splitted) == int(input_field):
                        oem_list.append(line)
                        incomplete_line = False

                if incomplete_line:
                    #logging.info('OEM Processing: Found incomplete data in input line {index}:')
                    #logging.info('   ' + line)
                    info_incomplete += 1
            else:
                oem_list.append(line)

            if 'META_STOP' in line:
                data_start = True

            #  OEM2SPK Program; Ver. 2.2.0, 01-DEC-2016; Toolkit Ver. N0065
            #  ================================================================================
            #  Toolkit version: N0065
            #  SPICE(TIMETAGERROR) --
            #  Consecutive time tags 2022-MAY-24 02:21:45.131521 and 2022-MAY-24
            #  02:21:45.131898 have spacing less than the limit 1.0000000000000E-03 seconds.
            #  A traceback follows.  The name of the highest level module is first.
            #  OEM2SPK --> CVTOEM --> CVT02 --> MAKSEG
            #  ================================================================================

            #  ================================================================================
            #  Toolkit version: N0065
            #  SPICE(TIMETAGERROR) --
            #  Consecutive time tags 2023-JAN-01 02:14:47.324165 2023-JAN-01 03:14:36.702917
            #  2023-JAN-01 03:14:36.846376 have spacing ratio 2.5020268339399E+04; the limit
            #  is 1.0000000000000E+02.
            #  A traceback follows.  The name of the highest level module is first.
            #  OEM2SPK --> CVTOEM --> CVT02 --> MAKSEG
            #  ================================================================================

    #
    # We report the number of exceptions here
    #
    if info_incomplete:
        logging.warning(f'   OEM Processing: Found {info_incomplete} incomplete data line(s)')

    return(oem_list)


def oasw_orbit_reader(oasw_orbit_file, config):

    #
    # We init the variables to log the processing
    #
    info_duplicate  = 0

    with open(oasw_orbit_file, 'r') as t:

        data_start = False
        oasw_orbit_list = []
        index = 0
        previous_time = False

        for line in t:

            index += 1

            # MEX2KER Program; Ver. 2.2.0, 15-JUL-2014; Toolkit Ver. N0066
            # ================================================================================
            # Toolkit version: N0066
            # SPICE(TIMESOUTOFORDER) --
            # EPOCH 8.5707450000000E+08 having index 290 is not greater than its predecessor
            # 8.5707450000000E+08.
            # A traceback follows.  The name of the highest level module is first.
            # MEX2KER --> CVTMEX --> CVT01 --> MKSKSG --> SPKW18
            # ================================================================================

            if 'META_START' in line:
                data_start = False

            #
            # We identify the line as data line (after the META_STOP tag and
            # not an empty line)
            #

            if data_start and line.strip() != '':

                time = line.split()[0]

                if previous_time == time:

                    #logging.info(f'OASW Orbit Processing: Duplicated time in input line {index}:')
                    #logging.info('   ' + line)
                    info_duplicate += 1
                else:
                    oasw_orbit_list.append(line)

                previous_time = time
            else:
                oasw_orbit_list.append(line)


            if 'META_STOP' in line:
                data_start = True

    #
    # We report the number of exceptions here
    #
    if info_duplicate:
        logging.warning(f'   OASW Processing: Found {info_duplicate} duplicate(s)')

    return(oasw_orbit_list)


def oasw_attitude_reader(oasw_attitude_file, trim=0):

    #
    # We init the variables to log the processing
    #

    t = open(oasw_attitude_file, 'r').readlines()
    stop_date = False
    trimindex = 0
    for line in reversed(t):
        if 'STOP_TIME' in line and not stop_date:
            stop_date = line.split('= ')[1].strip()
            flagdate = datetime.strptime(stop_date.split('.')[0], "%Y-%m-%dT%H:%M:%S") - timedelta(hours=trim)
        if len(line.rstrip()) > 80 and stop_date:
            stop_date = line.split(' ')[0]
            if datetime.strptime(stop_date.split('.')[0], "%Y-%m-%dT%H:%M:%S") < flagdate:
                break
        trimindex += 1

    #
    # trim the source file before the corresponding meta data start
    #
    trimindex = len(t) - trimindex - 1
    with open(oasw_attitude_file, 'w+') as newfile:
        for i in range(0, trimindex, 1):
            newfile.write(t[i])
        newfile.write((flagdate + timedelta(seconds=1)).strftime("%Y-%m-%dT%H:%M:%S") + '.00000000 ' + t[trimindex].split(' ', 1)[1])

    return


def hk_quaternions2ck_reader(kernel, tm_file, config):

    #
    # We init the variables to log the processing
    #
    info_duplicate   = 0
    info_gt_one      = 0
    info_negative    = 0
    info_incomplete  = 0
    info_quaternions = 0
    #
    # We obtain the time filed number from the configuration file
    # It has to be a list for there can be two columns
    #
    input_time_format = config['tm_file'][0]['time_format']
    header_lines  = int(config['tm_file'][0]['header_lines'])

    if input_time_format == 'SCLK':
        input_time_field_number = config['tm_file'][0]['data'][0]['scet']
    else:
        input_time_field_number = int(config['tm_file'][0]['data'][0]['utc'])


    #
    # We obtain the delimiter character
    #
    delimiter = config['tm_file'][0]['delimiter']


    #
    # We obtain the number of data fields and its correspondance
    #
    input_data_field_numbers = [int(config['tm_file'][0]['data'][0]['qx']),
                                int(config['tm_file'][0]['data'][0]['qy']),
                                int(config['tm_file'][0]['data'][0]['qz']),
                                int(config['tm_file'][0]['data'][0]['qs'])]

    tm_list = []
    previous_row_time = ''

    (sclk_partition, sclk_delimiter) = sclk_reader(kernel)


    filter_flag = False
    index = 0
    row_prev = []
    sclk_fraction_prev = ''
    sclk_fraction_pprev = ''
    sclk_fraction_ppprev = ''
    sclk_fraction_pppprev = ''
    sclk_initial = ''
    with open(tm_file, 'r', encoding='utf-8') as t:

        for line in t:

            index += 1

            if header_lines == 0 or header_lines < index:

                row_data = []

                # We need to remove the end of line character:
                line = line.split('\n')[0]

                try:
                    if ',' in delimiter:

                        if input_time_format == 'SCLK':
                            if ',' in str(input_time_field_number):
                                row_time = sclk_partition + '/' + str(line.split(delimiter)[
                                                   int(input_time_field_number[0]) - 1]) + \
                                           sclk_delimiter + str(line.split(delimiter)[
                                                   int(input_time_field_number[2]) - 1])

                            else:
                                input_time_field_number = int(input_time_field_number)
                                row_time = str(line.split(delimiter)[
                                                input_time_field_number - 1])
                        elif input_time_format == 'TDB':

                            if 'Z' in line:
                                row_time = str(line.split(delimiter)[input_time_field_number-1]).strip()[0:-1]
                            else:
                                row_time = str(line.split(delimiter)[input_time_field_number-1])
                            row_time = row_time.replace('T', '-') + 'TDB'
                        else:
                            if 'Z' in line:
                                row_time = str(line.split(delimiter)[input_time_field_number-1]).strip()[0:-1]
                            else:
                                row_time = str(line.split(delimiter)[input_time_field_number-1])

                        if (' ' in row_time):
                            if input_time_format == 'SCLK':
                                row_time = row_time.replace(' ','')
                            else:
                                row_time = row_time.replace(' ','T')

                        for data_element_field_number in input_data_field_numbers:

                            #
                            # Solar Orbiter specific processing
                            #
                            if kernel.config['spacecraft'] == 'solo':
                                if 'EVENT' in line:
                                    row_data.append('EVENT')
                                else:
                                    row_data.append(float(line.split(',')[
                                                              data_element_field_number - 1]))
                            else:
                                row_data.append(float(line.split(',')[data_element_field_number-1]))

                    else:

                        line = line.strip()

                        if input_time_format == 'SCLK':
                            if ',' in str(input_time_field_number):
                                row_time = sclk_partition + '/' + str(line.split(delimiter)[
                                                   int(input_time_field_number[0]) - 1]) + \
                                           sclk_delimiter + str(line.split(delimiter)[
                                                   int(input_time_field_number[2]) - 1])

                            else:
                                input_time_field_number = int(input_time_field_number)
                                row_time = str(line.split()[
                                                input_time_field_number - 1])
                        elif input_time_format == 'TDB':
                            if 'Z' in line:
                                row_time = str(line.split()[input_time_field_number-1]).strip()[0:-1]
                            else:
                                row_time = str(line.split()[input_time_field_number-1])
                            row_time = row_time.replace('T', '-') + 'TDB'
                        else:
                            if 'Z' in line:
                                row_time = str(line.split()[input_time_field_number-1]).strip()[0:-1]
                            else:
                                row_time = str(line.split()[input_time_field_number-1])

                        if (' ' in row_time):
                            if input_time_format == 'SCLK':
                                row_time = row_time.replace(' ','')
                            else:
                                row_time = row_time.replace(' ','T')

                        for data_element_field_number in input_data_field_numbers:
                            #
                            # We need to check that
                            #
                            row_data.append(float(line.split()[data_element_field_number-1]))
                except:
                    info_incomplete += 1
                    #logging.info('   HM TM Processing: Found incomplete data line in line {}:'.format(index))
                    #logging.info('   {}'.format(line))
                    continue

                row = row_time + ' '

                # As indicated by Boris Semenov in an e-mail "ROS and MEX "measured" CKs"
                # sometimes the scalar value is negative and the sign of the rest of the
                # components of the quaternions needs to be changed!
                if 'EVENT' not in row_data:
                    if row_data[-1] < 0:
                        neg_data = [-x for x in row_data]
                        info_negative += 1
                        #logging.info(f'   HM TM Processing: Found negative QS on input line {row_data}:')
                        #logging.info('   ' + str(neg_data))
                        row_data = neg_data

                for element in row_data:

                    row += str(element) + ' '

                # We filter out "bad quaternions"

                row += '\n'

                # We remove the latest entry if a time is duplicated
                if row_time == previous_row_time:
                    info_duplicate += 1
                    #logging.info(f'   HM TM Processing: Found duplicate time at {row_time}')

                else:
                    # We do not include the entry if one element equals 1 or gt 1
                    append_bool = True
                    for quaternion in row_data:
                        if 'EVENT' not in row_data:
                            if quaternion > 1.0:
                                append_bool = False
                                info_gt_one += 1
                                #logging.info(f'   HM TM Processing: Found quaternion GT 1 on input line {row_data}:')
                                #logging.info('   ' + str(row))


                    if kernel.input_processing == 'True':

                        if kernel.config['spacecraft'] == 'tgo':
                            (append_bool, line, row, row_prev, index,
                             tm_list, filter_flag, sclk_fraction_prev,
                             sclk_fraction_pprev, sclk_fraction_ppprev,
                             sclk_fraction_pppprev, sclk_initial, info_quaternions) = \
                                quaternions_filter(append_bool,line, row,
                                                        row_prev, index, tm_list,
                                                        filter_flag, sclk_fraction_prev,
                                                        sclk_fraction_pprev, sclk_fraction_ppprev,
                                                        sclk_fraction_pppprev, sclk_initial, info_quaternions)

                    else:
                        append_bool = True

                    if append_bool:
                        tm_list.append(row)

                previous_row_time = row_time

    #
    # We report the number of exceptions here
    #
    if info_duplicate:
        logging.warning(f'   HK TM Processing: Found {info_duplicate} duplicate(s)')
    if info_gt_one:
        logging.warning(f'   HK TM Processing: Found {info_gt_one} quaternion(s) GT 1')
    if info_negative:
        logging.warning(f'   HK TM Processing: Found {info_negative} negative QS')
    if info_incomplete:
        logging.warning(f'   HK TM Processing: Found {info_incomplete} incomplete data line(s)')
    if info_quaternions:
        logging.warning(f'   HK TM Processing: Found {info_quaternions} coarse quaternion(s)')

    # We remove the carriage return from the last line if present
    try:
        last_line = tm_list[-1].split('\n')[0]
        tm_list = tm_list[:-1]
        tm_list.append(last_line)

        return (tm_list)

    except:

        return (tm_list)


def hk_tm2ck_reader(kernel, tm_file, config):

    #
    # We init the variables to log the processing
    #
    info_duplicate  = 0
    info_incomplete = 0

    #
    # We obtain the time filed number from the configuration file
    # It has to be a list for there can be two columns
    #
    input_time_format = config['time_format']

    if input_time_format == 'SCLK':
        input_time_field_number = config['data'][0]['scet']
    elif input_time_format == 'UTC':

        #
        # If there is no 'utc' in the 'data' of the TM input, then the time
        # is present in the input FILENAME, and there is only one data element
        # hence we extract the time from there
        #
        if 'utc' in config['data'][0]:
            input_time_field_number = int(config['data'][0]['utc'])
        else:
            input_time_field_number = False

            #
            # For the time being we only need this for CaSSIS, so we only
            # include this option
            #
            filename_time = tm_file.split('-')[-4]
            filename_time = filename_time[0:4]+'-'+\
                            filename_time[4:6]+'-'+ \
                            filename_time[6:9]+ \
                            filename_time[9:11]+':'+\
                            filename_time[11:13]+':'+\
                            filename_time[13:15]+'.'+\
                            filename_time[15:]

    #
    # We obtain the delimiter character and header lines
    #
    delimiter = config['delimiter']
    header_lines = int(config['header_lines'])

    #
    # We obtain the number of data fields and its correspondance
    #
    input_data_field_number = int(config['data'][0]['ang'])


    tm_list = []
    previous_row_time = ''

    (sclk_partition, sclk_delimiter) = sclk_reader(kernel)

    index = 0
    with open(tm_file, 'r') as t:

        for line in t:

            index += 1

            if header_lines == 0 or header_lines < index:

                row_data = []

                # We need to remove the end of line character:
                line = line.split('\n')[0]

                try:
                    if ',' in delimiter:

                        if input_time_format == 'SCLK':
                            if ',' in input_time_field_number:
                                row_time = sclk_partition + '/' + str(line.split(delimiter)[
                                                   int(input_time_field_number[0]) - 1]) + \
                                           sclk_delimiter + str(line.split(delimiter)[
                                                   int(input_time_field_number[2]) - 1])

                            else:
                                input_time_field_number = int(input_time_field_number)
                                row_time = str(line.split(delimiter)[
                                                input_time_field_number - 1])

                        else:
                            if input_time_field_number:
                                if 'Z' in line:
                                    row_time = str(line.split(delimiter)[
                                                       input_time_field_number - 1]).strip()[
                                               0:-1]
                                else:
                                    row_time = str(line.split(delimiter)[
                                                       input_time_field_number - 1])

                            else:
                                row_time = filename_time

                        if (' ' in row_time):
                            if input_time_field_number:
                                if input_time_format == 'SCLK':
                                    row_time = row_time.replace(' ','')
                                else:
                                    row_time = row_time.replace(' ','T')
                            else:
                                row_time = filename_time

                        row_data.append(float(line.split(',')[input_data_field_number-1]))

                    else:

                        proc_line = line.strip()

                        if input_time_format == 'SCLK':
                            input_time_field_number = int(input_time_field_number)
                            row_time = str(proc_line.split()[input_time_field_number - 1])
                        else:
                            row_time = str(proc_line.split()[input_time_field_number-1]).strip()
                            row_time = row_time.replace('Z', '')

                        if (' ' in row_time):
                            if input_time_format == 'SCLK':
                                row_time = row_time.replace(' ','')
                            else:
                                row_time = row_time.replace(' ','T')
                                row_time = row_time.replace('Z','')


                        row_data.append(float(proc_line.split()[input_data_field_number-1]))

                except:
                    #logging.info('   fHM TM Processing: Found incomplete data line in line {index}:')
                    #logging.info('   {}'.format(line))
                    info_incomplete += 1
                    continue

                row = row_time + ' '

                #
                # In principle there is only one element for this type of TM
                #
                for element in row_data:

                    #
                    # If we need to factor the data like for example for ACS
                    #
                    if config['data_factor'] != 'False':
                        element *= eval(config['data_factor'])

                    if config['rot_axis'] == "1":
                        ang_rot1 = element
                        ang_rot2 = 0.0
                        ang_rot3 = 0.0
                    if config['rot_axis'] == "2":
                        ang_rot1 = 0.0
                        ang_rot2 = element
                        ang_rot3 = 0.0
                    if config['rot_axis'] == "3":
                        ang_rot1 = 0.0
                        ang_rot2 = 0.0
                        ang_rot3 = element

                    row += '{} {} {}'.format(ang_rot1, ang_rot2, ang_rot3)

                #
                # We filter out the input
                #

                # We remove the latest entry if a time is duplicated
                if row_time == previous_row_time:
                    #logging.info(f'   HM TM Processing: Found duplicate time at {row_time}')
                    info_duplicate += 1
                else:
                    append_bool = True

                if append_bool:
                    tm_list.append(row)

                    #
                    # This is the CaSSIS filter, which introduces another element
                    # after 5 seconds only applies to "file_schema":"cas_raw_sc"
                    #
                    if config['file_schema'] == 'cas_raw_sc':
                        utc = row.split(config['delimiter'])[0]

                        cmd = kernel.directories.executables + os.sep + 'utc2tdb ' + str(
                            utc) + ' ' + kernel.directories.kernels + os.sep + kernel.config['lsk']
                        command_line_process = subprocess.Popen(cmd, shell=True,
                                                                stdout=subprocess.PIPE,
                                                                stderr=subprocess.STDOUT)
                        command_line_process.wait()
                        process_output = command_line_process.communicate()[0]
                        et = float(process_output.decode('utf-8')) + 5.0

                        cmd = kernel.directories.executables + os.sep + 'tdb2utc ' + str(
                                et) + ' ' + kernel.directories.kernels + os.sep + kernel.config['lsk']
                        command_line_process = subprocess.Popen(cmd, shell=True,
                                                                stdout=subprocess.PIPE,
                                                                stderr=subprocess.STDOUT)
                        command_line_process.wait()
                        process_output = command_line_process.communicate()[0]
                        utc = str(process_output.decode('utf-8')).split('\n')[0].strip()

                        tm_list.append(utc+' '+' '.join(row.split()[1:]))


                append_bool = False
                previous_row_time = row_time

    #
    # We report the number of exceptions here
    #
    if info_duplicate:
        logging.warning(f'   HK TM Processing: Found {info_duplicate} duplicate(s)')
    if info_incomplete:
        logging.warning(f'   HK TM Processing: Found {info_incomplete} incomplete data line(s)')

    return(tm_list)


def hk_tm2ck_dafcat_reader(kernel, tm_file, config, dafcat_element):

    #
    # We init the variables to log the processing
    #
    info_duplicate  = 0
    info_incomplete = 0

    #
    # We obtain the time filed number from the configuration file
    # It has to be a list for there can be two columns
    #
    input_time_format = config['tm_file'][0]['time_format']

    if input_time_format == 'SCLK':
        input_time_field_number = config['tm_file'][0]['data'][0]['scet']
    else:
        input_time_field_number = int(config['tm_file'][0]['data'][0]['utc'])

    #
    # We obtain the delimiter character and header lines
    #
    delimiter = config['tm_file'][0]['delimiter']
    header_lines = int(config['tm_file'][0]['header_lines'])

    #
    # We obtain the number of data fields and its correspondence
    #
    if dafcat_element == 1:
        input_data_field_number = int(config['tm_file'][0]['data'][0]['ang1'])
        rot_axis = config['tm_file'][0]['rot_axis1']
    if dafcat_element == 2:
        input_data_field_number = int(config['tm_file'][0]['data'][0]['ang2'])
        rot_axis = config['tm_file'][0]['rot_axis2']

    tm_list = []
    previous_row_time = ''
    previous_row_data = ''

    (sclk_partition, sclk_delimiter) = sclk_reader(kernel)

    index = 0
    with open(tm_file, 'r') as t:

        for line in t:

            index += 1

            if header_lines == 0 or header_lines < index:

                row_data = []

                # We need to remove the end of line character:
                line = line.split('\n')[0]

                try:
                    if ',' in delimiter:

                        if input_time_format == 'SCLK':
                            if ',' in input_time_field_number:
                                row_time = sclk_partition + '/' + str(line.split(delimiter)[
                                                   int(input_time_field_number[0]) - 1]) + \
                                           sclk_delimiter + str(line.split(delimiter)[
                                                   int(input_time_field_number[2]) - 1])

                            else:
                                input_time_field_number = int(input_time_field_number)
                                row_time = str(line.split(delimiter)[
                                                input_time_field_number - 1])

                        else:
                            row_time = str(line.split(delimiter)[input_time_field_number-1]).strip()
                            row_time = row_time.replace('Z', '')

                        if (' ' in row_time):
                            if input_time_format == 'SCLK':
                                row_time = row_time.replace(' ','')
                            else:
                                row_time = row_time.replace(' ','T')
                                row_time = row_time.replace('Z','')

                        row_data.append(float(line.split(',')[input_data_field_number-1]))

                    else:
                        proc_line = line.strip()

                        if input_time_format == 'SCLK':
                            input_time_field_number = int(input_time_field_number)
                            row_time = str(proc_line.split()[input_time_field_number - 1])
                        else:
                            row_time = str(proc_line.split()[input_time_field_number-1]).strip()
                            row_time = row_time.replace('Z', '')

                        if (' ' in row_time):
                            if input_time_format == 'SCLK':
                                row_time = row_time.replace(' ','')
                            else:
                                row_time = row_time.replace(' ','T')
                                row_time = row_time.replace('Z','')


                        row_data.append(float(proc_line.split()[input_data_field_number-1]))

                except:
                    #logging.info(f'   HM TM Processing: Found incomplete data line in line {index}:')
                    #logging.info('   {}'.format(line))
                    info_incomplete += 1
                    continue

                row = row_time + ' '

                #
                # In principle there is only one element for this type of TM
                #
                for element in row_data:

                    if config['tm_file'][0]['data_factor'] != 'False':
                        element *= eval(config['tm_file'][0]['data_factor'])
                    #
                    # For BepiColombo Phebus data
                    #
                    if kernel.config["object_name"] == "Phebus" and  dafcat_element == 1:
                        angles = phebus_fov_reader(element)
                        row += f'{angles[2]} {angles[1]} {angles[0]}\n'

                    else:

                        if rot_axis == "1":
                            ang_rot1 = element
                            ang_rot2 = 0.0
                            ang_rot3 = 0.0
                        if rot_axis == "2":
                            ang_rot1 = 0.0
                            ang_rot2 = element
                            ang_rot3 = 0.0
                        if rot_axis == "3":
                            ang_rot1 = 0.0
                            ang_rot2 = 0.0
                            ang_rot3 = element

                        row += f'{ang_rot1} {ang_rot2} {ang_rot3}\n'

                #
                # We filter out the input
                #

                # We remove the latest entry if a time is duplicated
                if row_time == previous_row_time:
                    #logging.info('   HM TM Processing: Found duplicate time at {row_time}')
                    info_duplicate += 1
                    append_bool = False
                else:
                    append_bool = True

                if append_bool:
                    tm_list.append(row)

                append_bool = False
                previous_row_time = row_time
                previous_row_data = row_data

    #
    # We report the number of exceptions here
    #
    if info_duplicate:
        logging.warning(f'   HK TM Processing: Found {info_duplicate} duplicate(s)')
    if info_incomplete:
        logging.warning(f'   HK TM Processing: Found {info_incomplete} incomplete data line(s)')


    return(tm_list)


def event_reader(event_file, config):

    event_flag = False    # Event flag definition
    event_list = []
    event_strg = ''
    with open(event_file, 'r') as e:
        for line in e:

            if event_flag and '/>' in line:
                event_flag = False
                if 'PTEL' in event_file:
                    event_strg += '{} '.format(line.strip())
                event_list.append(event_strg)
                event_strg = ''

            if '<CALIB_ROLL_MAG' in line or '<CALIB_ROLL_RPW' in line or '<ROLL' in line:
                event_flag = True

            if event_flag:
                event_strg += '{} '.format(line.strip())

    return(event_list)


def sclk_reader(kernel):

    sclk_delimiter = ''
    sclk_partition = ''
    if kernel.k_type != 'sclk':
        k_type = ''
    else:
        k_type = kernel.k_type + os.sep
    try:
        sclk_kernel = open(os.path.join(kernel.directories.kernels + os.sep + k_type + kernel.config['sclk']), 'r')
    except:
        sclk_kernel = open(os.path.join(kernel.directories.output + os.sep + k_type + kernel.config['sclk']), 'r')
    with sclk_kernel:
        for line in sclk_kernel:
            if 'SCLK01_OUTPUT_DELIM' in line:
                if line.split(')')[0].split('(')[-1].strip() == '1':
                    sclk_delimiter = '.'
                else:
                    sclk_delimiter = ':'
            if 'SCLK_PARTITION_START' in line:
                if ')' in line:
                    sclk_partition = '1'
                else:
                    logging.error(
                        f'   {kernel.config["sclk"]} has more than one partition!')
                    raise
    if not sclk_delimiter or not sclk_partition:
        logging.error(
                f'   {kernel.config["sclk"]} is incorrect!')
        raise

    return(sclk_partition, sclk_delimiter)