import os
import shutil
import logging
import datetime

from datetime import datetime
from datetime import timedelta

from .solo import fecs_processing
from .em16 import oem_to_cog
from .emrsp import oem_extension
from .emrsp import aem_to_quat
from .emrsp import quat_extension
from .bepic import sepm_to_cog
from .bepic import events_to_sep

from .readers import tcp_reader, oem_reader, aem_reader, \
    hk_quaternions2ck_reader, hk_tm2ck_reader, \
    hk_tm2ck_dafcat_reader, oasw_orbit_reader, oasw_attitude_reader, \
    event_reader


def input_writer(input_list, input_file):

    with open(input_file, 'w+') as f:
        for row in input_list:
            if '\n' not in row:
                row = row + '\n'
            f.write(row)



def process_input(kernel, setup, directories, kernels):
    """
    This function processes the input file and extracts the start and finish
    times for the Kernel Name and sets them in the kernel.

    :param kernel: Kernel
    :type kernel: object
    :param source_path: Directory where the sources are located
    :type source_path: str
    """

    # We need to add this because we will change the name of the source
    # a bit  of time from now
    kernel.source_filename = kernel.source

    if kernel.input_type == 'TCP':

        sources = kernel.source
        if not isinstance(sources, list):
            sources = [sources]

        tcp_list = []
        #
        # The source files are processed
        #
        kernel_sources = []
        for source in sources:
            kernel_sources.append(source)

        for source in sorted(kernel_sources):
            tcp_list += tcp_reader(kernel, tcp_file=source, config=kernel.config)

        source = source + '.input'
        input_writer(tcp_list, source)

        # Since the input is processed we need to update the kernel source
        kernel.source = source

        # We want to store the processed source and the original sources
        kernel.source_filename = [kernel.source, sources]

    if kernel.input_type == 'TCP_MEX':

        #
        # we need to merge all the TCPs to create the input for tcp2scet
        #
        sources = kernel.source
        if not isinstance(sources, list):
            sources = [sources]

        tcp_list = []
        #
        # The source files are processed
        #
        kernel_sources = []
        for source in sources:
            kernel_sources.append(source)

        kernel_sources = sorted(kernel_sources)
        merged_tcp = sorted([f for f in os.listdir(directories.processed + '/tcp') if '.MEX' in f])[-1]
        appended_tcp = 'TCP_' + kernel_sources[-1][10:22] + '.MEX'
        if merged_tcp == appended_tcp:
            merged_tcp = sorted([f for f in os.listdir(directories.processed + '/tcp') if '.MEX' in f])[-2]
        shutil.copyfile(directories.processed + '/tcp/' + merged_tcp, appended_tcp)
        for source in kernel_sources:
            with open(appended_tcp, "ab") as file1, \
                    open(source, "rb") as file2:
                file1.write(file2.read())

        # Since the input is processed we need to update the kernel source
        kernel.source = appended_tcp

        # We want to store the processed source and the original sources
        kernel.source_filename = [kernel.source, sources]


    if kernel.input_type == 'TM_QUATERNIONS' or kernel.input_type == 'MAPPS_QUATERNIONS':


        sources = kernel.source
        if not isinstance(sources, list):
            sources = [sources]

        tm_list = []
        #
        # The source files are processed
        #
        kernel_sources = []
        for source in sources:
            kernel_sources.append(source)

        for source in sorted(kernel_sources):
            tm_list += hk_quaternions2ck_reader(kernel, tm_file=source, config=kernel.config)
        #
        # We need to sort the list and remove possible duplicated based on time
        # due to various inputs if needed.
        #
        tm_list_original = tm_list
        tm_list = hk_sort_duplicate_filter(tm_list, kernel.source_config)
        tm_list_filtered = tm_list


        duplicated_lines = len(tm_list_original) - len(tm_list_filtered)

        if duplicated_lines != 0:
            logging.warning('')
            logging.warning(f'   HK TM Processing: Found {duplicated_lines} duplicated line(s)')

        source = source + '.input'
        input_writer(tm_list, source)


        # Since the input is processed we need to update the kernel source
        kernel.source = source


    if kernel.input_type == 'TM':


        sources = kernel.source
        if not isinstance(sources, list):
            sources = [sources]

        tm_list = []

        #
        # The source files are processed
        #
        kernel_sources = []
        for source in sources:
            kernel_sources.append(source)

        for source in sorted(kernel_sources):
            if kernel.config['tm_file_1'][0]['file_schema'] in source:
                tm_list += hk_tm2ck_reader(kernel, tm_file=source, config=kernel.config['tm_file_1'][0])
            elif kernel.config['tm_file_2'][0]['file_schema'] in source:
                tm_list += hk_tm2ck_reader(kernel, tm_file=source, config=kernel.config['tm_file_2'][0])

        #
        # Now we need to fix and sort out the order of the list for the list is not time-ordered
        # Duplicates are removed as well

        #
        # If TM gaps are avoided:
        #
        # We remove the duplicates that appear from different filters acquiring images
        # For that we check the angle value and we remove the duplicates, keeping them
        # in the boundaries
        #
        if kernel.config['fill_tm_gaps'] == 'True':

            tm_list = list(set(tm_list))
            tm_list.sort()

            first_line_with_value = False
            previous_line = False
            delete_list = []
            count = 0

            for line in tm_list:

                if first_line_with_value and first_line_with_value.split()[1:] == line.split()[1:]:

                    if count > 0:
                        delete_list.append(previous_line)

                    count += 1

                else:
                    first_line_with_value = line
                    count = 0

                previous_line = line

            tm_list = list(set(tm_list)-set(delete_list))
            tm_list.sort()

        else:

            tm_list_original = tm_list
            tm_list = hk_sort_duplicate_filter(tm_list, kernel.source_config)
            tm_list_filtered = tm_list

            duplicated_lines = len(tm_list_original) - len(tm_list_filtered)

            if duplicated_lines != 0:
                logging.warning('')
                logging.warning(
                    f'   HK TM Processing: Found {duplicated_lines} duplicated line(s)')


        #
        # We do the TIRVIM post-processing, which might be applicable to other instruments.
        #
        if 'tm_file_2' in kernel.config.keys():
            print('Test')


        #
        # We add the tm_list into the kernel config for we want to dump it in
        # the comments.
        #
        tm_list_string = ''
        for line in tm_list:
            tm_list_string += line
        kernel.config['input_data'] = tm_list_string

        source = source + '.input'
        input_writer(tm_list, source)


        # Since the input is processed we need to update the kernel source
        kernel.source = source


    if kernel.input_type == 'TM_DAFCAT':


        sources = kernel.source
        if not isinstance(sources, list):
            sources = [sources]

        tm_list = []
        #
        # The source files are processed
        #
        kernel_sources = []
        for source in sources:
            kernel_sources.append(source)

        for source in sorted(kernel_sources):
            tm_list += hk_tm2ck_dafcat_reader(kernel, tm_file=source,
                                                  config=kernel.config,
                                                  dafcat_element = kernel.dafcat)

        #
        # We need to sort the list and remove possible duplicated based on time
        # due to various inputs if needed.
        #
        tm_list_original = tm_list
        tm_list = hk_sort_duplicate_filter(tm_list, kernel.source_config)
        tm_list_filtered = tm_list

        duplicated_lines = len(tm_list_original) - len(tm_list_filtered)

        if duplicated_lines != 0:
            logging.warning('')
            logging.warning(f'   HK TM Processing: Found {duplicated_lines} duplicated line(s)')

        #
        # We add the tm_list into the kernel config for we want to dump it in
        # the comments.
        #
        tm_list_string = ''
        for line in tm_list:
            tm_list_string += line
        kernel.config['input_data'] = tm_list_string

        source = source + '_dafcat' + str(kernel.dafcat) + '.input'
        input_writer(tm_list, source)


        # Since the input is processed we need to update the kernel source
        kernel.source = source



    elif kernel.input_type == 'OEM' and kernel.input_processing == 'True':

        #
        # The source files are processed
        #
        oem_list = oem_reader(kernel.source, kernel.config)

        if 'extended_days' in kernel.config:
            oem_extension_list = oem_extension(oem_list, kernel)
            oem_list += oem_extension_list

        source = kernel.source + '.input'
        input_writer(oem_list, source)

        # Since the input is processed we need to update the kernel source
        kernel.source = source


    elif kernel.input_type == 'OEM_MERGED' and kernel.input_processing == 'True':

        sources = kernel.source
        if not isinstance(sources, list):
            sources = [sources]

        oem_list = []

        #
        # The source files are processed
        #
        kernel_sources = []
        for source in sources:
            kernel_sources.append(source)

        for source in sorted(kernel_sources):
            if oem_list:
                oem_list += '\n'
            oem_list += oem_reader(source, kernel.config)

        if 'extended_days' in kernel.config:
            oem_extension_list = oem_extension(oem_list, kernel)
            oem_list += oem_extension_list

        source = source + '_merged.input'
        input_writer(oem_list, source)


        # Since the input is processed we need to update the kernel source
        kernel.source = source


    elif kernel.input_type == 'OASW_ORBIT' and kernel.input_processing == 'True':

        #
        # The source files are processed
        #
        oasw_orbit_list = oasw_orbit_reader(kernel.source, kernel.config)

        source = kernel.source + '.input'
        input_writer(oasw_orbit_list, source)

        # Since the input is processed we need to update the kernel source
        kernel.source = source


    elif kernel.input_type == 'OASW_ATTITUDE':

        kernel.config['sclk_rate'] = \
            str(1.0/float(kernel.config['ticks_per_second']))

        if 'sgs_trim' in kernel.source_config[1] and kernel.source_config[1]['sgs_trim'] == 'True':
            oasw_attitude_reader(kernel.source, trim=int(kernel.source_config[1]['trim_hours']))


    elif kernel.input_type == 'AEM' and kernel.input_processing == 'True':

        kernel.config['sclk_rate'] = \
            str(1.0/float(kernel.config['ticks_per_second']))

        #
        # The source files are processed
        #
        aem_list = aem_reader(kernel.source, kernel.config)

        source = kernel.source + '.input'
        input_writer(aem_list, source)

        # Since the input is processed we need to update the kernel source
        kernel.source = source


    elif 'AEM_QUAT' in kernel.input_type and kernel.input_processing == 'True':

        #
        # The source files are processed
        # For the time being no further filtering is done
        #
        aem_to_quat(kernel.source)
        quat_list = hk_quaternions2ck_reader(kernel, tm_file=kernel.source, config=kernel.config)
        quat_list = aem_zero_interpolation(quat_list)

        if 'MERGED' in kernel.config['input_type'] or 'EXTENDED' in kernel.config['input_type']:
            quat_list = quat_extension(quat_list, kernel)

        source = kernel.source + '.input'
        input_writer(quat_list, source)

        #
        # We add the parameters for the configuration
        #
        if not 'EXTENDED' in kernel.config['input_type']:
            kernel.config["target_frame"] = kernel.source_config[1][kernel.source]["target_frame"]
            kernel.config["object_id"] = kernel.source_config[1][kernel.source]["object_id"]
            kernel.config["mission_accr"] = kernel.source_config[1][kernel.source]["mission_accr"]

        # Since the input is processed we need to update the kernel source
        kernel.source = source


    elif 'COG' in kernel.input_type and kernel.input_processing == 'True':

        #
        # The source files are processed
        # For the time being no further filtering is done
        #
        cog_list = oem_to_cog(kernel.source)
        if not cog_list:
            cog_list = sepm_to_cog(kernel.source)

        source = kernel.source + '.input'
        input_writer(cog_list, source)

        # Since the input is processed we need to update the kernel source
        kernel.source = source


    elif 'SEP' in kernel.input_type and kernel.input_processing == 'True':

        #
        # The source files are processed
        # For the time being no further filtering is done
        #
        sep_list = events_to_sep(kernel.source)

        source = kernel.source + '.input'
        input_writer(sep_list, source)

        # Since the input is processed we need to update the kernel source
        kernel.source = source


    elif kernel.input_type == 'EVENTS':

        #
        # The source files are processed
        #
        event_list = event_reader(kernel.source, kernel.config)

        #
        # SOLO FECS processing
        #
        fecs_processing(event_list, kernel)


        default_attitude =  kernel.config['default_attitude']
        if kernel.config['planning'] == 'True':
            frame_id = \
                setup['specs2ck'][0]['attitude_types'][0][default_attitude][0]['frame_id_plan']
            object_name = \
                setup['specs2ck'][0]['attitude_types'][0][default_attitude][0]['object_name_plan']
        else:
            frame_id = \
                setup['specs2ck'][0]['attitude_types'][0][default_attitude][0]['frame_id']
            object_name = \
                setup['specs2ck'][0]['attitude_types'][0][default_attitude][0]['object_name']


        dt_creation = datetime.now()

        kernel.config['gen_month'] = dt_creation.strftime("%B")
        kernel.config['gen_day'] = str(dt_creation.day)
        kernel.config['gen_year'] = str(dt_creation.year)
        kernel.config['frame_id'] =  frame_id
        kernel.config['object_name'] = object_name


        #
        # We get the latest meta-kernel and use it
        # but be careful, we need the latest LTP orbit as well. If that orbit is
        # being processed in this ADCSng run, it will be included.
        #
        spk_in_run = False
        spk_name = ''
        for kernel_spk in kernels:
            if kernel_spk.k_type == 'spk' and not '-stp' in kernel_spk.name.tokens['kernel_name']:
                spk_in_run = True
                shutil.copy(kernel_spk.working_directory +
                            os.sep + str(kernel_spk.name), str(kernel_spk.name))
                spk_name = str(kernel_spk.name)


        mk_kernel = kernel.config['kernels_dir'] + '/mk/' + kernel.mk
        with open('prediCkt.tm', 'w') as mk_predickt:
            with open(mk_kernel, 'r') as mk:
                kernels_to_load = False
                for line in mk:
                    if '\\begindata' in line: kernels_to_load = False
                    if '\\begindata' in line: kernels_to_load = True
                    if "'..'" in line:
                        mk_predickt.write(line.replace("..",
                                          kernel.config['kernels_dir']))
                    elif line.strip() == ')' and spk_in_run:
                        mk_predickt.write(f"'{spk_name}')")
                    else:
                        if kernels_to_load and len(line.strip()) > 82:
                            line = os.sep.join(
                                    line.split(os.sep)[0:-1]) + \
                                          os.sep + "+'\n" + 27 * " " + "'" + \
                                          line.split(os.sep)[-1]
                        mk_predickt.write(line)
                    #
                    # We also need to obtain the OEM version from the latest LTP SPK in
                    # order to put it in the filename
                    if not spk_name and 'solo_ANC_soc_orbit_' in line:
                        spk_name = line.split(os.sep)[-1].strip()

        #
        # We add the orbit version in the filename
        #
        description = kernel.name.tokens['description']
        orbit_version = spk_name[37:50]
        kernel.name.tokens['description'] = f"{orbit_version}_{description}"

        # Since the input is processed we need to update the kernel source
        # in this case we only use this for the record
        source = kernel.source + '.input'
        input_writer(event_list, source)
        kernel.source = source

    elif kernel.input_type == 'SPECS':

        #
        # We need to obtain the coverage of the SPK and we need to
        # generate an input meta-kernel for the prediCkt run.
        #
        kernel.spk.coverage()
        coverage = kernel.spk.config['coverage']

        coverage = coverage.strip().split('\n')[-1]

        #      2017 APR 06 04:02:53.439            2017 APR 16 05:52:54.693    "

        coverage = coverage.strip()
        # 2017 APR 06 04:02:53.439            2017 APR 16 05:52:54.693"

        start_date = coverage.split('   ')[0].strip().replace(' ', '-') + \
                         coverage.split('   ')[1].strip()

        finish_date = coverage.split('   ')[-1].strip().replace(' ', '-') + \
                         coverage.split('   ')[1].strip()


        #
        # We need to add 1 minute from the start time otherwise it is
        # out of coverage.
        #
        dt = datetime.strptime(start_date,
                               "%Y-%b-%d-%H:%M:%S.%f")
        difference = timedelta(minutes=+1)
        start_dt = dt + difference
        start_date = start_dt.strftime("%Y-%b-%d-%H:%M:%S.%f").upper()

        #
        # We need to substract 1 minute from the finish time otherwise it is
        # out of coverage.
        #
        dt = datetime.strptime(finish_date,
                               "%Y-%b-%d-%H:%M:%S.%f")
        difference = timedelta(minutes=-1)
        finish_dt = dt + difference
        finish_date = finish_dt.strftime("%Y-%b-%d-%H:%M:%S.%f").upper()

        #
        # We need to tune the finish date otherwise we fall out of the bounds
        # of the SPK
        #
        if finish_date.split(':')[-2][-2:] != '00':
           minutes =  int(finish_date.split(':')[-2][-2:])

        spk_in_run = kernel.spk.name.__str__()

        #
        # We obtain the description item for the kernel name from the source of
        # the spk kernel
        #
        description = kernel.spk.name.tokens['description']

        default_attitude =  kernel.config['default_attitude']
        if setup['specs2ck'][0]['attitude_types'][0][default_attitude][0]['planning'] == 'True':
            frame_id = \
                setup['specs2ck'][0]['attitude_types'][0][default_attitude][0]['frame_id_plan']
            object_name = \
                setup['specs2ck'][0]['attitude_types'][0][default_attitude][0]['object_name_plan']
        else:
            frame_id = \
                setup['specs2ck'][0]['attitude_types'][0][default_attitude][0]['frame_id']
            object_name = \
                setup['specs2ck'][0]['attitude_types'][0][default_attitude][0]['object_name']


        dt_creation = datetime.now()

        kernel.config['gen_month'] = dt_creation.strftime("%B")
        kernel.config['gen_day'] = str(dt_creation.day)
        kernel.config['gen_year'] = str(dt_creation.year)
        kernel.config['frame_id'] =  frame_id
        kernel.config['object_name'] = object_name
        kernel.config['start_date'] = start_date
        kernel.config['finish_date'] = finish_date
        kernel.config['spk_input'] = spk_in_run
        kernel.config['coverage'] = coverage
        kernel.config['description'] = description


        mk_kernel = directories.kernels + '/mk/' + kernel.mk
        if not os.path.exists(mk_kernel):
            mk_kernel = directories.kernels + '/mk/former_versions/' + kernel.mk
            try:
                os.path.exists(mk_kernel)
            except ValueError as e:
                print(e)
                raise

        with open('prediCkt.tm', 'w') as mk_predickt:
            with open(mk_kernel, 'r') as mk:
                kernels_to_load = False
                for line in mk:
                    if '\\begindata' in line: kernels_to_load = False
                    if '\\begindata' in line: kernels_to_load = True
                    if "'..'" in line:
                        mk_predickt.write(line.replace("..",
                                          directories.kernels))
                    elif "'../..'" in line:
                        mk_predickt.write(line.replace("../..",
                                          directories.kernels))
                    elif line.strip() == ')' and spk_in_run:
                        mk_predickt.write(f"'{spk_in_run}')")
                    else:
                        if kernels_to_load and len(line.strip()) > 82:
                            line = os.sep.join(
                                    line.split(os.sep)[0:-1]) + \
                                   os.sep + "+'\n" + 27 * " " + "'" + \
                                   line.split(os.sep)[-1]
                        mk_predickt.write(line)

    return


def sclk_order(tm_list):
    # '1/100892218:17278'

    tm_order = []
    for e in tm_list:

        sclk_string = e.split()[0]

        if ':' in sclk_string:
            sep = ':'
        else:
            sep = '.'
        sclk_par = float(sclk_string.split('/')[0])
        sclk_tick = float(sclk_string.split('/')[-1].split(sep)[0])
        sclk_subt = float(f'0.{sclk_string.split(sep)[-1]}')

        tm_order.append(int(sclk_par*(sclk_tick+sclk_subt)*10E6))



    return tm_order


def hk_sort_duplicate_filter(tm_list, source_config):

    tm_list = list(set(tm_list))
    tm_list.sort()
    delete_list = []
    first_line_with_value = ''

    for line in tm_list:
        if first_line_with_value:
            time_first_line = first_line_with_value.split()[0]#.split('.')[0]
            time_current_line = line.split()[0]#.split('.')[0]
            if time_first_line == time_current_line:
                    delete_list.append(previous_line)
            else:
                first_line_with_value = line
        if not first_line_with_value:
            first_line_with_value = line
        previous_line = line

    tm_list = list(set(tm_list) - set(delete_list))

    #
    # Sorting the list is not straightforward for inputs with SCLK clocks
    # For the SCLK string cannot be sorted with the sort() method
    #
    if source_config[1]['time_format'] == 'SCLK':
        tm_order = sclk_order(tm_list)
        tm_list = [x for _, x in sorted(zip(tm_order, tm_list))]
    else:
        tm_list.sort()

    return tm_list


def aem_zero_interpolation(tm_list):

    line_minus_one = ''
    data_list = []
    for line in tm_list:
        if not line_minus_one:
            line_minus_one = line
            continue
        start = line_minus_one.split()[0]
        data = line.split()
        data_list.append(f'{start} {data[0]} {data[1]} {data[2]} {data[3]} '
                         f'{data[4]} 0.0 0.0 0.0 0.0\n')
        line_minus_one = line

    return data_list