# import the required modules
import glob
import os
import json
import re
import copy
import logging
import subprocess
from datetime import datetime, timedelta

import adcsng
import adcsng.classes.kernel as Kernel
from adcsng.classes.exceptions import KernelNotFound
from adcsng.classes.missiondir import MissionDir
from adcsng.classes.comments import Comments
from adcsng.core.solo import events_filter
from adcsng.core.emrsp import mechanism_filter
from adcsng.core.readers import sclk_reader
from adcsng.utils.files import get_exe_dir
from adcsng.utils.files import kernel_type2extension

def generic_kernel_config(config, kernel, source, directories,
                          commnt_template='', fk='', lsk='', sclk='', mk=''):


    #
    # Add the kernels to be loaded. Note that if we are dealing with
    # an SPK, we do not have to load the SCLK.
    #
    kernel.set('ker2load', fk)
    kernel.set('ker2load', lsk)
    kernel.set('ker2load', sclk)
    kernel.set('ker2load', mk)
    #
    # Specify the kernel type and the input type and the originator
    #
    kernel.set('source_config',source)
    kernel.set('source_type', source[1]['kernel_type'])
    kernel.set('input_type', source[1]['input_type'])
    kernel.set('input_processing', source[1]['input_processing'])
    kernel.set('originator', source[1]['originator'])
    if 'orbnum' in source[1]:
        kernel.set('orbnum', source[1]['orbnum'])
    else:
        kernel.set('orbnum', 'False')

    #
    # Create the part of the output file name that we can produced
    # at this stage.
    #
    kernel.name.set('kernel_name', source[1]['kernel_name'])

    #
    # Some kernels don't have description in their filenames:
    #
    if 'description' in source[1]:
        description = '_' + source[1]['description']

    else:
        description = get_description(source[0],
                                      source[1]['input_description_index'])

    kernel.name.set('description', description)
    kernel.config['description'] = description
    kernel.config['adcsng_version'] = config['generic'][0]['adcsng_version']


    #
    # Setup the Comments object if present
    #
    if commnt_template:

        comments = Comments(kernel.config,
                            path=directories.templates,
                            template=commnt_template)

        kernel.set('comments', comments)
        kernel.comments.load_kernels(kernel.ker2load)


    return


def analyze_input_data(path, setup, sclk=False):
    '''
    This function provides a list of sources.

    :param path: Directory where the source files are -incoming directory-
    :type path: str
    :param setup: Mission specific adcsng configuration from json file
    :type setup: dict
    :return: List of sources where each element of the list contains a source
       filename and the corresponding input
    :rtype: list
    '''
    sources = []
    source_files = []

    input_types = setup["input_types"][0]

    for key, value in input_types.items():

        files = glob.glob(path+'/'+ key)
        for filename in files:
            filename = filename.split('/')[-1]
            if sclk:
                if key == setup[value[0]['spacecraft']][0]["tcp_schema"]:
                    sources.append([filename, value[0].copy()])
                    source_files.append(filename)
            elif filename not in source_files:
                if key !=setup[value[0]['spacecraft']][0]["tcp_schema"]:
                    
                    sources.append([filename, value[0].copy()])
                    source_files.append(filename)

                #
                # We need to split the sources for Solar Orbiter and ExoMarsRSP
                #
                if 'solo' in setup.keys():
                    new_filenames = events_filter(path, sources)
                    if new_filenames:
                        for name in new_filenames:
                            if name not in source_files:
                                source_files.append(name)

                if 'rm' in setup.keys():
                    new_filenames = mechanism_filter(path, sources)
                    for name in new_filenames:
                        if name not in source_files:
                            source_files.append(name)

    return sources, source_files


def configure_execution(configuration):
    """
    This function configures the adcsng execution by providing a list of
    kernel objects from the sources -and that need to be generated-, a
    directories object, that contains all the relevant directories to
    be used and a dictionary containing all the configuration information
    from the given mission from the original {mission}.json file.

    In order to provide this information this function scans the incoming
    directory and generates a kernel for each source file. Please note
    that if a given SPK kernel has a default orientation, this file
    is also generated.

    :param mission: Mission for which adcsng is executed
    :type mission: str
    :return: A List of Kernel Objects to be converted by adcsng, A MissionDir
        object that contains all the relevant directories for the execution and
        a dictionary containing the mission specific adcsng JSON configuration
    :rtype: (list, Object, dict)
    """

    #
    # We determine the platform system and machine for the executables
    #
    executables_dir = get_exe_dir()

    root_dir = os.path.dirname(adcsng.command_line.__file__)

    #
    # We check that the fields of the configuration are present
    #
    with open(root_dir + '/config/config.json', 'r') as f:
        config_template = json.load(f)

    for key in config_template.keys():
        if key not in configuration.keys():
            error_message = "Error: The ADCSng JSON configuration file has missing keys."
            raise NameError(error_message)

    configuration['root_dir'] = root_dir
    configuration['executables_dir'] = root_dir + executables_dir

    #
    # We read the mission configuration
    #
    config = get_config(configuration)

    #
    # To avoid repetition we create the setup variable for the generic part of
    # the configuration dictionary
    #
    setup = config['generic'][0]
    adcsng_config = configuration


    #
    # We implement the possibility of providing relative paths in the
    # configuration (in order to implement test)
    #
    if not os.path.isabs(adcsng_config['source_dir']):
        source = os.path.join(os.getcwd(), adcsng_config['source_dir'])
    else: source = adcsng_config['source_dir']
    if not os.path.isabs(adcsng_config['output_dir']):
        output = os.path.join(os.getcwd(), adcsng_config['output_dir'])
    else: output = adcsng_config['output_dir']
    if adcsng_config['processed_dir']:
        if not os.path.isabs(adcsng_config['processed_dir']):
            processed = os.path.join(os.getcwd(), adcsng_config['processed_dir'])
        else: processed = adcsng_config['processed_dir']
    else:
        processed = ''
    if not os.path.isabs(adcsng_config['kernels_dir']):
        kernels = os.path.join(os.getcwd(), adcsng_config['kernels_dir'])
    else: kernels = adcsng_config['kernels_dir']
    if not os.path.isabs(adcsng_config['temp_dir']):
        temp = os.path.join(os.getcwd(), adcsng_config['temp_dir'])
    else: temp = adcsng_config['temp_dir']
    if not os.path.isabs(adcsng_config['log_dir']):
        log = os.path.join(os.getcwd(), adcsng_config['log_dir'])
    else: log =  adcsng_config['log_dir']
    try:
        if not os.path.isabs(adcsng_config['exec_log']):
            exec_log = os.path.join(os.getcwd(), adcsng_config['exec_log'])
        else: exec_log =  adcsng_config['exec_log']
    except:
        exec_log = ''
    try:
        if not os.path.isabs(adcsng_config['reports_dir']):
            reports = os.path.join(os.getcwd(), adcsng_config['reports_dir'])
        else: reports =  adcsng_config['reports_dir']
    except:
        reports = ''

    #
    # For convenience we generate an object that contains all the relevant
    # directories and the existance of the directories is checked.
    #
    directories = MissionDir(config,
                             mission=configuration['mission'],
                             root_dir=adcsng_config['root_dir'],
                             source=source,
                             output=output,
                             processed=processed,
                             kernels=kernels,
                             temp=temp,
                             templates='/$root_dir/etc',
                             executables=adcsng_config['executables_dir'],
                             log=log,
                             exec_log=exec_log,
                             reports=reports
                             )



    return setup, config, directories


def sources2kernels(setup, config, directories,
                    sclk=False, original_sources=''):

    if not original_sources:
        original_sources = []
    #
    # We obtain the latest LSK present in the mission directories.
    #
    lsk = os.path.join('lsk',
          get_latest_kernel('lsk',directories.kernels, setup['lsk_schema']))

    #
    # The input data is analyzed to know which type of data it is and what kind
    # of kernel is to be generated. Note that if sclk is set to False,
    # TCP will not be searched for.
    #
    (input_data, source_files) = analyze_input_data(directories.source, setup, sclk=sclk)
    original_sources += source_files


    kernels                 = []
    source_tcp              = []
    sources_tm_quat         = []
    sources_mapps_quat      = []
    sources_tm              = []
    sources_tm_daf          = []
    sources_oem             = []
    sources_aem_quat        = []
    sources_csmithed        = []
    source_types_tm_daf     = []
    source_types_tm_quat    = []
    source_types_aem_quat   = []
    source_types_tm         = []
    source_types_mapps_quat = []
    input_data_elements     = len(input_data)

    index = 1
    for source in input_data:

        #
        # We discard the sources that do not come from the appropriate sources
        # directory
        #
        if 'input_directory' in source[1]:
            if source[1]['input_directory'] != directories.source.split(os.sep)[-1]:
                original_sources.remove(source[0])
                logging.warning(f'Source file: {source[0]} will not be processed.')
                logging.warning(f'The source is only accepted from: {source[1]["input_directory"]}')
                logging.info('')
                continue


        if source[1]['input_type'] == 'TCP' or source[1]['input_type'] == 'TCP_MEX':

            source_tcp.append(source)


        if source_tcp and index == input_data_elements:

            #
            # We determine if several sources need to be merged
            #
            merged_sources = []
            if config['tcp2sclk'][0]['merge_sources'] == 'True':
                merged_sources = source_tcp

            sources = []

            for src in merged_sources:
                    sources.append(src[0])

            kernel = Kernel.SCLK(sources, directories,
                                    Kernel.KernelName(source[1]['kernel_schema']))

            if source[1]['input_type'] == 'TCP_MEX':
                merged_sources = sorted(merged_sources)
                source = merged_sources[-1]

            kernel.set('config', source[1])

            #
            # We need to load the previous version of the SCLK, it's an input
            #
            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                         setup[source[1]['spacecraft']][0]['sclk_step_schema'])
            if source[1]['input_type'] == 'TCP_MEX' and sclk[:3] != 'sclk':
                sclk = 'sclk/' + sclk
                sclkid = re.findall('\d+', sclk)[-1]
                kernel.config['sclkid'] = sclkid

            if source[1]['input_type'] == 'TCP_MEX':
                commnt_template = 'makclk_mex.commnt'
                dt_creation = datetime.now()
                kernel.config['gen_utc'] = dt_creation.strftime("%Y-%m-%dT%H:%M:%S")
                kernel.config['mission_name'] = kernel.config['main_sc_name'].upper()
                kernel.config['filename'] = str(kernel.name)
                kernel.config['sclk_kernel'] = sclk
                kernel.config['lsk_kernel'] = lsk
                kernel.config['bad_tcps'] = config['tcp2scet'][0]['bad_tcps']
            else:
                commnt_template = 'tcp2sclk.commnt'


            generic_kernel_config(config, kernel, source, directories,
                                      commnt_template=commnt_template,
                                      lsk=lsk, sclk=sclk)

            if source[1]['input_type'] != 'TCP_MEX':

                (sclk_partition, sclk_delimiter) = sclk_reader(kernel)

                #
                # We now check if this coincides with configuration if not the
                # message is shown later to be in the run.
                #
                kernel.config['previous_sclk_separator'] = sclk_delimiter

                if sclk_delimiter == ':':
                    kernel.config['sclk_separator_id'] = 2
                elif sclk_delimiter == '.':
                    kernel.config['sclk_separator_id'] = 1
                else:
                    raise Exception('SCLK separator not specified in Configuration file.')

            kernels.append(kernel)

        #
        # We create the corresponding Kernel object
        #
        if source[1]['input_type'] == 'AEM':


            kernel = Kernel.CK(source[0], directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            #
            # We introduce the possibility of having a planning CK
            #
            if source[1]['planning'] == 'True':
                sclk_schema = 'sclk_fict_schema'
                commnt_template = 'aem2ck_plan.commnt'
            else:
                sclk_schema = 'sclk_step_schema'
                commnt_template = 'aem2ck.commnt'

            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                     setup[source[1]['spacecraft']][0][sclk_schema])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid

            kernel.name.set('sclkid', sclkid)
            kernel.set('config', config['aem2ck'][0])

            sclk = os.path.join('sclk', sclk)

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk, sclk=sclk)

            kernels.append(kernel)


        #
        # ExoMars RSP is a special case
        #
        if source[1]['input_type'] == 'OEM':

            kernel = Kernel.SPK(source[0], directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            kernel.set('config', config['oem2spk'][0])

            commnt_template = 'oem2spk.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            generic_kernel_config(config, kernel,source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            try:
                kernel.config['extended_days'] = setup['rm'][0]['extended_days']
            except: pass

            kernels.append(kernel)


        if source[1]['input_type'] == 'COG':

            kernel = Kernel.SPK(source[0], directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            commnt_template = 'mkspk.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            kernel.set('config', config['mkspk'][0])

            generic_kernel_config(config, kernel,source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernel.set('config', source[1])

            kernels.append(kernel)


        if source[1]['input_type'] == 'SEP':

            kernel = Kernel.CK(source[0], directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            commnt_template = 'msopck.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            sclk_schema = 'sclk_step_schema'
            sclk = os.path.join('sclk', get_latest_kernel('sclk', directories.kernels,
                                                          setup[source[1]['spacecraft']][0][sclk_schema]))

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.name.set('sclkid', sclkid)

            kernel.config['sclkid'] = sclkid

            kernel.config['ck_type'] = '2'
            kernel.set('config', config['msopck_sep'][0])

            generic_kernel_config(config, kernel, source,
                                  directories, sclk=sclk, lsk=lsk,
                                  fk=fk, commnt_template=commnt_template,)

            kernel.set('config', source[1])

            kernels.append(kernel)


        if source[1]['input_type'] == 'OASW_ORBIT':

            kernel = Kernel.SPK(source[0], directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            kernel.set('config', config['mex2ker_orbit'][0])

            commnt_template = 'mex2ker_orbit.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            generic_kernel_config(config, kernel,source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernels.append(kernel)


        if source[1]['input_type'] == 'OASW_ATTITUDE':

            kernel = Kernel.CK(source[0], directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            #
            # We introduce the possibility of having a planning CK
            #
            if source[1]['planning'] == 'True':
                sclk_schema = 'sclk_fict_schema'
                commnt_template = 'mex2ker_attitude_plan.commnt'
            else:
                sclk_schema = 'sclk_step_schema'
                commnt_template = 'mex2ker_attitude.commnt'

            kernel.set('config', config['mex2ker_attitude'][0])

            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                     setup[source[1]['spacecraft']][0][sclk_schema])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid

            kernel.name.set('sclkid', sclkid)

            sclk = os.path.join('sclk', sclk)

            generic_kernel_config(config, kernel,source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk, sclk=sclk)


            #
            # do sgs triming
            #
            if 'sgs_trim' in source[1] and source[1]['sgs_trim'] == 'True':
                kernelsgs = copy.deepcopy(kernel)
                kernelsgs.name = str(kernel.name).split('_', 1)[0] + '_SGS_' + str(kernel.name).split('_', 1)[1]
                kernelsgs.source = kernel.source.split('_', 1)[0] + '_SGS_' + kernel.source.split('_', 1)[1]
                kernelsgs.source_config[1]['type6_conversion'] = 'True'
                kernelsgs.source_config[1]['type6_token'] = ''
                kernelsgs.source_config[1]['type5_keep'] = 'False'
                kernels.append(kernelsgs)

            kernel.source_config[1]['sgs_trim'] = 'False'
            kernels.append(kernel)



        if source[1]['input_type'] == 'SPK13':

            kernel = Kernel.SPK(source[0], directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            kernel.set('config', config['oem2spk'][0])

            commnt_template = 'oem2spk.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernels.append(kernel)


        if source[1]['input_type'] == 'SPK18':

            kernel = Kernel.SPK(source[0], directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            kernel.set('config', config['mex2ker_orbit'][0])

            commnt_template = 'mex2ker_orbit.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernels.append(kernel)


        if source[1]['input_type'] == 'CK5':

            kernel = Kernel.CK(source[0], directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            #
            # We introduce the possibility of having a planning CK
            #
            if source[1]['planning'] == 'True':
                sclk_schema = 'sclk_fict_schema'
                commnt_template = 'mex2ker_attitude_plan.commnt'
            else:
                sclk_schema = 'sclk_step_schema'
                commnt_template = 'mex2ker_attitude.commnt'

            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                     setup[source[1]['spacecraft']][0][sclk_schema])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid

            kernel.name.set('sclkid', sclkid)

            kernel.config['spacecraft'] = source[1]['spacecraft']

            sclk = os.path.join('sclk', sclk)

            generic_kernel_config(config, kernel,source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk, sclk=sclk)

            kernels.append(kernel)


        if source[1]['input_type'] == 'CK':

            kernel = Kernel.CK(source[0], directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))


            sclk = get_from_kernel('sclk', kernel)

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid

            kernel.name.set('sclkid', sclkid)

            kernel.config['spacecraft'] = source[1]['spacecraft']

            sclk = os.path.join('sclk', sclk)

            commnt_template = 'aem2ck.commnt'

            generic_kernel_config(config, kernel,source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk, sclk=sclk)

            kernels.append(kernel)


        if source[1]['input_type'] == 'C-SMITHED':

            sources_csmithed.append(source)


        if source[1]['input_type'] == 'OEM_MERGED':
            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            sources_oem.append(source)

            #
            # We also generate the individual SPK!
            # For it we need to change the source configuration
            #
            for input_type in setup['input_types'][0]:
                if input_type.split('*')[0] in source[0] and \
                   input_type.split('*')[0] != 'emrsp_rm_mes' and \
                        setup['input_types'][0][input_type][0]['kernel_type'] != 'ck':
                    new_source = [source[0], setup['input_types'][0][input_type][0]]

            kernel = Kernel.SPK(new_source[0], directories,
                                 Kernel.KernelName(new_source[1]['kernel_schema']))

            kernel.set('config', config['oem2spk'][0])
            kernel.config['append_kernel'] = new_source[1]['append_kernel']
            kernel.config['kernel_name'] = new_source[1]['kernel_name']
            kernel.config['kernel_schema'] = new_source[1]['kernel_schema']

            commnt_template = 'oem2spk.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[new_source[1]['spacecraft']][
                                                    0]['fk_schema']))

            generic_kernel_config(config, kernel,new_source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernels.append(kernel)


        # When all the source files have been processed, we obtain one kernel
        # object out of the tm. Later on, this can result into
        # multiple kernels.
        if sources_oem and index == input_data_elements:

            sources = []
            for source_oem in sources_oem:
                sources.append(source_oem[0])

            source_cfg = sources_oem[0]

            kernel = Kernel.SPK(sources, directories,
                                Kernel.KernelName(
                                source_cfg[1]['kernel_schema']))

            kernel.set('config', config['oem2spk'][0])
            kernel.config['append_kernel'] = source[1]['append_kernel']
            kernel.config['kernel_name'] = source[1]['kernel_name']
            kernel.config['kernel_schema'] = source[1]['kernel_schema']

            kernel.config['extended_days'] = setup['rm'][0]['extended_days']

            commnt_template = 'oem2spk.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk',
                                                directories.kernels,
                                                setup[source[1]['spacecraft']][0]['fk_schema']))

            generic_kernel_config(config, kernel,source_cfg, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernels.append(kernel)


        if source[1]['input_type'] == 'TM_QUATERNIONS':

            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            sources_tm_quat.append(source)
            source_types_tm_quat.append(source[1]['kernel_name'])

        if source[1]['input_type'] == 'MAPPS_QUATERNIONS':

            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            sources_mapps_quat.append(source)
            source_types_mapps_quat.append(source[1]['kernel_name'])

        if source[1]['input_type'] == 'AEM_QUAT_EXTENDED':

            kernel = Kernel.CK(source[0], directories, Kernel.KernelName(
                    source[1]['kernel_schema']))

            fk = os.path.join('fk', get_latest_kernel('fk', directories.kernels,
                                                      setup[source[1][
                                                          'spacecraft']][0][
                                                          'fk_schema']))

            sclk = get_latest_kernel('sclk',
                                     [directories.kernels, directories.output],
                                     setup[source[1]['spacecraft']][0][
                                         'sclk_step_schema'])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid
            if source[1]['ck_type'] == '2':
                kernel.config['angular_rate_present'] = 'YES'
            else:
                kernel.config[
                    'angular_rate_present'] = 'MAKE UP/NO AVERAGING'

            kernel.name.set('sclkid', sclkid)


            kernel.set('config', config['msopck_quat'][0])
            kernel.set('config', source[1])
            kernel.config['extended_days'] = setup['rm'][0][
                'extended_days']

            sclk = os.path.join('sclk', sclk)

            commnt_template = 'msopck_quat.commnt'

            generic_kernel_config(config, kernel, source,
                                  directories,
                                  fk=fk, lsk=lsk, sclk=sclk,
                                  commnt_template=commnt_template)

            kernels.append(kernel)


        if source[1]['input_type'] == 'AEM_QUAT':

            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            sources_aem_quat.append(source)
            source_types_aem_quat.append(source[1]['kernel_name'])

        # When all the source files have been processed, we obtain one kernel
        # object out of the tm_quaternions. Later on, this can result into
        # multiple kernels.
        if sources_aem_quat and index == input_data_elements:

            for source in sources_aem_quat:

                for source_file in source[1]['source_list']:

                    kernel = Kernel.CK(source_file, directories,Kernel.KernelName(
                                       source[1]['kernel_schema']))

                    fk = os.path.join('fk', get_latest_kernel('fk',directories.kernels,
                                                        setup[source[1]['spacecraft']][0]['fk_schema']))

                    sclk = get_latest_kernel('sclk',[directories.kernels, directories.output],setup[source[1]['spacecraft']][0][
                                             'sclk_step_schema'])

                    sclkid = re.findall('\d+', sclk)[-1]

                    kernel.config['sclkid'] = sclkid
                    if source[1]['ck_type'] == '2':
                        kernel.config['angular_rate_present'] = 'YES'
                    else:
                        kernel.config[
                            'angular_rate_present'] = 'MAKE UP/NO AVERAGING'

                    kernel.name.set('sclkid', sclkid)

                    #
                    # The name needs to be updated.
                    #
                    if len(source[1]['source_list']) > 1:
                        kernel.name.xkma = kernel.source.split('.')[0] + '.bc'

                    kernel.set('config', config['msopck_quat'][0])
                    kernel.set('config', source[1])
                    kernel.config['extended_days'] = setup['rm'][0][
                        'extended_days']

                    sclk = os.path.join('sclk', sclk)

                    commnt_template = 'msopck_quat.commnt'


                    generic_kernel_config(config, kernel,source,
                                          directories,
                                          fk=fk, lsk=lsk, sclk=sclk,
                                          commnt_template=commnt_template)

                    kernels.append(kernel)

                    #
                    # We need to generate the interpolation duplicate
                    #
                    kernel_interp = copy.deepcopy(kernel)
                    kernel_interp.config['input_type'] = 'AEM_QUAT_MERGED'
                    kernel_interp.set('interp_config' , setup['input_types'][0]['emrsp_rm_mes_interp*.aem'])
                    generic_kernel_config(config, kernel_interp,source,
                                          directories,
                                          fk=fk, lsk=lsk, sclk=sclk,
                                          commnt_template=commnt_template)
                    kernels.append(kernel_interp)


        # When all the source files have been processed, we obtain one kernel
        # object out of the tm_quaternions. Later on, this can result into
        # multiple kernels.
        if sources_tm_quat and index == input_data_elements:

            #
            # We need to distinguish in between source types.
            #
            source_types_tm_quat = list(set(source_types_tm_quat))

            for source_type in source_types_tm_quat:

                merge_flag = False

                for source_tm_quat in sources_tm_quat:
                    if source_type == source_tm_quat[1]['kernel_name'] and not merge_flag:

                        sources = []

                        #
                        # We determine if several sources need to be merged
                        #
                        if source_tm_quat[1]['merge_sources'] == 'True':
                            merged_sources = sources_tm_quat
                            for merged_source_tm_quat in merged_sources:
                                if source_type == merged_source_tm_quat[1]['kernel_name']:
                                    sources.append(merged_source_tm_quat[0])
                                    merge_flag = True
                                    source_cfg = merged_source_tm_quat

                        #
                        # We determine if we have to concatenate kernels
                        #
                        generate_single_kernel = False
                        if 'merge_kernel' in source_tm_quat[1].keys():
                            if source_tm_quat[1]['merge_kernel'] == 'True':

                                #
                                # We obtain the sources
                                #
                                sources = source_tm_quat[1]['source_list']

                                index = 0
                                for source_tm_quat_daf in sources:

                                    source_cfg = source_tm_quat

                                    #
                                    # We generate concatenated kernel i
                                    #
                                    kernel = Kernel.CK(source_tm_quat_daf, directories,
                                                       Kernel.KernelName(source_tm_quat[1]['kernel_schema'].split('.')[0]+'_{0:03}.bc'.format(index)))

                                    fk = os.path.join('fk',
                                                      get_latest_kernel('fk', directories.kernels,
                                                                        setup[source_tm_quat[1]['spacecraft']][0]['fk_schema']))

                                    sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                                             setup[source_tm_quat[1]['spacecraft']][0]['sclk_step_schema'])

                                    sclkid = re.findall('\d+', sclk)[-1]

                                    kernel.config['sclkid'] = sclkid
                                    if source[1]['ck_type'] == '2':
                                        kernel.config[
                                            'angular_rate_present'] = 'YES'
                                    else:
                                        kernel.config['angular_rate_present'] = 'MAKE UP/NO AVERAGING'


                                    kernel.name.set('sclkid', sclkid)
                                    kernel.set('config', config['msopck_quat'][0])
                                    kernel.set('config', source_tm_quat[1])

                                    sclk = os.path.join('sclk', sclk)

                                    commnt_template = 'msopck_quat.commnt'

                                    generic_kernel_config(config, kernel,source_cfg,
                                                          directories,
                                                          fk=fk, lsk=lsk, sclk=sclk,
                                                          commnt_template=commnt_template)

                                    kernels.append(kernel)
                                    index += 1
                            #
                            # We need to distinguish the special case of Solar Orbiter
                            # were we need to split the generation for the interpolation
                            # intervals
                            #
                            elif 'source_list' in source_tm_quat[1]:
                                if len(source_tm_quat[1]['source_list']) >= 1: # or "merge_intervals":"True"
                                    #
                                    # We obtain the sources
                                    #
                                    sources = source_tm_quat[1]['source_list']

                                    index = 0
                                    for source_tm_quat_daf in sources:

                                        source_cfg = source_tm_quat

                                        #
                                        # We generate concatenated kernel i
                                        #
                                        kernel = Kernel.CK(source_tm_quat_daf,
                                                           directories,
                                                           Kernel.KernelName(
                                                               source_tm_quat[1][
                                                                   'kernel_schema'].split(
                                                                   '.')[
                                                                   0] + '_{0:03}.bc'.format(
                                                                   index)))

                                        fk = os.path.join('fk',
                                                          get_latest_kernel('fk',
                                                                            directories.kernels,
                                                                            setup[
                                                                                source_tm_quat[
                                                                                    1][
                                                                                    'spacecraft']][
                                                                                0][
                                                                                'fk_schema']))

                                        sclk = get_latest_kernel('sclk', [
                                            directories.kernels,
                                            directories.output],
                                                                 setup[
                                                                     source_tm_quat[
                                                                         1][
                                                                         'spacecraft']][
                                                                     0][
                                                                     'sclk_step_schema'])

                                        sclkid = re.findall('\d+', sclk)[-1]

                                        kernel.config['sclkid'] = sclkid
                                        if source[1]['ck_type'] == '2':
                                            kernel.config[
                                                'angular_rate_present'] = 'YES'
                                        else:
                                            kernel.config[
                                                'angular_rate_present'] = 'MAKE UP/NO AVERAGING'

                                        kernel.name.set('sclkid', sclkid)
                                        kernel.set('config',
                                                   config['msopck_quat'][0])
                                        kernel.set('config', source_tm_quat[1])

                                        sclk = os.path.join('sclk', sclk)

                                        commnt_template = 'msopck_quat.commnt'

                                        generic_kernel_config(config, kernel,
                                                              source_cfg,
                                                              directories,
                                                              fk=fk, lsk=lsk,
                                                              sclk=sclk,
                                                              commnt_template=commnt_template)

                                        kernels.append(kernel)
                                        index += 1
                                else:
                                    source_cfg = source_tm_quat
                                    generate_single_kernel = True
                            else:
                                source_cfg = source_tm_quat
                                generate_single_kernel = True


                        else:
                            source_cfg = source_tm_quat
                            generate_single_kernel = True


                        if generate_single_kernel:

                            kernel = Kernel.CK(sources, directories,
                                            Kernel.KernelName(source_tm_quat[1]['kernel_schema']))

                            fk = os.path.join('fk',
                                          get_latest_kernel('fk',
                                                            directories.kernels,
                                                            setup[source_tm_quat[1][
                                                                'spacecraft']][
                                                                0][
                                                                'fk_schema']))

                            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                                setup[source_tm_quat[1]['spacecraft']][0]['sclk_step_schema'])


                            sclkid = re.findall('\d+', sclk)[-1]

                            kernel.config['sclkid'] = sclkid
                            kernel.config['ck_type'] = '3'
                            kernel.config['angular_rate_present'] = 'MAKE UP/NO AVERAGING'

                            kernel.name.set('sclkid', sclkid)
                            kernel.set('config', config['msopck_quat'][0])
                            kernel.set('config', source_cfg[1])


                            sclk = os.path.join('sclk', sclk)

                            commnt_template = 'msopck_quat.commnt'

                            generic_kernel_config(config, kernel,source_cfg, directories,
                                                  fk=fk, lsk=lsk, sclk=sclk,
                                                  commnt_template=commnt_template)

                            kernels.append(kernel)


        if sources_csmithed and index == input_data_elements:

            csmithed = []
            for source_csmithed in sources_csmithed:
                csmithed.append(source_csmithed[0])

            csmithed.sort()

            source_cfg = sources_csmithed[0]
            kernel = Kernel.CK(csmithed, directories, Kernel.KernelName(source_cfg[1]['kernel_schema']))

            fk = os.path.join('fk',get_latest_kernel('fk',directories.kernels,setup[source_cfg[1]['spacecraft']][0]['fk_schema']))

            sclk = get_latest_kernel('sclk', [directories.kernels,directories.output],setup[source_cfg[1]['spacecraft']][0]['sclk_step_schema'])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid
            kernel.config['ck_type'] = '3'

            kernel.name.set('sclkid', sclkid)
            kernel.set('config', config['msopck_quat'][0])
            kernel.set('config', source_cfg[1])

            sclk = os.path.join('sclk', sclk)

            commnt_template = 'msopck_quat.commnt'

            generic_kernel_config(config, kernel, source_cfg,
                                          directories,
                                          fk=fk, lsk=lsk, sclk=sclk,
                                          commnt_template=commnt_template)

            kernels.append(kernel)


        if sources_mapps_quat and index == input_data_elements:

            #
            # We need to distinguish in between source types.
            #
            source_types_mapps_quat = list(set(source_types_mapps_quat))

            for source_type in source_types_mapps_quat:

                merge_flag = False

                for source_mapps_quat in sources_mapps_quat:
                    if source_type == source_mapps_quat[1][
                        'kernel_name'] and not merge_flag:

                        sources = []

                        #
                        # We determine if several sources need to be merged
                        #
                        if source_mapps_quat[1]['merge_sources'] == 'True':
                            merged_sources = sources_mapps_quat
                            for source_mapps_quat in merged_sources:
                                if source_type == source_mapps_quat[1]['kernel_name']:
                                    sources.append(source_mapps_quat[0])
                                    merge_flag = True
                                    source_cfg = source_mapps_quat
                        else:
                            sources = source_mapps_quat[0]
                            source_cfg = source_mapps_quat

                        kernel = Kernel.CK(sources, directories,
                                            Kernel.KernelName(source_mapps_quat[1]['kernel_schema']))

                        fk = os.path.join('fk',
                                          get_latest_kernel('fk',
                                                            directories.kernels,
                                                            setup[source[1][
                                                                'spacecraft']][
                                                                0][
                                                                'fk_schema']))

                        sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                                setup[source[1]['spacecraft']][0]['sclk_step_schema'])


                        sclkid = re.findall('\d+', sclk)[-1]

                        kernel.config['sclkid'] = sclkid
                        kernel.config['ck_type'] = '3'

                        kernel.name.set('sclkid', sclkid)
                        kernel.set('config', config['msopck_quat'][0])
                        kernel.set('config', source_cfg[1])


                        sclk = os.path.join('sclk', sclk)

                        commnt_template = 'msopck_quat.commnt'

                        generic_kernel_config(config, kernel,source_cfg, directories,
                                              fk=fk, lsk=lsk, sclk=sclk,
                                              commnt_template=commnt_template)

                        kernels.append(kernel)


        if source[1]['input_type'] == 'TM':

            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            sources_tm.append(source)
            source_types_tm.append(source[1]['kernel_name'])


        # When all the source files have been processed, we obtain one kernel
        # object out of the tm. Later on, this can result into
        # multiple kernels.
        if sources_tm and index == input_data_elements:

            #
            # We need to distinguish in between source types.
            #
            source_types_tm = list(set(source_types_tm))

            for source_type in source_types_tm:

                merge_flag = False

                for source_tm in sources_tm:
                    if source_type == source_tm[1][
                        'kernel_name'] and not merge_flag:

                        spacecraft = source_tm[1]['spacecraft']

                        sources = []

                        #
                        # We determine if several sources need to be merged
                        #
                        if source_tm[1]['merge_sources'] == 'True':
                            merged_sources = sources_tm
                            for src in merged_sources:
                                if source_type == source_tm[1]['kernel_name']:
                                    sources.append(src[0])
                                    merge_flag = True
                                    source_cfg = src
                        else:
                            sources = source_tm[0]
                            source_cfg = source_tm

                        kernel = Kernel.CK(sources, directories,
                                            Kernel.KernelName(source_tm[1]['kernel_schema']))

                        fk = os.path.join('fk',
                                          get_latest_kernel('fk',
                                                            directories.kernels,
                                                            setup[spacecraft][
                                                                0][
                                                                'fk_schema']))

                        sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                                setup[spacecraft][0]['sclk_step_schema'])


                        sclkid = re.findall('\d+', sclk)[-1]

                        kernel.config['sclkid'] = sclkid

                        kernel.name.set('sclkid', sclkid)


                        kernel.set('config', source_cfg[1])
                        kernel.set('config', config['msopck'][0])

                        kernel.config['ck_type'] = '3'
                        kernel.config['angular_rate_present'] = 'MAKE UP/NO AVERAGING'

                        sclk = os.path.join('sclk', sclk)

                        commnt_template = 'msopck.commnt'

                        generic_kernel_config(config, kernel,source_cfg, directories,
                                              fk=fk, lsk=lsk, sclk=sclk,
                                              commnt_template=commnt_template)

                        kernels.append(kernel)

        if source[1]['input_type'] == 'TM_DAFCAT':

            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            sources_tm_daf.append(source)
            source_types_tm_daf.append(source[1]['kernel_name'])

        # When all the source files have been processed, we obtain one kernel
        # object out of the tm. Later on, this can result into
        # multiple kernels.
        if sources_tm_daf and index == input_data_elements:

            #
            # We need to distinguish in between source types.
            #
            source_types_tm_daf = list(set(source_types_tm_daf))

            for source_type in source_types_tm_daf:

                merge_flag = False

                for source_tm_daf in sources_tm_daf:
                    if source_type == source_tm_daf[1]['kernel_name'] and not merge_flag:

                        sources = []

                        #
                        # We determine if several sources need to be merged
                        #
                        if source_tm_daf[1]['merge_sources'] == 'True':
                            merged_sources = sources_tm_daf
                            for source_tm_daf_merged in merged_sources:
                                if source_type == source_tm_daf_merged[1]['kernel_name']:
                                    sources.append(source_tm_daf_merged[0])
                                    merge_flag = True
                                    source_cfg = source_tm_daf_merged
                        else:
                            sources = source_tm_daf[0]
                            source_cfg = source_tm_daf

                        #
                        # We generate DAFCAT kernel 1
                        #
                        kernel = Kernel.CK(sources, directories,
                                            Kernel.KernelName(source_tm_daf[1]['kernel_schema']))

                        fk = os.path.join('fk',
                                          get_latest_kernel('fk',
                                                            directories.kernels,
                                                            setup[source_tm_daf[1][
                                                                'spacecraft']][
                                                                0][
                                                                'fk_schema']))

                        sclk = get_latest_kernel('sclk',[directories.kernels, directories.output],
                                                 setup[source_tm_daf[1]['spacecraft']][0]['sclk_step_schema'])

                        sclkid = re.findall('\d+', sclk)[-1]

                        kernel.name.set('sclkid', sclkid)

                        kernel.config['sclkid'] = sclkid

                        kernel.set('config', source_cfg[1])
                        kernel.set('config', config['msopck'][0])

                        kernel.config['ck_type'] = '3'
                        kernel.config['angular_rate_present'] = 'MAKE UP/NO AVERAGING'
                        kernel.set('dafcat', 1)
                        kernel.name.set('dafcat', "1")

                        sclk = os.path.join('sclk', sclk)

                        commnt_template = 'msopck_dafcat_1.commnt'

                        generic_kernel_config(config, kernel,source_cfg, directories,
                                              fk=fk, lsk=lsk, sclk=sclk,
                                              commnt_template=commnt_template)

                        kernels.append(kernel)


                        #
                        # We generate DAFCAT kernel 2
                        #
                        kernel = Kernel.CK(sources, directories,
                                                Kernel.KernelName(
                                                        source_tm_daf[1][
                                                            'kernel_schema']))

                        kernel.name.set('sclkid', sclkid)
                        kernel.config['sclkid'] = sclkid


                        kernel.set('config', source_cfg[1])
                        kernel.set('config', config['msopck'][0])

                        kernel.set('dafcat', 2)
                        kernel.name.set('dafcat', "2")
                        kernel.config['ck_type'] = '3'
                        kernel.config['angular_rate_present'] = 'MAKE UP/NO AVERAGING'

                        commnt_template = 'msopck_dafcat_2.commnt'

                        generic_kernel_config(config, kernel,source_cfg,
                                                  directories,
                                                  fk=fk, lsk=lsk, sclk=sclk,
                                                  commnt_template=commnt_template)

                        kernels.append(kernel)

        index += 1

        if source[1]['input_type'] == 'EVENTS':

                specs2ck_config = config['specs2ck'][0]['attitude_types']
                attitude_type = source[1]['default_attitude']


                kernel = Kernel.CK(source[0], directories,
                                    Kernel.KernelName(
                                            source[1]['kernel_schema']))

                #
                # We introduce the possibility of having a planning CK
                #
                if source[1]['planning'] == 'False':
                    sclk_schema = 'sclk_step_schema'
                else:
                    sclk_schema = 'sclk_fict_schema'

                fk = os.path.join('fk',
                                  get_latest_kernel('fk', directories.kernels,
                                                    setup[source[1][
                                                        'spacecraft']][0][
                                                        'fk_schema']))

                sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                         setup[source[1]['spacecraft']][0][
                                             sclk_schema])

                sclkid = re.findall('\d+', sclk)[-1]

                kernel.name.set('sclkid', sclkid)

                kernel.set('config', specs2ck_config[0][attitude_type][0])

                kernel.config['sclkid'] = sclkid

                sclk = os.path.join('sclk', sclk)


                mk = get_latest_kernel('mk', directories.kernels,
                                       setup['mk_schema'])


                generic_kernel_config(config, kernel,source,
                                      directories, mk=mk, sclk=sclk, lsk=lsk,
                                      fk=fk)

                #
                # We need to load the mk and the future SPK kernel
                # that the CK will be generated from
                #
                kernel.mk = mk

                kernels.append(kernel)

        #
        # Configuration for the default attitude (SPECS)
        #
        if source[1]['kernel_type'] == "spk":
            if source[1]['default_attitude'] != '':

                specs2ck_config = config['specs2ck'][0]
                attitude_types = specs2ck_config['attitude_types'][0]
                default_attitude = source[1]['default_attitude']
                attitude_type = attitude_types[default_attitude]

                source = [source[0], attitude_type[0]]

                kernel = Kernel.CK(source[0], directories,
                                    Kernel.KernelName(
                                            attitude_type[0]['kernel_schema']))

                #
                # We introduce the possibility of having a planning CK
                #
                if source[1]['planning'] == 'False':
                    sclk_schema = 'sclk_step_schema'
                else:
                    sclk_schema = 'sclk_fict_schema'

                fk = os.path.join('fk',
                                  get_latest_kernel('fk', directories.kernels,
                                                    setup[source[1]['spacecraft']][0]['fk_schema']))

                sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                         setup[source[1]['spacecraft']][0][sclk_schema])

                sclkid = re.findall('\d+', sclk)[-1]

                kernel.name.set('sclkid', sclkid)

                kernel.config['sclkid'] = sclkid

                sclk = os.path.join('sclk', sclk)

                kernel.set('config',
                           attitude_types[default_attitude][0])


                mk = get_latest_kernel('mk', directories.kernels, setup['mk_schema'])

                generic_kernel_config(config, kernel,source,
                                      directories,  mk=mk, sclk=sclk, lsk=lsk, fk=fk)

                #
                # We need to load the mk and the future SPK kernel
                # that the CK will be generated from
                #
                kernel.mk = mk

                for k in kernels:
                    if k.source == source[0]:
                        kernel.spk = k

                kernels.append(kernel)

    #
    # We also need to return the list of original soruces in order to
    # clean-up the source directory if there is any error or if we are
    # testing ADCSng
    #
    return kernels, original_sources


def get_config(config):
    """
    Return a dictionary containing the configuration provided for the mission
    in the mission specific *.jSOn configuration file.

    :param mission: Mission name
    :type mission: str
    :return: Mission specific adcsng configuration from json file.
    :rtype: dict
    """
    configuration = {}
    mission = config['mission']

    filename = open(config['root_dir'] + '/config/' + mission.lower() + '.json', 'r')

    configuration = json.load(filename)
    filename.close()

    #
    # We add the ADCSng version
    #
    with open(config['root_dir'] + '/config/version', 'r') as f:
        for line in f:
            version = line
        configuration['generic'][0]['adcsng_version'] = version

    return configuration


def get_description(filename, index):
    """
    This function gets the description value to populate the kernel in creation
    name from the input file. If no index is provided the function returns
    no description.

    :param filename: Source file name
    :type filename: str
    :param index: Position of the description in the source file.
       Format is "x-y" where x is the location of the first letter in the
       string and y the last one
    :type index:
    :return: Description value for the Kernel filename
    :rtype: str
    :raises
    """
    if index == "":
        return ""

    idx = []
    idx.append(int(index.split('-')[0]))
    idx.append(int(index.split('-')[1]))
    if len(idx) != 2 or idx[0] > idx[1]:
        raise ValueError("invalid indexing sequence: {}".format(idx))

    description = filename

    #
    # Get the description field from the input filename.
    #
    if len(description) > idx[1] + 1:
        description = description[idx[0]: idx[1] + 1]
    elif idx[1] + 1 > len(description):
        description = description[idx[0]: len(description)]

    #
    # Remove leading and training underscores.
    #
    description = description.strip('_')

    return description


def get_latest_kernel(kernel_type, paths, pattern, dates=False,
                      include_latest=0,
                      excluded_kernels=False, mkgen=False):
    """
    Returns the name of the latest MK, LSK, FK or SCLK present in the path

    :param kernel_type: Kernel type (lsk, sclk, fk) which also defines the subdirectory name.
    :type kernel_type: str
    :param path: Path to the root of the SPICE directory where the kernels are store in a directory named ``type``.
    :type path: str
    :param patterns: Patterns to search for that defines the kernel ``type`` file naming scheme.
    :type patterns: list
    :return: Name of the latest kernel of ``type`` that matches the naming scheme defined in ``token`` present in the ``path`` directory.
    :rtype: strÐ
    :raises:
       KernelNotFound if no kernel of ``type`` matching the naming scheme
       defined in ``token is present in the ``path`` directory
    """
    kernels = []
    if not isinstance(paths, list):
        paths = [paths]

    paths = list(dict.fromkeys(paths))

    #
    # Get the kernels of type ``type`` from the ``path``/``type`` directory.
    #
    kernels_with_path = []
    for path in paths:
        kernel_path = os.path.join(path, kernel_type)
        kernels_with_path += glob.glob(kernel_path + '/' + pattern)

        #
        # Include kernels in former_versions if the directory exists except for
        # meta-kernel generation
        #
        if os.path.isdir(kernel_path + '/former_versions') and not mkgen:
            kernels_with_path += glob.glob(kernel_path + '/former_versions/' + pattern )


        for kernel in kernels_with_path:
            kernels.append(kernel.split('/')[-1])


    if not kernels:
        raise KernelNotFound(kernel_type, kernel_path)


    #
    # We remove possible duplicates
    #
    kernels = list(dict.fromkeys(kernels))

    #
    # Put the kernels in order
    #
    kernels.sort()

    #
    # We remove the kernel if it is included in the excluded kernels list
    #
    if excluded_kernels:
        for kernel in excluded_kernels:
           if kernel in kernels:
                kernels.remove(kernel)

    if not dates:
        #
        # Return the latest kernel
        #
        if include_latest > 0:
            return kernels[-include_latest:]
        if include_latest > len(kernels):
            return kernels
        try:
            latest_kernel = kernels.pop()
        except:
            latest_kernel = []

        return latest_kernel
    else:
        #
        # Return all the kernels with a given date
        #
        previous_kernel = ''
        kernels_date = []
        for kernel in kernels:
            if previous_kernel \
                    and re.split('_V\d\d', previous_kernel.upper())[0] == re.split('_V\d\d', kernel.upper())[0] \
                    or re.split('_V\d\d\d', previous_kernel.upper())[0] == re.split('_V\d\d\d', kernel.upper())[0]:
                kernels_date.remove(previous_kernel)

            previous_kernel = kernel
            kernels_date.append(kernel)

        return kernels_date


def get_from_kernel(kernel_type, kernel):
    """
    Returns the name of the latest LSK, FK or SCLK present in the kernel (wtih which the kernel has been generated)
    this has

    """
    command_line_process = subprocess.Popen(
            [kernel.directories.executables + '/' +
             'commnt', '-r', kernel.directories.source + '/' + kernel.source],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output = command_line_process.communicate()[0]
    commnt = process_output.decode('utf-8').split('\n')

    for line in commnt:
        if kernel_type.upper() + '_FILE_NAME' in line:
            kernel_type_kernel = line.split("'")[1].split("\\")[-1]
            if '.' + kernel_type2extension(kernel_type, case='lower')[0] not in kernel_type_kernel:
                kernel_type_kernel = line.split("'")[1].split(os.sep)[-1]

    return kernel_type_kernel






def get_version_number(k_type, path, filename):

    #
    # So far ExoMarsRSP is the only mission with 3 zeros in the version
    # number, that is why it is not generalised
    #
    if 'emrsp' in filename:
        version_format = '{:03d}'
    else:
        version_format = '{:02d}'

    if k_type == 'mk':
        version_format = '{:03d}'

    #
    # Create the pattern to look for and find if there's a kernel in the
    # given path that matches the criteria.
    #
    pattern = filename.replace('{version}', '*')

    #
    # DAFCAT CK kernels need to be treated separately otherwise we do not
    # get the appropiate filename
    #
    if '_1.' in pattern:
        pattern = pattern.split('_1.')[0]
    elif '_2.' in pattern:
        pattern = pattern.split('_2.')[0]

    try:
        kernel = get_latest_kernel(k_type, path, pattern)
        version = version_format.format(int(kernel.split('.')[0][-2:]) + 1)
    except:
        version = version_format.format(1)

    return version

