#
# Import the required libraries and modules
#
import datetime
import os
import platform
from adcsng.utils.files import format_str


class Comments(object):
      """
      The class Comments is a class to be used as an input on a Kernel Object.
      Like the Kernel Object this class exists because the comments of the
      kernel are being completed as the kernel is generated.
      """

      __type2descr__ = {
        'tls': 'a Leapseconds Kernel file is used in\n'
               '{SPC}conversions between ephemeris time and UTC\n'
               '{SPC}(coordinated universal time).\n',
        'tsc': 'a Spacecraft Clock Kernel file is used\n'
               '{SPC}in conversions between the {mission_accr} Clock\n'
               '{SPC}time and ephemeris time.\n',
        'tf': 'a Frame Kernel file required to\n'
              '{SPC}establish relationships between frames\n'
              '{SPC}used in geometry computations.\n',
        'tm': 'a Meta-Kernel file required to\n'
              '{SPC}load a list of required kernels\n'
              '{SPC}to generate the given attitude profile.\n'
      }

      def __init__(self, config='', path='', template=''):
          """
          Initialization method of the Comments Class

          :param config: The generic part of the adcsng JSON Configuration
             file
          :type config: dict
          :param path: The path of the directory where the Comments Templates
             are placed.
          :type path: str
          :param template: The given template to be used for the kernel type.
            Usually it also depends on the NAIF utility that generates the
            kernel.
          :type template: str
          """
          self.path = path
          self.template_source = template

          #
          # This adds the fields which are not present in the adcsng JSON
          # configuration file.
          #
          self.tokens = {
              'gen_month': datetime.datetime.now().strftime("%B"),
              'gen_day': str(datetime.datetime.now().day),
              'gen_year': str(datetime.datetime.now().year),
              'SPC': ' ' * 35   # 5 spaces + a maximum filename of 30 chars.
          }

          self.load_tokens(config)


      def __setattr__(self, key, value):
          if key in ['path', 'template_source', 'tokens']:
              super(Comments, self).__setattr__(key, value)
          else:
              self.tokens.update({key: value})

      def __str__(self):
          comments = ''
          with open(os.path.join(self.path, self.template_source)) as f:
              for line in f.readlines():
                  if not line.startswith('%'):
                      comments += format_str(line, self.tokens)
          return comments

      def load_kernels(self, kernel_list):
          """
          Creates the {KERNELS} keyword value from the ``kernel_list`` based
          on the default descriptions.

          :param kernel_list: list of str
          :return: NoneType
          """
          if type(kernel_list) != list:
              kernel_list = [kernel_list]

          kernels = ''
          for kernel_file in kernel_list:
              name = os.path.basename(kernel_file)
              kernels += (' ' * 5) + name.ljust(30)
              kernels += self.__type2descr__[name.split('.')[-1].lower()] + '\n'

          #
          # Apply some formatting to the output string and add the value to the
          # list of available tokens.
          #
          kernels = format_str(kernels, self.tokens)
          self.__setattr__('support_kernels', kernels[:-1])


      def load_tokens(self, dictionary):
          for (key, value) in dictionary.items():
              self.tokens.update({key: value})


      def text2value(self, key, text, tab=0):
          value = ''.join(' ' * tab + line + '\n' for line in text)
          self.tokens.update({key: value[:-1]})


      def load_source(self, source):

          src_dict  = {}
          sources = ''

          first_bool = True
          if isinstance(source, list):
              for file in source:

                  if first_bool:
                     sources = file
                     first_bool = False
                  else:
                     sources += '\n' + ' '*6 + file

          if sources:
              source_filename = sources
          else:
              source_filename = source

          if not isinstance(source, list):
              source = [source]

          # We only use the date birth of the last file.

          src_dict['input_file'] = source_filename


          if platform.system() == 'Windows':
              src_date = os.path.getctime(source[-1])
          else:
              stat = os.stat(source[-1])
              try:
                  src_date = stat.st_birthtime
              except AttributeError:
                  # We're probably on Linux. No easy way to get creation dates here,
                  # so we'll settle for when its content was last modified.
                  src_date = stat.st_mtime

          src_date = datetime.date.fromtimestamp(src_date)

          src_dict['src_year'] = str(src_date.year)
          src_dict['src_month'] = src_date.strftime("%B")
          src_dict['src_day'] =  str(src_date.day)

          self.load_tokens(src_dict)