#!/bin/tcsh
#
# This script generates SPICE collection PDS4 labels and inventories.
#
# Pre-requisites/requirements:
#
# -- complete kernel list, PDS4 values file, kernel collection XML template 
#    must exist
#
# -- PDS3->PDS4 tool, md5sum must be executables
#
# -- PDS4 labels (*.xml) must exists for all files in the list.
#
# -- PDS4 values file must include required keywords (see code for list)
#
# Version 1.0.0 -- May 19, 2015
#
#   Initial version.
#
# Version 1.1.0 -- November 9, 2015
#
#   BUG FIX: in the "if" that checks for presence of a LIDVID in existing
#   inventories changed "grep -c" to "grep ... | wc -l" to deal with 
#   different grep output formats for single and multiple files.
#
# Version 1.2.0 -- November 15, 2017
#
#   Updated Unix date format string (%X->%T).
#
# Script name and version:
#
set SCRIPTNAME = "make_pds4_prod_coll_label.csh"
set SCRIPTVER  = "1.2.0 -- November 15, 2017"

#
# Get command line stuff.
#
if ( $#argv != 3 ) then
    echo " "
    echo "$SCRIPTNAME, version $SCRIPTVER "
    echo " "
    echo "Usage: "
    echo
    echo "   % $SCRIPTNAME <listfile> <valuefile> <templatefile>"
    echo " "
    echo "where:"
    echo " "
    echo "   listfile     -- complete kernel list for all releases"
    echo "   valuefile    -- fixed PDS4 values KWD=VAL file"
    echo "   templatefile -- SPICE kernel collection label XML template file"
    echo
    exit 1
endif

set LISTFILE   = $1
set PDS4VALUES = $2
set XMLTEMPL   = $3

#
# Tools: PDS3-to-PDS4 converter, checksum utility.
#
set PDS4GEN    = "/home/bsemenov/maven/pds/make_pds4_xml_from_lbl.csh"
set MD5SUM     = "/usr/bin/md5sum"

#
# Collection directory and inventory and label name prefixes.
#
set COLLDIR    = "spice_kernels"
set XMLPREF    = "collection_spice_kernels_v"
set XMLEXT     = "xml"
set INVPREF    = "collection_spice_kernels_inventory_v"
set INVEXT     = "csv"

#
# Check input files and programs.
#
foreach FF ( $LISTFILE $PDS4VALUES $XMLTEMPL )
    if ( ! -e $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF does not exit."
	echo
	exit 1
    endif
end

foreach FF ( $PDS4GEN $MD5SUM )
    if ( ! -x $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF is not an excutable."
	echo
	exit 1
    endif
end

foreach FF ( $COLLDIR )
    if ( ! -d $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF is not a directory."
	echo
	exit 1
    endif
end


#
# Check that PDS4 values files has all required keywords. Extract
# BUNDLE_LID.
#
foreach KWD ( PDS4_MISSION_NAME \
              PDS4_MISSION_LID \
              PDS4_SPACECRAFT_NAME  \
              PDS4_SPACECRAFT_LID  \
              PDS4_TARGET_NAME  \
              PDS4_TARGET_TYPE  \
              PDS4_TARGET_LID  \
              BUNDLE_LID  \
              BUNDLE_DESCRIPTION_LID )

    set COUNT = `grep -c "^ *$KWD *=" $PDS4VALUES` 
    if ( $COUNT == 0 ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $PDS4VALUES does not include $KWD."
	echo
	exit 1
    endif
    if ( $COUNT > 1 ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $PDS4VALUES includes 2 or more $KWD."
	echo
	exit 1
    endif
end

set BUNDLID  = `grep BUNDLE_LID $PDS4VALUES         | sed 's/=/ /' | awk '{print $NF}'`

#
# Figure out the next collection version. Check that matching
# version inventory does not exist.
#
set COLLVER = "001"
while ( -e $COLLDIR/$XMLPREF$COLLVER.$XMLEXT ) 
    set COLLVER = `echo $COLLVER | sed 's/^0*//' | awk '{printf "%03d", $1+1}'`
end

set COLLXML = "$COLLDIR/$XMLPREF$COLLVER.$XMLEXT"
set COLLINV = "$COLLDIR/$INVPREF$COLLVER.$INVEXT"

if ( -e $COLLINV ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: exiting version mismatch. $COLLXML doesn't exist but $COLLINV does."
    echo
    exit 1
endif

#
# Loops through all files in the complete list.
#
set TMPINV = "$COLLINV.tmp"

\rm $TMPINV >& /dev/null

set TMPLIST = "" 

foreach FF ( `grep '^ *FILE *=' $LISTFILE  | sed 's/=/ /' | awk '{print $NF}'` ) 

    #
    # Check if the file and its PDS4 label exist.
    #
    set XML = "$FF:r.xml"

    if ( ! -e $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF listed in $LISTFILE does not exist."
	echo
	exit 1
    endif

    if ( ! -e $XML ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $XML, PDS4 label for $FF listed in $LISTFILE, does not exist."
	echo
	exit 1
    endif

    #
    # Check if the file's path is exactly three levels (2 dirs, 1 file name)
    #
    if ( `echo $FF | sed 's/\// /g' | wc -w` != 3 ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF name is not in <dir>/<dir>/<name> form."
	echo
	exit 1
    endif

    #
    # Check that the file's top directory name is the same as the 
    # collection directory name.
    #
    if ( `echo $FF | sed 's/\// /g' | awk '{print $1}'` != "$COLLDIR" ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF's top directory is not $COLLDIR."
	echo
	exit 1
    endif

    #
    # Make file's LID and LIDVID and check that LID present in the PDS4 label.
    # Do it differently for MKs.
    #
    if ( `echo $FF | grep -ic '^.*/mk/.*\.tm$'` == 1 ) then
        set LID = "${BUNDLID}":`echo $FF | sed 's/\//:/' | sed 's/\//_/' | sed 's/_v[0-9][0-9]*.tm//i'`
        set VID = `echo $FF | sed 's/^.*_v\([0-9][0-9]*\)\.tm/\1/i' | awk '{printf "%d", $1}'`".0"
    else 
	set LID = "${BUNDLID}:"`echo $FF | sed 's/\//:/' | sed 's/\//_/'`
	set VID = "1.0"
    endif
    set LIDVID = "${LID}::${VID}"

    if ( `grep -c $LID $XML` != 1 ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: LID $LID doesn't appear in $XML or it appears more than once."
	echo
	exit 1
    endif

    #
    # Check if file's LIDVID appears in any exixting collection inventories
    # and set its member status to "S" or "P" depending on that.
    #
    set MEMBER = "P"

    \ls $COLLDIR/$INVPREF*.$INVEXT >& /dev/null

    if ( $status == 0 ) then

	#
	# Grep existing inventories for LIDVID. If found, reset
	# member status to "S".
	#
	if ( `grep $LIDVID $COLLDIR/$INVPREF*.$INVEXT | wc -l` != 0 ) then
	    set MEMBER = "S"
 	endif

    endif

    echo "$MEMBER,$LIDVID" | sed 's/$/\r/' >> $TMPINV

end

#
# Move temporary inventory to actual inventory.
#
if ( -e $TMPINV ) then
    
    mv $TMPINV $COLLINV

    if ( $status ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: failed to move $TMPINV to $COLLINV."
	echo
	exit 1
    endif

    echo 
    echo "Created $COLLINV"

else
    echo 
    echo "ERROR: ${SCRIPTNAME}: temporary inventory $TMPINV does not exist."
    echo
    exit 1
endif

#
# Make collection label.
#

#
# Determine START_TIME and STOP_TIME time. Grep MK PDS labels
# and pick the earliest START and the latest STOP.
#
\ls $COLLDIR/mk/*.xml >& /dev/null

if ( $status ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot find any MK labels $COLLDIR/mk/*.xml."
    echo
    exit 1
endif

set START = `grep 'start_date_time.*[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]' $COLLDIR/mk/*.xml | sed 's/^.*\([0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]\).*$/\1/i' | sort | head -1`
set STOP  = `grep 'stop_date_time.*[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]'  $COLLDIR/mk/*.xml | sed 's/^.*\([0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]\).*$/\1/i' | sort | tail -1`

if ( "$START" == "" ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot determine start time from MK labels $COLLDIR/mk/*.xml."
    echo
    exit 1
endif

if ( "$STOP" == "" ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot determine stop time from MK labels $COLLDIR/mk/*.xml."
    echo
    exit 1
endif

#
# Set PRODUCT_CREATION_TIME and year to CPU clock.
#
set CURRTIME = `date +"%FT%T"`
set CURRYEAR = `date +"%Y"`

#
# Get file size from the file system.
#
set FILESIZE = `ls -l $COLLINV | awk '{print $5}'`

#
# Get MD5 checksum.
#
set CHECKSUM = `$MD5SUM $COLLINV | awk '{print $1}'`

#
# Get the number of records.
# 
set NREC = `wc -l $COLLINV | awk '{print $1}'`

#
# Set collection LID: <bundle_lid>:<colldir>
#
set COLLLID = "${BUNDLID}":"$COLLDIR"

#
# Set collection VID: version.0
#
set COLLVID = `echo $COLLVER | sed 's/^0*//' | awk '{printf "%d.0", $1}'`

#
# Make complete temporary KWD=VAL PDS3-like label
#
set TMPLBL = "$COLLXML:r.tmp.lbl"

\rm $TMPLBL >& /dev/null

echo                                        >>  $TMPLBL
echo "COLLECTION_LID        = "$COLLLID     >>  $TMPLBL
echo "COLLECTION_VID        = "$COLLVID     >>  $TMPLBL
echo "FILE_NAME             = "$COLLINV:t   >>  $TMPLBL
echo "CURRENT_YEAR          = "$CURRYEAR    >>  $TMPLBL
echo "FILE_SIZE             = "$FILESIZE    >>  $TMPLBL
echo "FILE_CHECKSUM         = "$CHECKSUM    >>  $TMPLBL
echo "N_RECORDS             = "$NREC        >>  $TMPLBL
echo                                        >>  $TMPLBL
cat $PDS4VALUES                             >>  $TMPLBL
echo                                        >>  $TMPLBL
echo "START_TIME            = "$START       >>  $TMPLBL
echo "STOP_TIME             = "$STOP        >>  $TMPLBL
echo "PRODUCT_CREATION_TIME = "$CURRTIME    >>  $TMPLBL
echo                                        >>  $TMPLBL

set TMPLIST = "$TMPLIST $TMPLBL"

#
# Make temporary PDS4 label.
#
set TMPXML    = "$TMPLBL.xml"
set TMPXMLLOG = "$TMPXML.log"

\rm $TMPXML $TMPXMLLOG >& /dev/null

$PDS4GEN $TMPLBL $XMLTEMPL >& $TMPXMLLOG

if ( $status ) then

    #
    # Show error output.
    #
    cat $TMPXMLLOG
    echo
    echo "Values   file was $TMPLBL."
    echo "Template file was $XMLTEMPL."
    echo 

else 

    #
    # Success. Rename temporary XML label.
    #
    mv $TMPXML $COLLXML
    if ( $status ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: failed to move $TMPXML to $COLLXML."
	echo
	exit 1
    endif

    set TMPLIST = "$TMPLIST $TMPXMLLOG"

    echo 
    echo "Created $COLLXML"

endif

#
# Cleanup.
#
\rm $TMPLIST >& /dev/null

#
# All done.
#
echo

exit
