#!/usr/bin/perl
#
# This script replaces MISSION_PHASE_NAME in a ROS label based on
# label's START_TIME and STOP_TIME. Input and output label files
# must be provided on the command line.
#
# BVS/NAIF, 11/24/25.
#
if ( @ARGV != 2 ){
  die "\nUSAGE: % fix_mission_phase.pl <inputlabel> <outputlabel>\n\n";
} else {
  $input  = $ARGV[0];
  $output = $ARGV[1];
}
if ( ! -e $input ) {
  die "\nERROR: $input does not exist.\n\n";
}
if ( -e $output ) {
  die "\nERROR: $output already exists.\n\n";
}

#
# Set distant dates to be used for non-date START_ and STOP_TIMEs.
#
$early = "1900-01-01";
$late  = "2100-01-01";

#
# Phase name list from RO_EST_TN_3372_9_0_*_20151020.pdf
#
#    "GROUND"                  "2000-01-01" "2004-03-02"
#    "LAUNCH"                  "2004-03-03" "2004-03-04"
#    "COMMISSIONING 1"         "2004-03-05" "2004-06-06"
#    "CRUISE 1"                "2004-06-07" "2004-09-05"
#    "COMMISSIONING 2"         "2004-09-06" "2004-10-16"
#    "EARTH SWING-BY 1"        "2004-10-17" "2005-04-04"
#    "CRUISE 2"                "2005-04-05" "2006-07-28"
#    "MARS SWING-BY "          "2006-07-29" "2007-05-28"
#    "CRUISE 3"                "2007-05-29" "2007-09-12"
#    "EARTH SWING-BY 2"        "2007-09-13" "2008-01-27"
#    "CRUISE 4-1"              "2008-01-28" "2008-08-03"
#    "STEINS FLY-BY"           "2008-08-04" "2008-10-05"
#    "CRUISE 4-2"              "2008-10-06" "2009-09-13"
#    "EARTH SWING-BY 3"        "2009-09-14" "2009-12-13"
#    "CRUISE 5"                "2009-12-14" "2010-05-16"
#    "LUTETIA FLY-BY"          "2010-05-17" "2010-09-03"
#    "RENDEZVOUS MANOEUVRE 1"  "2010-09-04" "2011-06-07"
#    "CRUISE 6"                "2011-06-08" "2014-01-20"
#    "PRELANDING"              "2014-01-21" "2014-11-18"
#    "COMET ESCORT 1"          "2014-11-19" "2015-03-10"
#    "COMET ESCORT 2"          "2015-03-11" "2015-06-30"
#    "COMET ESCORT 3"          "2015-07-01" "2015-10-20"
#    "COMET ESCORT 4"          "2015-10-21" "2016-01-12"
#    "ROSETTA EXTENSION 1"     "2016-01-13" "2016-04-05"
#    "ROSETTA EXTENSION 2"     "2016-04-06" "2016-06-28"
#    "ROSETTA EXTENSION 3"     "2016-06-29" "2016-09-30"
#
# Set up three parallel arrays with this data.
#
@pname = (
            "GROUND",
            "LAUNCH",
            "COMMISSIONING 1",
            "CRUISE 1",
            "COMMISSIONING 2",
            "EARTH SWING-BY 1",
            "CRUISE 2",
            "MARS SWING-BY",
            "CRUISE 3",
            "EARTH SWING-BY 2",
            "CRUISE 4-1",
            "STEINS FLY-BY",
            "CRUISE 4-2",
            "EARTH SWING-BY 3",
            "CRUISE 5",
            "LUTETIA FLY-BY",
            "RENDEZVOUS MANOEUVRE 1",
            "CRUISE 6",
            "PRELANDING",
            "COMET ESCORT 1",
            "COMET ESCORT 2",
            "COMET ESCORT 3",
            "COMET ESCORT 4",
            "ROSETTA EXTENSION 1",
            "ROSETTA EXTENSION 2",
            "ROSETTA EXTENSION 3"
);

@pstart = (
            "2000-01-01",
            "2004-03-03",
            "2004-03-05",
            "2004-06-07",
            "2004-09-06",
            "2004-10-17",
            "2005-04-05",
            "2006-07-29",
            "2007-05-29",
            "2007-09-13",
            "2008-01-28",
            "2008-08-04",
            "2008-10-06",
            "2009-09-14",
            "2009-12-14",
            "2010-05-17",
            "2010-09-04",
            "2011-06-08",
            "2014-01-21",
            "2014-11-19",
            "2015-03-11",
            "2015-07-01",
            "2015-10-21",
            "2016-01-13",
            "2016-04-06",
            "2016-06-29"
);

@pstop = (
            "2004-03-02",
            "2004-03-04",
            "2004-06-06",
            "2004-09-05",
            "2004-10-16",
            "2005-04-04",
            "2006-07-28",
            "2007-05-28",
            "2007-09-12",
            "2008-01-27",
            "2008-08-03",
            "2008-10-05",
            "2009-09-13",
            "2009-12-13",
            "2010-05-16",
            "2010-09-03",
            "2011-06-07",
            "2014-01-20",
            "2014-11-18",
            "2015-03-10",
            "2015-06-30",
            "2015-10-20",
            "2016-01-12",
            "2016-04-05",
            "2016-06-28",
            "2016-09-30"
);

#
# Pull out START_TIME and STOP_TIME
#
$start = `grep START_TIME $input | sed 's/^.*= *//' | sed 's/T.*\$//'`;
$stop  = `grep STOP_TIME  $input | sed 's/^.*= *//' | sed 's/T.*\$//'`;
chop( $start );
chop( $stop );

if ( "$start" eq "" ) {
  die "\nERROR: cannot extract START_TIME from input.\n\n";
}
if ( "$stop" eq "" ) {
  die "\nERROR: cannot extract STOP_TIME from input.\n\n";
}
if ( $start !~ /^\d{4}-\d{2}-\d{2}$/ ) {
  $start = $early;
}
if ( $stop  !~ /^\d{4}-\d{2}-\d{2}$/ ) {
  $stop  = $late;
}

#
# Loop through mission phase array to collect mission phases within
# this range.
#
@phases = ();

for ($i = 0; $i < @pname; $i++)  {

  if ( ( "$pstart[$i]" ge "$start" && "$pstart[$i]" le "$stop" ) ||
       ( "$pstop[$i]"  ge "$start" && "$pstop[$i]"  le "$stop" ) ||
       ( "$pstart[$i]" lt "$start" && "$pstop[$i]"  gt "$stop" )    ) {

    @phases = ( @phases, $pname[$i] );

  }

}

#
# Open input label and output labels. Parse input labels line by line.
#
open ( INPUT,  "< $input" )
  || die "\nERROR: cannot open '$input'.\n\n";

open ( OUTPUT, "> $output" )
  || die "\nERROR: cannot open '$output'.\n\n";

while ( $line = <INPUT> ) {
  #
  # Look for MISSION_PHASE_NAME line. Print all other lines to output
  # label.
  #
  if ( $line =~ /^ *MISSION_PHASE_NAME *=/ ) {
    #
    # If MISSION_PHASE_NAME line has "{", read and drop lines
    # until find the one woth closing bracket.
    #
    if ( $line =~ /\{/ ) {
      while ( $line !~ /\}/  ) {
        $line = <INPUT>;
      }
    }
    #
    # Print new MISSION_PHASE_NAME list.
    #
    print OUTPUT "MISSION_PHASE_NAME           = {\n";
    for ($i = 0; $i < @phases-1; $i++)  {
      print OUTPUT "                                 \"$phases[$i]\",\n";
    }
    print OUTPUT "                                 \"$phases[$i]\"\n";
    print OUTPUT "                               }\n";
  } else {
    print OUTPUT $line;
  }
}

#
# Close files.
#
close ( INPUT )
  || die "\nERROR: cannot close '$input'.\n\n";

close ( OUTPUT )
  || die "\nERROR: cannot close '$output'.\n\n";


#
# All done.
#
exit;
