C$ Procedure     TCPCHK -- Check and Output TCP Discontinuities

      SUBROUTINE TCPCHK ( MSG, ON, NREC, DDHTIM, DDHS, DDHMS, 
     .                    TCPG, TCPO, TCPTS, UTC, ET, MAXDSC, OK ) 

C$ Abstract
C
C     This routine computes discontinuities between TCP records, checks
C     them against specified threshold, and, if requested, outputs them
C     along with all TCP data to STDOUT as a nice formatted table.
C
C     This routine calls SPEAK routine to do output. If SPEAK[er] is 
C     OFF, it will print nothing.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     None.
C
C$ Declarations
 
      IMPLICIT  NONE

      INCLUDE               'tcp2scet.inc'

      CHARACTER*(*)         MSG
      LOGICAL               ON
      INTEGER               NREC
      DOUBLE PRECISION      DDHTIM ( * )
      INTEGER               DDHS   ( * )
      INTEGER               DDHMS  ( * )
      DOUBLE PRECISION      TCPO   ( * )
      DOUBLE PRECISION      TCPG   ( * )
      INTEGER               TCPTS  ( * )
      CHARACTER*(*)         UTC    ( * )
      DOUBLE PRECISION      ET     ( * )
      DOUBLE PRECISION      MAXDSC
      LOGICAL               OK

C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     MSG        I   Message to be printed before table
C     ON         I   Flag indicating to output or not (.TRUE. to print)
C     NREC       I   Number of TCP records in the input buffers
C     DDHTIM     I   DDH time, as DP seconds
C     DDHS       I   DDH time, integer seconds 
C     DDHMS      I   DDH time, fractional seconds
C     TCPO       I   TCP offset
C     TCPG       I   TCP gradient
C     TCPTS      I   TCP generation time
C     UTC        I   UTC corresponding to DDH time
C     ET         I   ET corresponding to DDH time
C     MAXDSC     I   TCP discontinuity threshold
C     OK         O   Flag indicating that all discontinuity are with 
C                    threshold (if yes, set to .TRUE., otherwise
C                    .FALSE.)
C
C$ Detailed_Input
C
C     N/A.
C
C$ Detailed_Output
C
C     N/A
C
C$ Parameters
C
C     See 'tcp2scet.inc'.
C
C$ Exceptions
C
C     Error free.
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     N/A.
C
C$ Examples
C
C     N/A.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     B.V. Semenov     (JPL)
C
C$ Version
C
C-    TCP2SCET Version 1.2.0, 04-JAN-2006 (BVS)
C
C        Changed the discontinuity check algorithm to properly handle
C        records spanning over leapseconds.
C
C-    TCP2SCET Version 1.0.0, 17-SEP-2003 (BVS)
C
C-&
 
C
C     Local variables
C
      CHARACTER*(LINSIZ)    LINE
      CHARACTER*(WRDSIZ)    HWORD
      CHARACTER*(WRDSIZ)    TAG

      DOUBLE PRECISION      OBTCUR
      DOUBLE PRECISION      OBTDEL
      DOUBLE PRECISION      OBTPRV

      DOUBLE PRECISION      S1970
      DOUBLE PRECISION      UTCSCP
      DOUBLE PRECISION      TDTSCP
      DOUBLE PRECISION      DELTAP
      DOUBLE PRECISION      UTCSCC
      DOUBLE PRECISION      TDTSCC
      DOUBLE PRECISION      DELTAC
      DOUBLE PRECISION      LEAPSC

      INTEGER               RECIDX
      INTEGER               RTRIM

C
C     SPICELIB functions.
C      
      DOUBLE PRECISION      UNITIM

C
C     If print is ON, print the header.
C
      IF ( ON ) THEN
         CALL SPEAK( ' ' )
         CALL SPEAK( MSG )
         CALL SPEAK( ' ' )
         CALL SPEAK( 'UTC                   DHH seconds    ' //
     .               'Offset         Gradient       OBT(i)       ' //
     .               'OBT(i-1)     Delta     Good?' )
         CALL SPEAK( '--------------------- -------------- ' //
     .               '-------------- -------------- ------------ ' //
     .               '------------ --------- -------' )
      END IF

C
C     Compute and buffer the number of "formal" seconds between
C     1970-01-01 00:00:00 ("base" for the VAX time used in DDS header)
C     and J2000. (Ignore the ERROR as we know that input time string
C     has correct format :).
C
      CALL TPARSE( '1970-01-01 00:00:00', S1970, HWORD )
     
C
C     Set output logical OK to .TRUE.
C
      OK = .TRUE.

C
C     Loop over TCP records.
C
      DO RECIDX = 1, NREC

C
C        Compute discontinuities for all records but the first.
C
         IF ( RECIDX .NE. 1 ) THEN

C
C           Determine whether there was a leapsecond between 
C           current record and the previous one.
C
            UTCSCP = DDHTIM(RECIDX-1) + S1970
            TDTSCP = UNITIM ( ET(RECIDX-1), 'ET', 'TDT' )
            DELTAP = TDTSCP - UTCSCP

            UTCSCC = DDHTIM(RECIDX) + S1970
            TDTSCC = UNITIM ( ET(RECIDX), 'ET', 'TDT' )
            DELTAC = TDTSCC - UTCSCC

            LEAPSC = DELTAC - DELTAP

C
C           Compute discontinuities using ESOC's simple time
C           correlation formula adjusted to take leapseconds into
C           account.
C
            OBTCUR = ( DDHTIM(RECIDX) - TCPO(RECIDX) )
     .               / TCPG(RECIDX)
            OBTPRV = ( DDHTIM(RECIDX) - TCPO(RECIDX-1) + LEAPSC )
     .               / TCPG(RECIDX-1)
            OBTDEL = OBTCUR - OBTPRV

         ELSE

C
C           This is the first record. Nothing to check for this one.
C
            OBTCUR = ( DDHTIM(RECIDX) - TCPO(RECIDX) )/ TCPG(RECIDX)
            OBTPRV = OBTCUR
            OBTDEL = 0.D0

         END IF

C
C        Check how this discontinuity compares to the thresholds. If
C        it's too big, reset OK to .FALSE.
C
         IF ( DABS(OBTDEL) .GT. MAXDSC ) THEN
            TAG = 'TOO BIG'
            OK = .FALSE.
         ELSE
            TAG = 'ok' 
         END IF

C
C        If print is ON, assemble a nice output line and print it.
C
         IF ( ON ) THEN

            LINE = UTC(RECIDX)(1:22)

            CALL DPFMT( DDHTIM(RECIDX), 'xxxxxxxxxxx.xxx', HWORD )
            LINE = LINE(:RTRIM(LINE)) // HWORD

            CALL DPFMT( TCPO(RECIDX),   'xxxxxxxxxxx.xxx', HWORD )
            LINE = LINE(:RTRIM(LINE)) // HWORD

            CALL DPFMT( TCPG(RECIDX),   'xx.xxxxxxxxxxxx', HWORD )
            LINE = LINE(:RTRIM(LINE)) // HWORD

            CALL DPFMT( OBTCUR,         'xxxxxxxxx.xxx',   HWORD )
            LINE = LINE(:RTRIM(LINE)) // HWORD

            CALL DPFMT( OBTPRV,         'xxxxxxxxx.xxx',   HWORD )
            LINE = LINE(:RTRIM(LINE)) // HWORD

            CALL DPFMT( OBTDEL,         'xxxxxx.xxx',      HWORD )
            LINE = LINE(:RTRIM(LINE)) // HWORD

            LINE = LINE(:RTRIM(LINE)) // ' ' // TAG(:RTRIM(TAG))

            CALL SPEAK( LINE )

         END IF

      END DO

      IF ( ON ) THEN
         CALL SPEAK( ' ' )
      END IF

      RETURN
      END
