C$Procedure      SETFRM ( Set up a dynamic frame )
 
      SUBROUTINE SETFRM ( FRAME, PLTFRM, REF )
 
C$ Abstract
C
C     Look up the definition of a dynamic frame in the kernel
C     pool and set up the frame in the routine dynmfr.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C      None.
C
C$ Keywords
C
C      FRAME
C
C$ Declarations
 
      IMPLICIT NONE
      CHARACTER*(*)         FRAME
      INTEGER               PLTFRM
      CHARACTER*(*)         REF
 
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     FRAME      I   Name of a dynamic frame
C     PLTFRM     I   Id-code for  structure whose attitude is modelled
C     REF        O   Name of frame the dynamic frame will be relative to
C
C$ Detailed_Input
C
C     FRAME      is the name of some dynamic frame stored in the
C                kernel pool in the variable DYNAMIC_FRAMES.
C
C     PLTFRM     is the id-code for the structure whose attitude is
C                to be modelled by the frame specification given
C                by FRAME.
C
C$ Detailed_Output
C
C     REF        is the name of the reference frame that FRAME is
C                relative to.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     None.
C
C$ Particulars
C
C     This routine looks up frame definitions from the kernel pool.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 3-APR-2000 (WLT)
C
C
C-&
C
C     SPICELIB Functions
C
      INTEGER               ISRCHC
      LOGICAL               FAILED
      LOGICAL               RETURN
      LOGICAL               M2XIST
      LOGICAL               HAVE
 
      INTEGER               MAXFRM
      PARAMETER           ( MAXFRM = 100 )
 
      INTEGER               WDSIZE
      PARAMETER           ( WDSIZE = 32 )
 
      INTEGER               SMWDSZ
      PARAMETER           ( SMWDSZ = 4 )
 
      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 255 )
 
      CHARACTER*(WDSIZE)    NAMES ( MAXFRM )
      CHARACTER*(WDSIZE)    RELTO ( MAXFRM )
      CHARACTER*(WDSIZE)    WATCH ( 4 )
      CHARACTER*(WDSIZE)    FRNAME
      CHARACTER*(WDSIZE)    MYFRAM
      CHARACTER*(WDSIZE)    DTYPE
      CHARACTER*(WDSIZE)    SYSTEM
 
      CHARACTER*(LNSIZE)    WHEN
      CHARACTER*(LNSIZE)    ERROR ( 2 )
      CHARACTER*(LNSIZE)    SYNTAX( 2 )
      CHARACTER*(LNSIZE)    WORKSP
 
 
      CHARACTER*(SMWDSZ)    AXIS1 ( MAXFRM )
      CHARACTER*(SMWDSZ)    AXIS2 ( MAXFRM )
 
      CHARACTER*(LNSIZE)    PRIMRY( MAXFRM )
      CHARACTER*(LNSIZE)    SCNDRY( MAXFRM )
 
      DOUBLE PRECISION      EPOCH

      INTEGER               BTEMP
      INTEGER               COUNT
      INTEGER               DYNMID
      INTEGER               I
      INTEGER               N
      INTEGER               NF
      INTEGER               NP
      INTEGER               NR
      INTEGER               NS
      INTEGER               SIZE
 
      LOGICAL               FIRST
      LOGICAL               FOUND
 
      LOGICAL               FOUND1
      LOGICAL               FOUND2
      LOGICAL               FOUND3
      LOGICAL               FOUND4
      LOGICAL               FOUND5
 
      LOGICAL               FOUNDN
      LOGICAL               FOUNDP
      LOGICAL               FOUNDR
      LOGICAL               FOUNDS
      LOGICAL               MISSING
      LOGICAL               UPDATE
 
      SAVE                  NAMES
      SAVE                  RELTO
      SAVE                  AXIS1
      SAVE                  AXIS2
      SAVE                  PRIMRY
      SAVE                  SCNDRY
      SAVE                  COUNT
      SAVE                  FIRST
 
      DATA                  FIRST / .TRUE. /
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'SETFRM' )
 
      IF ( FIRST ) THEN
 
         FIRST = .FALSE.
 
         WATCH(1) = 'ORIENTATION_NAME'
         WATCH(2) = 'PRIMARY'
         WATCH(3) = 'SECONDARY'
         WATCH(4) = 'BASE_FRAME'
         CALL SWPOOL ( 'SETFRM', 4, WATCH )

      END IF
      SYNTAX ( 1 ) = 'SOLUTION[solve] (1:1){ OF '
     .//             '@name[framename] | TO @name[framename] '
     .//             '| OF @int[frameid] | TO @int[frameid] } '
     .//             '= @name[orientation] (0:1){ AT[at] '
     .//             '@calendar[when] } '
 
      SYNTAX ( 2 ) = '@name[orientation] (0:1){ AT[at] '
     .//             '@calendar[when] } '


C     Whenever the agents attached to our names get a notificaiton
C     of an update, we need to look up the current set of defined
C     frames and check that the number of names, primary, secondary
C     and relative_to are the same.  These values are then buffered
C     in this routine so that on subsequent calls we don't have to
C     go through the look up process all over again.
C
      CALL CVPOOL ( 'SETFRM', UPDATE )
 
      IF ( UPDATE ) THEN
         CALL GCPOOL('ORIENTATION_NAME',1, MAXFRM, NF, NAMES, FOUNDN)
         CALL GCPOOL('BASE_FRAME',      1, MAXFRM, NR, RELTO, FOUNDR)
 
         COUNT = NF
         NP    = 0
         NS    = 0
 
         FOUNDP = .TRUE.
         FOUNDS = .TRUE.
         FOUND  = .TRUE.
 
         DO WHILE ( FOUND )
 
            NP    = NP + 1
            NS    = NS + 1
 
            CALL STPOOL('PRIMARY',   NP, '-',PRIMRY(NP), SIZE, FOUNDP )
            CALL STPOOL('SECONDARY', NS, '-',SCNDRY(NS), SIZE, FOUNDS )
            FOUND = FOUNDS .AND. FOUNDP .AND. NP .LT. MAXFRM
 
            IF ( .NOT. FOUNDP ) THEN
               NP = NP - 1
            END IF
 
            IF ( .NOT. FOUNDS ) THEN
               NS = NS - 1
            END IF
 
         END DO
 
 
         MISSING = .NOT. (      FOUNDN
     .                    .AND. FOUNDR )
 
         MISSING =      MISSING
     .             .OR. COUNT .NE. NP
     .             .OR. COUNT .NE. NS
     .             .OR. COUNT .NE. NR
 
         IF ( MISSING ) THEN
 
            CALL SWPOOL ( 'SETFRM', 3, WATCH )
            CALL SETMSG ( 'The specification of the dynamic '
     .      //            'frames is incomplete. #' )
 
            IF      ( .NOT. FOUNDN ) THEN
               CALL ERRCH  ( '#', 'The variable ORIENTATION_NAME '
     .         //                 'is not present in the kernel '
     .         //                 'pool. ' )
            ELSE IF ( .NOT. FOUNDP ) THEN
               CALL ERRCH  ( '#', 'The variable PRIMARY '
     .         //                 'is not present in the kernel '
     .         //                 'pool. ' )
            ELSE IF ( .NOT. FOUNDS ) THEN
               CALL ERRCH  ( '#', 'The variable SECONDARY '
     .         //                 'is not present in the kernel '
     .         //                 'pool. ' )
            ELSE IF ( .NOT. FOUNDS ) THEN
               CALL ERRCH  ( '#', 'The variable BASE_FRAME '
     .         //                 'is not present in the kernel '
     .         //                 'pool. ' )
            ELSE IF ( COUNT .NE. NP ) THEN
               CALL ERRCH ( '#', 'The number of frames specified '
     .         //                'does not match the number of '
     .         //                'primary axis definitions '
     .         //                'supplied. ' )
            ELSE IF ( COUNT .NE. NS ) THEN
               CALL ERRCH ( '#', 'The number of frames specified '
     .         //                'does not match the number of '
     .         //                'secondary axis definitions '
     .         //                'supplied. ' )
            ELSE IF ( COUNT .NE. NR ) THEN
               CALL ERRCH ( '#', 'The number of frames specified '
     .         //                'does not match the number of '
     .         //                '"relative" frames.' )
            END IF
 
            CALL SIGERR ( 'SPICE(INCOMPLETESPEC)'  )
            CALL CHKOUT ( 'SETFRM' )
            RETURN
 
 
         ELSE IF ( COUNT .EQ. MAXFRM ) THEN
 
            CALL SETMSG ( 'The maximum number of dynamic frame '
     .      //            'specifications that can be supported '
     .      //            'is #.  More than this number have '
     .      //            'been supplied. ' )
            CALL ERRINT ( '#', MAXFRM-1 )
            CALL SIGERR ( 'SPICE(TOOMANYFRAMES)'  )
            CALL CHKOUT ( 'SETFRM' )
            RETURN
 
         END IF
C
C        Now for each frame check the axis and desription
C
         DO I = 1, COUNT
 
            CALL REPMC ( PRIMRY(I), '=', ' ', PRIMRY(I) )
            CALL REPMC ( SCNDRY(I), '=', ' ', SCNDRY(I) )
            CALL NEXTWD( PRIMRY(I), AXIS1(I), PRIMRY(I) )
            CALL NEXTWD( SCNDRY(I), AXIS2(I), SCNDRY(I) )
 
            CALL DYNFPR( AXIS1(I), PRIMRY(I) )
 
            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'SETFRM' )
               RETURN
            END IF
 
            CALL DYNFSC( AXIS2(I), SCNDRY(I) )
 
            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'SETFRM' )
               RETURN
            END IF
 
         END DO
C
C        If we get to this point, we are ready to go.
C
      END IF
C
C     Next see which syntax template the FRAME description
C     matches.
C
      ERROR(1) = ' '
      ERROR(2) = ' '
 
      CALL META_2 ( FRAME, SYNTAX, 2, WORKSP, BTEMP, ERROR )
 
      IF ( HAVE(ERROR) ) THEN
         CALL SETMSG ( ERROR(1) )
         CALL SIGERR ( 'SPICE(BADSPEC)' )
         CALL CHKOUT ( 'SETFRM' )
         RETURN
      END IF
 
C
C     First fetch the name of the orientation.
C
      CALL M2GETC ( 'orientation', FRAME, FOUND, MYFRAM )
 
C
C     Make sure the orientation is one that is recognized.
C
      I = ISRCHC ( MYFRAM, COUNT, NAMES )
 
      IF ( I .GT. 0 ) THEN
C
C        We've got a recognized orientation name, fetch the
C        primary and secondary axis specifications and give
C        the specs to the dynamic frame routine.
C
         CALL DYNFPR ( AXIS1(I), PRIMRY(I) )
         CALL DYNFSC ( AXIS2(I), SCNDRY(I) )
C
C        Set the "relative" frame in the dynamic frame routine.
C
         CALL DRELTO ( RELTO(I) )
 
         REF = RELTO(I)
 
      ELSE
 
         CALL SETMSG ( 'The specified frame: ''#'' is not one of '
     .   //            'those that have been supplied to the '
     .   //            'program. ' )
         CALL ERRCH  ( '#', MYFRAM )
         CALL SIGERR ( 'SPICE(UNKNOWNDYNAMICFRAME)'  )
 
      END IF
 
 
C
C     If the frame specification is a "solution" form (i.e. SOLUTION
C     OF/TO frame = orientation ) We need to extract the Id-code of the
C     frame portion of the equation.
C
      IF ( M2XIST ( 'solve' ) ) THEN
 
         IF ( M2XIST( 'frameid' ) ) THEN
C
C           If an integer ID was supplied, we can just pick it out of
C           the specification...
C
            CALL M2GETI ( 'frameid', FRAME, FOUND, DYNMID )
 
         ELSE
C
C           Otherwise, we need to get the name and let the frame
C           subsystem determine the ID code for us.
C
            FRNAME = ' '
 
            CALL M2GETC ( 'framename', FRAME, FOUND, FRNAME )
            CALL NAMFRM (  FRNAME, DYNMID )
 
            IF ( DYNMID .EQ. 0 ) THEN
               CALL SETMSG ( 'The frame # is not the name of a '
     .         //            'recognized frame. A frame '
     .         //            'definition kernel needs to be '
     .         //            'loaded that identifies the '
     .         //            'Id-code, class and subclass id for '
     .         //            'this frame. ' )
               CALL ERRCH  ( '#', FRNAME )
 
               CALL SIGERR ( 'SPICE(UNRECOGNIZEDFRAME)' )
               CALL CHKOUT ( 'SETFRM' )
               RETURN
            END IF
         END IF
C
C        We need to set up QAVGEN so that it will solve for the
C        appropriate rotation and angular velocity.
C
         CALL SOLV4 ( DYNMID, PLTFRM )
 
      ELSE
 
         CALL NOSOLV
 
      END IF
 
C
C     See if the orientation is to be frozen in time or time varying.
C
      IF ( M2XIST ( 'at' ) ) THEN
 
         CALL M2GETC ( 'when', FRAME, FOUND, WHEN )
C
C        See if we have a leapseconds kernel loaded.  If so we'll
C        use STR2ET to parse the input string.  Otherwise, we just
C        use TPARSE.  Note that no matter which routine we use, the
C        default time system is TDB.
C
         CALL DTPOOL ( 'DELTET/DELTA_T_A', FOUND1, N, DTYPE )
         CALL DTPOOL ( 'DELTET/K',         FOUND2, N, DTYPE )
         CALL DTPOOL ( 'DELTET/EB',        FOUND3, N, DTYPE )
         CALL DTPOOL ( 'DELTET/M',         FOUND4, N, DTYPE )
         CALL DTPOOL ( 'DELTET/DELTA_AT',  FOUND5, N, DTYPE )
 
         IF (     FOUND1
     .      .AND. FOUND2
     .      .AND. FOUND3
     .      .AND. FOUND4
     .      .AND. FOUND5 ) THEN
C
C           Being good, software citizens, we first find out the
C           state of the time defaults and then restore those
C           defaults once we've finished our task.
C
            CALL TIMDEF ( 'GET', 'SYSTEM', SYSTEM )
            CALL TIMDEF ( 'SET', 'SYSTEM', 'TDB'  )
            CALL STR2ET ( WHEN,  EPOCH )
            CALL TIMDEF ( 'SET', 'SYSTEM', SYSTEM )
 
         ELSE
C
C           A complete leapseconds kernel has not been loaded
C           (probably not even an partial one is available), so
C           we parse the epoch using the old stand by TPARSE.
C
            CALL TPARSE ( WHEN, EPOCH, ERROR )
 
         END IF
 
         IF ( HAVE(ERROR) ) THEN
            CALL SETMSG ( ERROR )
            CALL SIGERR ( 'SPICE(BADTIMESTRING)' )
            CALL CHKOUT ( 'SETFRM' )
            RETURN
         END IF
 
         CALL USEFXD ( EPOCH )
 
      ELSE
 
         CALL USEVAR
 
      END IF
 
      CALL CHKOUT ( 'SETFRM' )
      RETURN
 
 
 
      END
