C$Procedure PRSDR ( Parse data record from OEM file )
 
      SUBROUTINE PRSDR ( INPUT, TIMSYS, ET, STATE, FOUND )
 
C$ Abstract
C
C     Parse a logical data record from an OEM file.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     SPK
C     TIME
C
C$ Keywords
C
C     EPHEMERIS
C     FILES
C
C$ Declarations
 
      IMPLICIT NONE 

      INTEGER               MAXDL
      PARAMETER           ( MAXDL  = 10 )

      CHARACTER*(*)         INPUT
      CHARACTER*(*)         TIMSYS
      DOUBLE PRECISION      ET
      DOUBLE PRECISION      STATE  ( 6 )
      LOGICAL               FOUND

      CHARACTER*(*)         DELIMS
      PARAMETER           ( DELIMS = ', ' )

      INTEGER               MAXLIN
      PARAMETER           ( MAXLIN = 255 )      
 
C$ Brief_I/O
C
C     Variable  I/O  Description
C     --------  ---  --------------------------------------------------
C     MAXDL      P   Max number of physical records for one data line.
C     INPUT      I   Input OEM file name.
C     TIMSYS     I   Name of time system.
C     ET         O   Time tag of state vector.
C     STATE      O   State vector.
C     FOUND      O   Logical flag indicating whether record was found.
C
C$ Detailed_Input
C
C     INPUT          is the name of the OEM file from which a data
C                    record is to be read.  
C
C     TIMSYS         is the name of the time system associated with
C                    time tags in the OEM file.  This must be 
C                    a time system flag recognized by STR2ET.
C                    Examples:  TDB, TDB, UTC.
C
C$ Detailed_Output
C
C     ET             is the time tag parsed from a data record and
C                    converted to seconds past J2000 TDB.
C
C     STATE          is a state vector parsed from a data record.
C                    Units are always km and km/s.  Order of 
C                    components is: x, y, z, dx/dt, dy/dt, dz/dt.
C
C     FOUND          is a flag indicating whether a data record was
C                    found.  The file's record pointer should normally
C                    be positioned at the physical record at which a
C                    time tag begins.  When a new metadata block is
C                    detected, or if the end of file is reached, FOUND
C                    will be set to .FALSE.
C
C$ Parameters
C
C     MAXDL          is the maximum number of physical records that
C                    are expected to contain one logical data record.
C                    According to the OEM specification, a data
C                    record should fit on one line of the OEM text
C                    file.  However, transmission of an OEM file
C                    via e-mail may result in line wrapping.  This
C                    module attempts to parse data records even if
C                    they've been subjected to line wrapping, as long
C                    as line breaks occur only between the time tag
C                    or the numeric tokens.
C     
C$ Exceptions
C
C     1) If a line is seen that is classified as a data line, but no
C        time tag is recognized at the start of the line, the error
C        SPICE(TIMETAGNOTFOUND) is signaled.
C
C
C     2) If a data line is recognized, but does not terminate within
C        MAXDL physical records, the error SPICE(ENDNOTFOUND) is
C        signaled.
C
C     3) If a state vector position or velocity component does not 
C        parse as a double precision number, the error 
C        SPICE(INVALIDDPNUMBER) is signaled.
C
C     4) If fewer than six numbers are found following the time tag,
C        the error SPICE(TOOFEWTOKENS) is signaled.
C
C     5) If more than six numbers are found following the time tag,
C        the error SPICE(TOOMANYTOKENS) is signaled.
C
C$ Files
C
C     See the reference [1].
C
C$ Particulars
C
C     This routine expects the input file record pointer to be 
C     positioned correctly:  the first line read by this routine
C     should start with a time tag belonging to a data record.
C
C     According to the OEM specification, a data record should fit on
C     one line of the OEM text file.  However, transmission of an OEM
C     file via e-mail may result in line wrapping.  This module
C     attempts to parse data records even if they've been subjected to
C     line wrapping, as long as line breaks occur only between the time
C     tag or the numeric tokens.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     1) The input file must have line length no longer than 255
C        characters.
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-1,
C          September, 2004.
C     [2]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-2,
C          November, 2009.
C
C$ Author_and_Institution
C
C     N.J. Bachman   (JPL)
C     M.C. Kim       (JPL)
C
C$ Version
C
C-    OEM2SPK Version 2.0.0, 30-AUG-2013 (MCK)
C
C        Enhanced to read OEM 2; the program reads ephemeris data lines
C        augmented by acceleration components. However, anything after
C        the z-component velocity is not used.
C
C-    OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&
 
C
C     SPICELIB functions
C
      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C     
      INTEGER               MAXTTK
      PARAMETER           ( MAXTTK = 50 )

      INTEGER               MAXWRD
      PARAMETER           ( MAXWRD = MAXTTK + 6 )

      INTEGER               WRDLEN
      PARAMETER           ( WRDLEN = 80 )

      INTEGER               MSGLEN
      PARAMETER           ( MSGLEN = 320 )

      INTEGER               BIG
      PARAMETER           ( BIG    = MAXDL * MAXLIN )

      INTEGER               TIMLEN
      PARAMETER           ( TIMLEN = 50 )

C
C     Local variables
C
      CHARACTER*(TIMLEN)    TAG
      CHARACTER*(TIMLEN)    NXTTAG
      CHARACTER*(MAXLIN)    LINE
      CHARACTER*(MSGLEN)    MSG
      CHARACTER*(WRDLEN)    TYPE
      CHARACTER*(WRDLEN)    WRDBUF ( MAXWRD )
      CHARACTER*(BIG)       BIGLIN


      INTEGER               DSTART
      INTEGER               ELOC
      INTEGER               I
      INTEGER               LNUM
      INTEGER               N
      INTEGER               NXTLOC
      INTEGER               NLINES
      INTEGER               LSTART

      LOGICAL               EOF
      LOGICAL               NXTFND


      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'PRSDR' )

C
C     Nothing found yet.
C
      FOUND = .FALSE.

C
C     Read the next non-comment line.
C
      CALL GETNCL ( INPUT, LINE, LNUM, TYPE, EOF )

      IF ( EOF .OR. FAILED() ) THEN
C
C        No more data lines are to be found.
C
         CALL CHKOUT ( 'PRSDR' ) 
         RETURN

      END IF


      IF ( TYPE .NE. 'DATA_LINE' ) THEN
C
C        We've hit the end of a data block.  Put back the line
C        we just read.
C
         CALL PUSHLN

         CALL CHKOUT ( 'PRSDR' ) 
         RETURN

      END IF

C
C     Record this line number.
C     
      LSTART = LNUM

C
C     The file is supposed to be positioned at the first line of 
C     a logical data record.  Some initial token sequence constitutes
C     a time tag---normally, the tag is supposed to be the first
C     token.  Make sure we have the tag.
C
      CALL FNDTAG ( LINE, TAG, ELOC, FOUND )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'PRSDR' ) 
         RETURN
      END IF


      IF ( .NOT. FOUND ) THEN

         CALL SETMSG ( 'Data record starting at line #* in '  //
     .                 'file <*> did not start with a '       //
     .                 'recognized time tag.  Line was *'      )
         CALL ERRINT ( '*',  LNUM                              )
         CALL ERRCH  ( '*',  INPUT                             ) 
         CALL ERRCH  ( '*',  LINE                              ) 
         CALL SIGERR ( 'SPICE(TIMETAGNOTFOUND)'                )
         CALL CHKOUT ( 'PRSDR'                                 )
         RETURN

      END IF

      CALL TIMCVT ( INPUT, LINE, LNUM, TIMSYS, TAG, ET )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'PRSDR' ) 
         RETURN
      END IF

C
C     Since the original line may have wrapped onto successive lines,
C     keep reading lines until we hit the next time tag, non-data line,
C     end-of-file, or until we've read the max allowed number of lines.
C
      BIGLIN = LINE

C
C     Read the next non-comment line.
C
      NLINES = 1
      NXTFND = .FALSE.

      CALL GETNCL ( INPUT, LINE,   LNUM,   TYPE,  EOF )
      CALL FNDTAG ( LINE,  NXTTAG, NXTLOC, NXTFND     )

      DO WHILE (       ( .NOT. EOF             )
     .           .AND. ( .NOT. NXTFND          )
     .           .AND. ( .NOT. FAILED()        )
     .           .AND. ( TYPE .EQ. 'DATA_LINE' )  )

C
C        This line should be a continuation of the current 
C        record.
C
         CALL SUFFIX ( LINE, 1, BIGLIN )

         NLINES = NLINES + 1

         IF ( NLINES .EQ. MAXDL ) THEN
            
            CALL SETMSG ( 'Maximum (*) number of data lines were ' //
     .                    'read without finding the end of the '   //
     .                    'current logical record.  Input file '   //
     .                    'is <*> Last line number read was #*. '  //
     .                    'Data record started on line #*. '       //
     .                    'Initial line of record was <*>'          )
            CALL ERRINT ( '*',  MAXDL                               )
            CALL ERRCH  ( '*',  INPUT                               ) 
            CALL ERRINT ( '*',  LNUM                                )
            CALL ERRINT ( '*',  LSTART                              )
            CALL ERRCH  ( '*',  LINE                                ) 
            CALL SIGERR ( 'SPICE(ENDNOTFOUND)'                      )
            CALL CHKOUT ( 'PRSDR'                                   )
            RETURN

         END IF

C
C        Get the next line and see whether we're looking at the 
C        start of a new data record.
C         
         CALL GETNCL ( INPUT, LINE,   LNUM,   TYPE,  EOF )
         CALL FNDTAG ( LINE,  NXTTAG, NXTLOC, NXTFND     )

      END DO


      IF ( .NOT. EOF ) THEN
C
C        We've hit the start of the next data record, or else we've 
C        encountered a metadata block start marker.  Put back the line
C        we just read.
C
         CALL PUSHLN

      END IF

C
C     Time to collect the d.p. data.
C
C     Fetch the individual words from the data line.
C
      DSTART = ELOC + 1

      IF ( BIGLIN(DSTART:DSTART) .EQ. ',' ) THEN
         DSTART = DSTART + 1
      END IF

      CALL LPARSM ( BIGLIN(DSTART:), DELIMS, MAXWRD, N, WRDBUF )

C
C     In OEM 2, an ephemeris data line can contain either 6 or 9 d.p.
C     numbers. OEM2SPK ignores accelaration components after z-component
C     velocity, and read only 6 state (position & velocity) components.
C
      IF ( N .EQ. 6 .OR. N .EQ. 9 ) THEN
C
C        Hopefully this easy case is normal, so we handle it
C        separately, for the sake of speed.
C
         DO I = 1, 6
 
            MSG = ' '

            READ ( WRDBUF(I), '(D 24.1)' ) STATE(I)
 
            IF ( MSG .NE. ' ' ) THEN
C 
C              The current state component didn't parse as a d.p.
C              number.
C             
               CALL SETMSG ( 'Could not parse input as a double '  //
     .                       'precision number while attempting '  //
     .                       'to read a state vector.  File is '   //
     .                       '<*> Last line number read was #*. '  //
     .                       'Data record started on line #*. '    //
     .                       'Initial line of record was <*> '     //
     .                       'Unparsed token was *'                 )
               CALL ERRCH  ( '*',  INPUT                            ) 
               CALL ERRINT ( '*',  LNUM                             )
               CALL ERRINT ( '*',  LSTART                           )
               CALL ERRCH  ( '*',  LINE                             ) 
               CALL ERRCH  ( '*',  WRDBUF(I)                        ) 
               CALL SIGERR ( 'SPICE(INVALIDDPNUMBER)'               )
               CALL CHKOUT ( 'PRSDR'                                )
               RETURN
 
            END IF
 
         END DO



      ELSE IF ( N .LT. 6 ) THEN
C 
C        Something important is missing.
C 
         CALL SETMSG ( 'Could not find six numeric tokens '  //
     .                 'while attempting to read a state  '  //
     .                 'vector.  File is <*>  Last line '    //
     .                 'read was #*. Data record started '   //
     .                 'on line #*. Initial line of record ' //
     .                 'was <*> '                             )
         CALL ERRCH  ( '*',  INPUT                            ) 
         CALL ERRINT ( '*',  LNUM                             )
         CALL ERRINT ( '*',  LSTART                           )
         CALL ERRCH  ( '*',  LINE                             ) 
         CALL SIGERR ( 'SPICE(TOOFEWTOKENS)'                  )
         CALL CHKOUT ( 'PRSDR'                                )
         RETURN
 


      ELSE
C
C        We have extra tokens.  Normally, this is due to one or more
C        numbers wrapping across lines.  We may try to handle this
C        case later, but for now it's considered an error.
C
         CALL SETMSG ( 'Found more than six numeric tokens ' //
     .                 'while attempting to read a state  '  //
     .                 'vector.  File is <*>  Last line '    //
     .                 'read was #*. Data record started '   //
     .                 'on line #*. Initial line of record ' //
     .                 'was <*> '                             )
         CALL ERRCH  ( '*',  INPUT                            ) 
         CALL ERRINT ( '*',  LNUM                             )
         CALL ERRINT ( '*',  LSTART                           )
         CALL ERRCH  ( '*',  LINE                             ) 
         CALL SIGERR ( 'SPICE(TOOMANYTOKENS)'                 )
         CALL CHKOUT ( 'PRSDR'                                )
         RETURN
 
      END IF

      CALL CHKOUT ( 'PRSDR' )
      RETURN
      END
