C$Procedure CVTMEX ( Convert MEX file )
 
      SUBROUTINE CVTMEX ( SETUP, INPUT, OUTPUT, APPEND, CMTFIL )
 
C$ Abstract
C
C     Convert MEX orbit or attitude file to SPICE format.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
 
      IMPLICIT NONE
 
      CHARACTER*(*)         SETUP
      CHARACTER*(*)         INPUT
      CHARACTER*(*)         OUTPUT
      LOGICAL               APPEND
      CHARACTER*(*)         CMTFIL
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     SETUP      I   Name of setup file.
C     INPUT      I   Name of input MEX file.
C     OUTPUT     I   Name of output SPK file.
C     APPEND     I   Append flag.
C     CMTFIL     I   Comment file name.
C
C$ Detailed_Input
C
C     SETUP          is the name of a MEX2KER setup file.
C
C     INPUT          is the name of a MEX orbit/attitude file to be
C                    converted to SPK format.
C
C     OUTPUT         is the name of an SPK file to be written.
C                    OUTPUT may be a new or existing file.
C
C     APPEND         is a logical flag indicating whether to 
C                    append data to an existing output file.
C                    If APPEND is .TRUE., the file designated 
C                    by OUTPUT must exist.  If APPEND is .FALSE.,
C                    the file designated by OUTPUT must not exist.
C
C     CMTFIL         is the name of a comment file whose contents
C                    are to be added to the comment area of
C                    the SPK file created by this program.  The
C                    comment file contents precede the default
C                    comments added by MEX2KER.
C
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If the version ID line is not found in the MEX file,
C        the error SPICE(INVALIDFORMAT) will be signaled.
C
C     2) If the format version is not supported,
C        the error SPICE(NOTSUPPORTED) will be signaled.
C
C     3) Errors detected while reading the MEX file will be 
C        diagnosed by routines called by this routine.
C
C     4) Errors detected while parsing the MEX file will be
C        diagnosed by routines called by this routine.
C
C     5) Errors detected while opening the output SPK file will be
C        diagnosed by routines called by this routine.
C
C     6) Errors detected while writing the output SPK file will be
C        diagnosed by routines called by this routine.
C
C$ Files
C
C     See the descriptions of INPUT and OUTPUT above.
C
C$ Particulars
C
C     This routine is intended to enable MEX2KER to accommodate
C     updates to the MEX format specification.  CVTMEX reads the
C     MEX's version ID line, then calls the appropriate subroutine
C     to convert the MEX file to SPK format.
C
C     Currently, there is one MEX format version:  version 1.0.
C     The expected version ID line should be, up to variations
C     of white space:
C
C        CCSDS_MEX_VERS = 1.0
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    MEX2KER Version 1.1.1, 21-JUN-2012 (NJB)
C     
C        Various header updates were made.
C
C-    MEX2KER Version 1.1.0, 14-FEB-2003 (NJB)
C
C        Imported file from EPM2SPK; update comments.
C
C-    MEX2KER Version 1.0.0, 27-MAR-2002 (NJB)
C
C-&


C
C     SPICELIB functions
C
      LOGICAL               EQSTR
      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C
      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 255 )

      INTEGER               SHORT
      PARAMETER           ( SHORT  = 80 )
C
C     Local variables
C
      CHARACTER*(SHORT)     KEYWRD
      CHARACTER*(LNSIZE)    LINE
      CHARACTER*(SHORT)     TYPE
      CHARACTER*(SHORT)     VERSTR


      INTEGER               LNUM

      LOGICAL               EOF



      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'CVTMEX' )
C

C     Find the version line in the input file.  Hand off the
C     conversion problem to the appropriate routine.
C

      CALL RDLIN ( INPUT, LINE, LNUM, TYPE, EOF ) 

      DO WHILE (        ( TYPE .NE. 'VERSION_LINE' ) 
     .            .AND. ( .NOT. EOF                )
     .            .AND. ( .NOT. FAILED()           )  )
         
         CALL RDLIN ( INPUT, LINE, LNUM, TYPE, EOF ) 

      END DO

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'CVTMEX' )
         RETURN
      END IF

      IF ( EOF ) THEN

         CALL SETMSG ( 'Format error in MEX file <#>: version line ' //
     .                 'not found.'                                   )
         CALL ERRCH  ( '#', INPUT                                     )
         CALL SIGERR ( 'SPICE(INVALIDFORMAT)'                         )
         CALL CHKOUT ( 'CVTMEX'                                       )
         RETURN

      END IF

C
C     Pick off the format version from the version line.
C
      CALL KEYVLC ( INPUT, LINE, LNUM, KEYWRD, VERSTR ) 

      IF (  EQSTR( VERSTR, '1.0' )  ) THEN
C
C        Call the converter for this format type.
C
         CALL CVT01 ( SETUP, INPUT, OUTPUT, APPEND, CMTFIL )

      ELSE

         CALL SETMSG ( 'Format version <#> of MEX file <#> '      //
     .                 'is not supported by this program. '       //
     .                 'Check whether a later version of MEX2KER '//
     .                 'is available.'                             )
         CALL ERRCH  ( '#', VERSTR                                 )
         CALL ERRCH  ( '#', INPUT                                  )
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                       )
         CALL CHKOUT ( 'CVTMEX'                                    )
         RETURN

      END IF

      CALL CHKOUT ( 'CVTMEX' )
      RETURN
      END

      
