import shutil
import os
import sys
import logging
import glob as glob
import datetime
import socket

from adcsng.core.mkgap import mkgap
from adcsng.core import email, kernelgen
from adcsng.core.mkgen import mkgen
from adcsng.core.orbnumgen import orbnumgen
from adcsng.core.config import sources2kernels


from adcsng.utils.files import compare_mk
from adcsng.utils.files import clean_temp_input
from adcsng.utils.files import update_former_versions


def  director(config, setup, adcsng_setup, directories, debug, log_file,
              noemail):

    mission = config['mission']

    sources_text = ''
    kernels_text = ''
    kernel_names = []
    original_sources = []
    temp_dirs = []
    log_started = False
    tcp_present = False

    #
    # The current version of ADCSng is obtained
    #
    version = adcsng_setup['generic'][0]['adcsng_version']

    #
    # First we need to generate the new SCLK from TCPs
    #
    if config['mkgen'] != 'Only':
        (kernels, original_sources) = sources2kernels(setup, adcsng_setup, directories,
                                    sclk=True)
    else:
        kernels = []

    if kernels:
        tcp_present = True
        logging.info(
                '================================================================================')
        logging.info('ADCSng v{} for {} run on {} at {}'.format(
                version, mission, socket.gethostname(),
                str(datetime.datetime.now())[:-5]))
        logging.info(
                '================================================================================')

        log_started = True

        sources_text = ''
        logging.info('SCLK Generation executed first. TCP sources to be processed:')
        logging.info('')
        for kernel in kernels:
            if isinstance(kernel.source, list):
                for source in kernel.source:
                    if source not in sources_text and 'adcsng_temp' not in source:
                        sources_text += '   {}\n'.format(source)
                        log_text = '   {}'.format(source).replace('   ', '   >>')
                        logging.info(log_text)
            else:
                if kernel.source not in sources_text and 'adcsng_temp' not in kernel.source:
                    sources_text += '   {}\n'.format(kernel.source)
                    log_text = '   {}'.format(kernel.source).replace('   ', '   >>')
                    logging.info(log_text)
        logging.info(
                '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -')

    (kernel_names, sclk_kernel, temp_sclk_dir) = kernelgen.kernel_generation(
        kernels, directories, debug)

    temp_dirs += temp_sclk_dir

    #
    # Check the incoming directory for new OEM, AEM or HK TM
    #
    if config['mkgen'] != 'Only':
        (kernels, original_sources) = sources2kernels(setup, adcsng_setup,
                                                    directories,
                                                    original_sources=original_sources)

    #
    # We order the kernels list in order to ensure that orbit files go first
    # so we sort by kernel type and invert: spk to go before ck.
    #
    kernels = sorted(kernels, key=lambda kernel: kernel.source_type, reverse=True)

    if kernel_names or kernels:

        if not log_started:
            logging.info(
                    '================================================================================')
            logging.info('ADCSng v{} for {} run on {} at {}'.format(
                    version, mission, socket.gethostname(),
                    str(datetime.datetime.now())[:-5]))
            logging.info(
                    '================================================================================')

        sources_to_be_processed = False
        for source in original_sources:
            if source not in sources_text and 'adcsng_temp' not in source:
                sources_to_be_processed = True

        if sources_to_be_processed:
            logging.info('Sources to be processed:')
            logging.info('')
            for source in original_sources:
                if source not in sources_text and 'adcsng_temp' not in source:
                        sources_text += '   {}\n'.format(source)
                        log_text = '   {}'.format(source).replace('   ', '   >>')
                        logging.info(log_text)
            logging.info(
                '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -')

        (kernel_names, kernels, temp_ker_dirs) = kernelgen.kernel_generation(
            kernels, directories,debug,kernel_names=kernel_names, setup=adcsng_setup)
        temp_dirs += temp_ker_dirs
    else:

        #
        # ADCSng might need to generate a new mk without having had processed any file.
        #
        if 'mkgen' in adcsng_setup.keys():

            if config['mkgen'] == 'False':

                #
                # We need to remove the SCLK temp directories that have been created in case that the TCPs
                # have not been processed nor there were any inputs.
                #
                if not debug and temp_dirs:
                    for dir in temp_dirs:
                        if os.path.exists(dir):
                            shutil.rmtree(dir)
                if not debug and temp_dirs:
                    sources = sclk_kernel[0].source_filename[-1]
                    for source in sources:
                        try:
                            os.remove(os.path.join(directories.source, str(source)))
                        except:
                            pass

                #
                # We need to make sure the log file is not written if it was a new file
                #
                if os.stat(directories.log + os.sep + log_file.split(os.sep)[-1]).st_size == 0:
                    os.remove(directories.log + os.sep + log_file.split(os.sep)[-1])

                print(
                        'ADCSng v{} for {} did not process any source file.'.format(
                                version, mission))

                return tcp_present

        if not log_started:
            logging.info(
                    '================================================================================')
            logging.info('ADCSng v{} for {} run on {} at {}'.format(
                    version, mission, socket.gethostname(),
                    str(datetime.datetime.now())[:-5]))
            logging.info(
                    '================================================================================')

    moved_mk = []
    updated_mk = []
    updated_mk_flag = False
    logging.info('')

    logging.info('- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -')
    logging.info('+ Executing MKGEN to generate meta-kernels')
    for mk_type in adcsng_setup['mkgen'][0]['types'][0]:

        #
        # Generate meta-kernel for all the adcsng execution
        #
        mk = mkgen(config=adcsng_setup,
                   directories=directories,
                   kernel_grammar=directories.root_dir + '/config/' +
                                  adcsng_setup['mkgen'][0]['mission'].lower() +
                                  '_' + mk_type + '.grammar',
                   mk_type=mk_type)

        mk_path = os.path.join(directories.output, 'mk')

        #
        # We check if the meta-kernel has been updated
        #
        cwd = os.getcwd()
        try:
            os.chdir(mk_path)
        except:
            logging.warning('   MK directory was not present.')
            os.mkdir(mk_path)
        try:
            current_mk = mk_path + os.sep + glob.glob(
                adcsng_setup['mkgen'][0]['types'][0][mk_type][0]['filename'] + '*')[0]
            mk_are_different = compare_mk(current_mk, cwd + os.sep + mk)
        except:
           mk_are_different = True

        os.chdir(cwd)

        if not mk_are_different and config['mkgen'] != "Only":
            logging.warning('   MK {} has not been updated'.format(mk))
        else:
            #
            # Release the meta-kernel to the output area if the meta-kernel
            # has been updated at all.
            #
            if os.path.exists(mk_path):
                shutil.copy(mk, mk_path)
                logging.info('   Generated: {} '.format(mk))

            else:
                err_message = '{} directory does not exist'.format(
                            mk_path)

                logging.error(err_message)

                sys.exit(err_message)

            if config['mission'] != 'ExoMarsRSP':
                #
                # Overwrite the previous fix version of the meta-kernel
                #
                if mk.isupper():
                    mk_permanent = mk.split('_V')[0] + '.TM'
                else:
                    #
                    # Special case for Solar Orbiter that mixes uppercase and
                    # lowercase.
                    #
                    if '_V' in mk: mk_permanent = mk.split('_V')[0] + '.tm'
                    else: mk_permanent = mk.split('_v')[0] + '.tm'

                kernel_names.append(mk)

                shutil.copy(mk, os.path.join(mk_path, mk_permanent))
                logging.info('   Generated permanent MK: {} '.format(mk_permanent))

                #
                # Move the previous meta-kernel to former versions and update the PATH.
                #
                cwd = os.getcwd()
                os.chdir(mk_path)
                mks_in_dir = glob.glob(
                    adcsng_setup['mkgen'][0]['types'][0][mk_type][0]['filename'] + '*')

                for mk_in_dir in mks_in_dir:
                    if mk_in_dir != mk:
                        moved_mk.append(mk_in_dir)

                        try:
                            if not os.path.exists(os.path.join(directories.output,'mk/former_versions')):
                                os.makedirs(os.path.join(directories.output,'mk/former_versions'))

                            shutil.move(mk_in_dir, os.path.join(directories.output,'mk/former_versions'))

                        except:
                            logging.warning('   MK {} already present in former_versions directory'.format(mk_in_dir))

        os.chdir(cwd)
        os.remove(mk)

        #
        # Check the meta-kernels in the former versions directory. And
        # update them if needed.
        #
        if config['mission'] != 'ExoMarsRSP':
            try:
                updated_mk = update_former_versions(mk_path, directories.output, updated_mk)
            except:
                logging.error('   Error updating former meta-kernels:')
                error_list = sys.exc_info()[1]
                try:
                    for element in error_list.args[0]:
                        logging.error(f'      {element}')
                except:
                    logging.error(f'      {error_list}')
                updated_mk = []


        os.chdir(cwd)

    #
    # orbnum files are generated at the end of the run, execution moves forward
    # regardless of generation
    #
    orbnum_input = []
    after_error = False
    for kernel in kernels:
        if kernel.orbnum != 'False':
            orbnum_input.append(kernel)

    #
    # If there are SPKs that require ORBNUM generation
    #
    if orbnum_input:
        try:
            orbnum_input = sorted(orbnum_input, key=lambda x: x.source)
            for input_spk in orbnum_input:
                orbnum = orbnumgen(adcsng_setup, directories, input_spk, mk)

                #
                # In Mars-Express we generate two ORBNUMS which are the same
                #
                if orbnum not in kernel_names:
                    kernel_names.append(orbnum)
        except ValueError as value:
            after_error = True

            logging.error('   ORBNUM file not generated:')
            logging.error(f'      {value}')


    #
    # We generate the Gap report files
    #
    log_start = False
    for kernel in kernels:
        if 'gap_file' in kernel.config:
            try:
                if kernel.config['gap_file'] == 'True':
                    if log_start == False:
                        logging.info('- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -')
                        logging.info('+ Executing MKGAP to generate gap files for CK(s): ')

                    main_sc_name = kernel.config['main_sc_name']
                    object_frame = kernel.config['object_name']
                    exe_dir = directories.executables
                    lsk = kernel.working_directory + os.sep + kernel.config['lsk']
                    sclk = kernel.working_directory + os.sep + kernel.config['sclk']
                    fk = kernel.working_directory + os.sep + kernel.config['fk']
                    ck = kernel.working_directory + os.sep + str(kernel.name)



                    gap_string = mkgap(main_sc_name, object_frame, exe_dir, version,
                                       target_frame='J2000', minimum_duration='',
                                       lsk=lsk, sclk=sclk, fk=fk, ck=ck, histogram=False)

                    #
                    # We generate the gap file now
                    #
                    logging.info(f'   {str(kernel.name)}')
                    if str(kernel.name).split('.')[0] == str(kernel.name).split('.')[0].upper(): gapext = '.GAP'
                    else: gapext = '.gap'
                    try:
                        with open(directories.reports + os.sep + str(kernel.name).split('.')[0] + gapext, 'w+') as f:
                            for line in gap_string: f.write(line)
                    except:
                        try:
                            os.mkdir(directories.reports)
                        except:
                            logging.fatal('  Reports directory not able to be created.')
                        logging.error('  Reports directory for GAP file does not exist, it has been created:')
                        logging.error(f'    {directories.reports}')
                        with open(directories.reports + os.sep + str(kernel.name).split('.')[0] + gapext, 'w+') as f:
                            for line in gap_string: f.write(line)

                    log_start = True
            except:
                after_error = True
                logging.error('   GAP file not generated due to error:')
                error_list = sys.exc_info()[1]
                try:
                    for element in error_list.args[0]:
                        logging.error(f'      {element}')
                except:
                    logging.error(f'      {error_list}')

    #
    # Clean up the mess (unless we are testing)
    #
    if not debug:
        for dir in temp_dirs:
            if os.path.exists(dir):
                shutil.rmtree(dir)

    #
    # If there are no kernels generated for MKs no need to send an email
    #
    if not kernel_names and not mk_are_different:
        logging.info(
                '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -')
        logging.info('No kernels have been generated.')
        logging.info('')

    else:
        #
        # Write the log and information for the e-mail body
        #
        logging.info(
            '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -')
        logging.info('Generated Kernels:')
        logging.info('')

        for kernel in kernel_names:
            kernels_text += '   {}\n'.format(kernel)
            log_text = '   {}'.format(kernel).replace('   ', '   <<')
            logging.info(log_text)

        logging.info('')
    #
    # We move the old MK to the 'former_versions' directory
    #
    if moved_mk:
        logging.info("MK moved to 'former_versions' directory:")
        logging.info('')
        for mk in moved_mk:
            logging.info('   {}'.format(mk))

        logging.info('')

    if updated_mk:
        first_bool = True
        for mk in updated_mk:
            if mk not in moved_mk:
                if first_bool:
                    logging.warning(
                        "MK in 'former_versions' directory updated:")
                    logging.warning('')
                logging.warning('   {}'.format(mk))
                first_bool = False

        logging.info('')

    #
    # Here we report if there are files remaining in the incoming directory
    # This can happen if the sources do not match any source file in the
    # config
    #
    files = glob.glob(directories.source + '/*')
    if files:
        clean_temp_input(directories.source, debug=debug)

    if 'email' in config.keys() and kernel_names and not noemail:
        if config['email'][0]['send_email'] == "True":
            #
            # Send Kernel Generation e-mail notification to clients
            #
            if config['email'][0]['clients']:
                email.send_status_email(config,
                                        sources_text=sources_text,
                                        kernels_text=kernels_text,
                                        error=after_error)

            os.chdir(directories.root_dir + '/..')

            #
            # Send ADCSng log to the developer of the last run
            #
            email_body = ''
            with open(directories.temp + '/temp.log', 'r') as l:
                for line in l:
                    email_body += line

            email.send_status_email(config, sources_text=sources_text,
                                    body_text=email_body)


    logging.info('ADCSng v{} for {} run finished at {}'.format(
            version, mission, datetime.datetime.now()))


    return tcp_present