#!/bin/tcsh
#
# This script generates PDS4 labels for SPICE archive description HTML file(s).
#
# Pre-requisites/requirements:
#
# -- PDS4 values file and XML template for description HTML document
#    must exist
#
# -- a single un-labeled archive description HTML file with the latest
#    sequemtial verison must exist
#
# -- PDS3->PDS4 tool, md5sum must be executables
#
# -- PDS4 labels (*.xml) must exists for meta-kernels in SPICE kernel collection.
#
# -- PDS4 values file must include required keywords (see code for list)
#
# Version 1.0.0 -- May 19, 2015
#
#   Initial version.
#
# Version 1.1.0 -- November 9, 2015
#
#   Changed to set start and stop times based on any kernel XML 
#   labels rather than just MK labels.
#
# Version 1.2.0 -- November 15, 2017
#
#   Updated Unix date format string (%X->%T).
#
# Script name and verison:
#
set SCRIPTNAME = "make_pds4_doc_spiceds_label.csh"
set SCRIPTVER  = "1.2.0 -- November 15, 2017"

#
# Get command line stuff.
#
if ( $#argv != 2 ) then
    echo " "
    echo "$SCRIPTNAME, version $SCRIPTVER "
    echo " "
    echo "Usage: "
    echo
    echo "   % $SCRIPTNAME <valuefile> <templatefile>"
    echo " "
    echo "where:"
    echo " "
    echo "   valuefile    -- fixed PDS4 values KWD=VAL file"
    echo "   templatefile -- spiceds HTML document label XML template file"
    echo
    exit 1
endif

set PDS4VALUES = $1
set XMLTEMPL   = $2

#
# Tools: PDS3-to-PDS4 converter, checksum utility.
#
set PDS4GEN    = "/home/bsemenov/maven/pds/make_pds4_xml_from_lbl.csh"
set MD5SUM     = "/usr/bin/md5sum"

#
# Collection directory and inventory and label name prefixes.
#
set SCOLDIR    = "spice_kernels"
set DCOLDIR    = "document"
set NAMPREF    = "spiceds_v"
set HTMLEXT    = "html"
set XMLEXT     = "xml"

#
# Check input files and programs.
#
foreach FF ( $PDS4VALUES $XMLTEMPL )
    if ( ! -e $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF does not exit."
	echo
	exit 1
    endif
end

foreach FF ( $PDS4GEN $MD5SUM )
    if ( ! -x $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF is not an excutable."
	echo
	exit 1
    endif
end

foreach FF ( $SCOLDIR $DCOLDIR )
    if ( ! -d $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF is not a directory."
	echo
	exit 1
    endif
end

#
# Check that PDS4 values files has all required keywords. Extract
# BUNDLE_LID.
#
foreach KWD ( PDS4_MISSION_NAME \
              PDS4_MISSION_LID \
              PDS4_SPACECRAFT_NAME  \
              PDS4_SPACECRAFT_LID  \
              PDS4_TARGET_NAME  \
              PDS4_TARGET_TYPE  \
              PDS4_TARGET_LID  \
              BUNDLE_LID  \
              BUNDLE_DESCRIPTION_LID )

    set COUNT = `grep -c "^ *$KWD *=" $PDS4VALUES` 
    if ( $COUNT == 0 ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $PDS4VALUES does not include $KWD."
	echo
	exit 1
    endif
    if ( $COUNT > 1 ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $PDS4VALUES includes 2 or more $KWD."
	echo
	exit 1
    endif
end

set BUNDLID  = `grep BUNDLE_LID $PDS4VALUES         | sed 's/=/ /' | awk '{print $NF}'`

#
# Get the name of the description HTML doc to be labelled. Make sure
# there is only one such doc, that its version was correctly
# incremented and that it does not yet have a label.
#
# First check that HTML doc are named correctly.
#
set COUNT1 = `ls -1 $DCOLDIR/$NAMPREF*.$HTMLEXT | wc -l`
set COUNT2 = `ls -1 $DCOLDIR/$NAMPREF*.$HTMLEXT | egrep "/$NAMPREF"'[0-9][0-9][0-9]'".$HTMLEXT" | wc -l`

if ( $COUNT1 != $COUNT2 ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: one or more $DCOLDIR/$NAMPREF*.$HTMLEXT do not follow the required naming/versioning scheme."
    echo
    exit 1
endif

if ( $COUNT1 == 0 ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: did not find any $DCOLDIR/$NAMPREF*.$HTMLEXT to label."
    echo
    exit 1
endif

#
# Next check that only one HTML doc dooes not have a label.
#
set COUNT = 0
foreach FF ( $DCOLDIR/$NAMPREF*.$HTMLEXT )
    if ( ! -e $FF:r.$XMLEXT ) then
	@ COUNT = $COUNT + 1
	set DOCNAME = $FF
    endif
end

if ( $COUNT == 0 ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: all $DCOLDIR/$NAMPREF*.$HTMLEXT files seem to have labels."
    echo
    exit 1
endif

if ( $COUNT > 1 ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: more than one $DCOLDIR/$NAMPREF*.$HTMLEXT doesn't have a label."
    echo
    exit 1
endif

#
# Next check that versions start from 1 and are sequential and the
# unlabeled one is the latest version.
#
set DOCVER = "000" 
foreach FF ( `ls -1 $DCOLDIR/$NAMPREF*.$HTMLEXT | sed "s/^.*$NAMPREF//" | sed "s/.$HTMLEXT//" | sort -n` )
    set TMPVER = `echo $FF | sed 's/^0*//' | awk '{printf "%03d", $1-1}'`
    if ( "$DOCVER" != "$TMPVER" ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $DCOLDIR/$NAMPREF*.$HTMLEXT version are not sequential. Version $FF follows version $DOCVER."
	echo
	exit 1
    endif
    set DOCVER = "$FF"
end

if ( "$DOCNAME" != "$DCOLDIR/$NAMPREF$DOCVER.$HTMLEXT" ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: the latest un-labelled HTML description $DOCNAME doesn't have the latest version $DOCVER."
    echo
    exit 1
endif

#
# Make sure that the document has lines terminated by CR-LF.
#
if ( `cat $DOCNAME | wc -l` != `sed 's/\r/ CR/' $DOCNAME | grep ' CR$' | wc -l` ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: the count of lines and CRs in $DOCNAME don't agree."
    echo
    exit 1
endif

#
# Make document label.
#
set DOCXML  = $DOCNAME:r.$XMLEXT

set TMPLIST = ""

#
# Determine START_TIME and STOP_TIME time. Grep PDS labels
# and pick the earliest START and the latest STOP.
#
\ls $SCOLDIR/*/*.xml >& /dev/null

if ( $status ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot find any MK labels $SCOLDIR/*/*.xml."
    echo
    exit 1
endif

set START = `grep 'start_date_time.*[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]' $SCOLDIR/*/*.xml | sed 's/^.*\([0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]\).*$/\1/i' | sort | head -1`
set STOP  = `grep 'stop_date_time.*[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]'  $SCOLDIR/*/*.xml | sed 's/^.*\([0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]\).*$/\1/i' | sort | tail -1`

if ( "$START" == "" ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot determine start time from MK labels $SCOLDIR/*/*.xml."
    echo
    exit 1
endif

if ( "$STOP" == "" ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot determine stop time from MK labels $SCOLDIR/*/*.xml."
    echo
    exit 1
endif

#
# Set PRODUCT_CREATION_TIME and year to CPU clock.
#
set CURRTIME = `date +"%FT%T"`
set CURRDATE = `date +"%F"`
set CURRYEAR = `date +"%Y"`

#
# Get file size from the file system.
#
set FILESIZE = `ls -l $DOCNAME | awk '{print $5}'`

#
# Get MD5 checksum.
#
set CHECKSUM = `$MD5SUM $DOCNAME | awk '{print $1}'`

#
# Set document LID: <bundle_lid>:<colldir>
#
set DOCLID = "${BUNDLID}":"${DCOLDIR}":`echo $NAMPREF | sed 's/_v$//i'`

#
# Set document VID: version.0
#
set DOCVID = `echo $DOCVER | sed 's/^0*//' | awk '{printf "%d.0", $1}'`

#
# Make complete temporary KWD=VAL PDS3-like label
#
set TMPLBL = "$DOCXML:r.tmp.lbl"

\rm $TMPLBL >& /dev/null

echo                                        >>  $TMPLBL
echo "PRODUCT_LID           = "$DOCLID      >>  $TMPLBL
echo "PRODUCT_VID           = "$DOCVID      >>  $TMPLBL
echo "FILE_NAME             = "$DOCNAME:t   >>  $TMPLBL
echo "CURRENT_YEAR          = "$CURRYEAR    >>  $TMPLBL
echo "CURRENT_DATE          = "$CURRDATE    >>  $TMPLBL
echo "CURRENT_TIME          = "$CURRTIME    >>  $TMPLBL
echo "FILE_SIZE             = "$FILESIZE    >>  $TMPLBL
echo "FILE_CHECKSUM         = "$CHECKSUM    >>  $TMPLBL
echo                                        >>  $TMPLBL
cat $PDS4VALUES                             >>  $TMPLBL
echo                                        >>  $TMPLBL
echo "START_TIME            = "$START       >>  $TMPLBL
echo "STOP_TIME             = "$STOP        >>  $TMPLBL
echo                                        >>  $TMPLBL

set TMPLIST = "$TMPLIST $TMPLBL"

#
# Make temporary PDS4 label.
#
set TMPXML    = "$TMPLBL.xml"
set TMPXMLLOG = "$TMPXML.log"

\rm $TMPXML $TMPXMLLOG >& /dev/null

$PDS4GEN $TMPLBL $XMLTEMPL >& $TMPXMLLOG

if ( $status ) then

    #
    # Show error output.
    #
    cat $TMPXMLLOG
    echo
    echo "Values   file was $TMPLBL."
    echo "Template file was $XMLTEMPL."
    echo 

else 

    #
    # Success. Rename temporary XML label.
    #
    mv $TMPXML $DOCXML
    if ( $status ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: failed to move $TMPXML to $DOCXML."
	echo
	exit 1
    endif

    set TMPLIST = "$TMPLIST $TMPXMLLOG"

    echo 
    echo "Created $DOCXML"

endif

#
# Cleanup.
#
\rm $TMPLIST >& /dev/null

#
# All done.
#
echo

exit
