#!/usr/bin/perl
#   metacharacters:        {}[]()^$.|*+?\
#   for character class:   -]\^$

#****************************************************************************#
#- Table of contents
##TOC
#****************************************************************************#
#- 1 Settings
#-   1.1 Metakernel
#-   1.2 Archive DATA directory
#-   1.3 SKD kernels directory
#-   1.4 SPICE executables
#-   1.5 Seperator between file reports
#- 2 Read metakernel
#- 3 Get list of kernels in archive
#- 4 Loop through list
#-   4.1 Get path to kernel in archive
#-   4.2 Get path to kernel in SKD
#-   4.3 Treat explicitely
#-   4.4 Check difference
#-       4.4.1 Get difference for text kernels
#-       4.4.2 Get difference for SPKs
#-             4.4.2.1 Comment
#-             4.4.2.2 Data

#****************************************************************************#
#- 1
## Settings
#****************************************************************************#

#============================================================================#
#- 1.1
### Metakernel
#============================================================================#
$metakernel = '~/VULCAN/SPICE/ARCGEN/ARCGEN_OUTPUT/EXTRAS/MK/ROS_V06.TM';

#============================================================================#
#- 1.2
### Archive DATA directory
#============================================================================#
$data = '~/VULCAN/SPICE/ARCGEN/ARCGEN_OUTPUT/DATA';

#============================================================================#
#- 1.3
### SKD kernels directory
#============================================================================#
$kernels = '/var/ftp/pub/data/SPICE/ROSETTA/kernels';

#============================================================================#
#- 1.4
### SPICE executables
#============================================================================#
$exe = '~/SUN/FORTRAN/toolkit/exe';

#============================================================================#
#- 1.5
### Seperator between file reports
#============================================================================#
$sep = "\n====================================".
         "====================================\n\n";

#****************************************************************************#
#- 2
## Read metakernel
#****************************************************************************#
@mk = `cat $metakernel`;

#****************************************************************************#
#- 3
## Get list of kernels in archive
#****************************************************************************#
@list = `cd $data; ls */*.* | grep -v '\.LBL'`;

#****************************************************************************#
#- 4
## Loop through list
#****************************************************************************#
foreach ( @list ) {
    chomp;

#============================================================================#
#- 4.1
### Get path to kernel in archive
#============================================================================#
    $inarc = "$data/$_";

#============================================================================#
#- 4.2
### Get path to kernel in SKD
#============================================================================#
    if ( /^(\w+)\/(\S+)$/ ) {
        $dir = $1;
        $file = $2;
    } else {
	die;
    }
    $dir =~ tr/[A-Z]/[a-z]/;
    $inskd = "$kernels/$dir/$file";

#============================================================================#
#- 4.3
### Treat explicitely
#============================================================================#
#  These files were dosified in the archive (in a copy of it used for this
#  comparison), after which there was no more difference.
    if ( $file eq 'ROS_DIM_V10.TI' ) {
	print $sep;
	print "$file has DOS line endings in SKD.\n";
    } elsif ( $file eq 'ROS_GIADA_V11.TI' ) {
	print $sep;
	print "$file has DOS line endings in SKD.\n";
    }

#============================================================================#
#- 4.4
### Check difference
#============================================================================#
    $isdiff = 0;
    $diff = `diff $inarc $inskd 2>&1`;
    if ( $diff ) {
	if ( $diff =~ /No such file or directory/ ) {
	    $inskd = "$kernels/$dir/former_versions/$file";
	    $diff = `diff $inarc $inskd 2>&1`;
	    if ( $diff ) {
		print $sep;
		if ( $diff =~ /No such file or directory/ ) {
		    $inmk = 0;
		    foreach ( @mk ) {
			if ( /$file/ ) { $inmk = 1 }
		    }
		    if ( $inmk ) {
			print "$file missing in SKD (in MK).\n";
		    } else {
			print "$file missing in SKD (not in MK).\n";
		    }
		} else {
		    $isdiff = 1;
		    print "$file different in SKD (former).\n";
		}
	    }
	} else {
	    $isdiff = 1;
	    print $sep;
	    $inmk = 0;
	    foreach ( @mk ) {
		if ( /$file/ ) { $inmk = 1 }
	    }
	    if ( $inmk ) {
		print "$file different in SKD (in MK).\n";
	    } else {
		print "$file different in SKD (not in MK).\n";
	    }
	}
    }

    if ( $isdiff ) {

#----------------------------------------------------------------------------#
#- 4.4.1
#### Get difference for text kernels
#----------------------------------------------------------------------------#
	if ( $file =~ /\.T/ ) {
	    print `diff $inarc $inskd`;

#----------------------------------------------------------------------------#
#- 4.4.2
#### Get difference for SPKs
#----------------------------------------------------------------------------#
	} elsif ( $file =~ /\.BSP/ ) {
#	    print `diff $inarc $inskd`;

#............................................................................#
#- 4.4.2.1
##### Comment
#............................................................................#
	    `$exe/commnt -r $inarc > ../tmp/comarc.txt`;
	    `$exe/commnt -r $inskd > ../tmp/comskd.txt`;
	    print `diff ../tmp/comarc.txt ../tmp/comskd.txt`;

#............................................................................#
#- 4.4.2.2
##### Data
#............................................................................#
	    print `$exe/spkdiff -s 3600 $inarc $inskd`;

	}


    }

}
