C$Procedure PRSDR ( Parse data record from MEX Orbit/Attitude file )
 
      SUBROUTINE PRSDR ( INPUT, FILTYP, SUBTYP, TIMSYS, 
     .                   ET,    PACKET, FOUND           )
 
C$ Abstract
C
C     Parse a logical data record from a MEX/Rosetta Orbit or 
C     Attitude file.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     SPK
C     TIME
C
C$ Keywords
C
C     EPHEMERIS
C     FILES
C
C$ Declarations
 
      IMPLICIT NONE 

      INCLUDE 'spk18.inc'
      INCLUDE 'ck05.inc'

      CHARACTER*(*)         INPUT
      CHARACTER*(*)         FILTYP
      INTEGER               SUBTYP
      CHARACTER*(*)         TIMSYS
      DOUBLE PRECISION      ET
      DOUBLE PRECISION      PACKET  ( * )
      LOGICAL               FOUND

      CHARACTER*(*)         DELIMS
      PARAMETER           ( DELIMS = ', ' )

      INTEGER               MAXLIN
      PARAMETER           ( MAXLIN = 255 )      
 
C$ Brief_I/O
C
C     Variable  I/O  Description
C     --------  ---  --------------------------------------------------
C     INPUT      I   Input ESOC OASW orbit/attitude file name.
C     FILTYP     I   File type.
C     SUBTYP     I   SPK type 18 or CK type 5 subtype code.
C     TIMSYS     I   Name of time system.
C     ET         O   Time tag of packet.
C     PACKET     O   Data packet.
C     FOUND      O   Logical flag indicating whether record was found.
C
C$ Detailed_Input
C
C     INPUT          is the name of the ESOC OASW orbit/attiude file
C                    from which a data record is to be read.
C
C     FILTYP         is the type of input file to be read. Supported
C                    values are:
C
C                       'ORBIT FILE'
C                       'ATTITUDE FILE'
C
C                    Case and blanks (including embedded blanks) are
C                    not significant in FILTYP.
C             
C     SUBTYP         is the subtype corresponding to the SPICE data
C                    type that will be used to represent the data
C                    from the input file. SUBTYP must be a valid
C                    SPK type 18 subtype if the input file is an orbit
C                    file; SUBTYP must be a valid CK type 6 subtype
C                    if the input file is an attitude file.
C
C     TIMSYS         is a string indicating the time system associated
C                    with time tags in the input file. TIMSYS must be
C                    a time system name recognized by STR2ET.
C
C$ Detailed_Output
C
C     ET             is the epoch, expressed as seconds past J2000 TDB,
C                    represented by the time tag in the input data
C                    record.
C
C     PACKET         is the data packet contained in the data record
C                    read by this routine. The contents of the packet
C                    depend on the input file type and SPICE SPK or
C                    CK subtype.
C                    
C     FOUND          is a logical flag indicating whether a data packet
C                    was found. The outputs ET and PACKET are valid
C                    only if FOUND is .TRUE.
C
C$ Parameters
C
C     See the INCLUDE files spk18.inc and ck05.inc.
C     
C$ Exceptions
C
C     1)  If FILTYP is not recognized, the error SPICE(NOTSUPPORTED) is
C         signaled.
C
C     2)  If the input file is an orbit file and SUBTYP is not a
C         recognized SPK type 18 subtype, the error SPICE(NOTSUPPORTED)
C         is signaled.
C
C     3)  If the input file is an attitude file and SUBTYP is not a
C         recognzied CK type 6 subtype, the error SPICE(NOTSUPPORTED)
C         is signaled.
C
C     4)  If no time tag is found in the data record read from the 
C         input file, the error SPICE(TIMETAGNOTFOUND) is signaled.
C
C     5)  If the input data line or lines constituting the time tag and
C         data packet are not followed by a time tag, non-data line,
C         end of file, or if the maximum number of data lines is
C         exceeded while reading a single packet, the error
C         SPICE(ENDNOTFOUND) is signaled.
C
C     6)  If a token in the input packet is expected to represent
C         a d.p. number but doesn't parse correctly as such, the
C         error SPICE(INVALIDDPNUMBER) is signaled.
C
C
C     7)  If the number of d.p. numbers found in the input packet
C         is less than the correct packet size, the error
C         SPICE(TOOFEWTOKENS) is signaled.
C
C     8)  If the number of d.p. numbers found in the input packet
C         is greater than the correct packet size, the error
C         SPICE(TOOMANYTOKENS) is signaled.
C
C$ Files
C
C     See the description of the argument INPUT.
C
C$ Particulars
C
C     This routine is meant to be called repeatedly to fetch packets
C     from a data block in an ESOC OASW orbit or attitude file. When
C     the line or lines constituting a packet have been read, and the
C     next input line is the first line of the succeeding packet, that
C     line is "pushed" back onto the input stream so that the next call
C     to GETNCL executed by this routine will pick up that line. See
C     the entry points of the routine RDMEX for details.
C
C$ Examples
C
C     See usage in MKSKSG and MKCKSG.
C
C$ Restrictions
C
C     This routine is meant to be used only within the MEX2KER program.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman   (JPL)
C
C$ Version
C
C-    MEX2KER Version 2.2.0, 15-JUL-2014 (NJB)
C
C        Fixed duplicate argument bug in velocity
C        scaling VSCLG call. Updated header.
C
C-    MEX2KER Version 2.1.0, 02-SEP-2003 (NJB)
C
C        Bug fix:  swap code installed in last update now operates only
C        on SPK type 18, subtype 0 packets.
C
C-    MEX2KER Version 2.0.0, 14-JUL-2003 (NJB)
C
C        Now swaps velocity and "position derivative" packet components
C        when processing an orbit file.  This is necessary because
C        of the possible incompatibility of these values.
C
C-    MEX2KER Version 1.0.0, 13-MAY-2002 (NJB)
C
C-&
 
C
C     SPICELIB functions
C
      DOUBLE PRECISION      SPD

      LOGICAL               EQSTR
      LOGICAL               RETURN

C
C     Local parameters
C     
      INTEGER               BUFSIZ
      PARAMETER           ( BUFSIZ = 100 )

      INTEGER               MAXTTK
      PARAMETER           ( MAXTTK = 50 )

      INTEGER               MAXWRD
      PARAMETER           ( MAXWRD = MAXTTK + 6 )

      INTEGER               WRDLEN
      PARAMETER           ( WRDLEN = 80 )

      INTEGER               MSGLEN
      PARAMETER           ( MSGLEN = 320 )

      INTEGER               MAXDL
      PARAMETER           ( MAXDL  = 10 )

      INTEGER               BIG
      PARAMETER           ( BIG    = MAXDL * MAXLIN )

      INTEGER               TIMLEN
      PARAMETER           ( TIMLEN = 50 )

C
C     Local variables
C
      CHARACTER*(TIMLEN)    TAG
      CHARACTER*(MAXLIN)    INITLN
      CHARACTER*(TIMLEN)    NXTTAG
      CHARACTER*(MAXLIN)    LINE
      CHARACTER*(MSGLEN)    MSG
      CHARACTER*(WRDLEN)    TYPE
      CHARACTER*(WRDLEN)    WRDBUF ( MAXWRD )
      CHARACTER*(BIG)       BIGLIN

      DOUBLE PRECISION      TMPBUF ( BUFSIZ )

      INTEGER               DSTART
      INTEGER               ELOC
      INTEGER               I
      INTEGER               LNUM
      INTEGER               N
      INTEGER               NXTLOC
      INTEGER               NLINES
      INTEGER               PACKSZ
      INTEGER               LSTART

      LOGICAL               EOF
      LOGICAL               NXTFND


      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'PRSDR' )

C
C     Nothing found yet.
C
      FOUND = .FALSE.

C
C     Set the packet size, which is a function of the file type
C     and subtype.
C
      IF (  EQSTR(FILTYP, 'ORBIT FILE' )  ) THEN

         IF ( SUBTYP .EQ. S18TP0 ) THEN

            PACKSZ = S18PS0

         ELSE IF ( SUBTYP .EQ. S18TP1 ) THEN

            PACKSZ = S18PS1

         ELSE

            CALL SETMSG ( 'Unsupported SPK type 18 subtype # seen.' ) 
            CALL ERRINT ( '#', SUBTYP                               )
            CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                     )
            CALL CHKOUT ( 'PRSDR'                                   )
            RETURN

         END IF


      ELSE IF (  EQSTR(FILTYP, 'ATTITUDE FILE' )  ) THEN

         IF ( SUBTYP .EQ. C05TP0 ) THEN

            PACKSZ = C05PS0

         ELSE IF ( SUBTYP .EQ. C05TP1 ) THEN

            PACKSZ = C05PS1

         ELSE IF ( SUBTYP .EQ. C05TP2 ) THEN

            PACKSZ = C05PS2

         ELSE IF ( SUBTYP .EQ. C05TP3 ) THEN

            PACKSZ = C05PS3

         ELSE

            CALL SETMSG ( 'Unsupported CK type 5 subtype # seen.' ) 
            CALL ERRINT ( '#', SUBTYP                             )
            CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                   )
            CALL CHKOUT ( 'PRSDR'                                 )
            RETURN

         END IF

      ELSE

         CALL SETMSG ( 'Unsupported file type # seen.' ) 
         CALL ERRCH  ( '#', FILTYP                     )
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'           )
         CALL CHKOUT ( 'PRSDR'                         )
         RETURN

      END IF

C
C     Read the next non-comment line.
C
      CALL GETNCL ( INPUT, LINE, LNUM, TYPE, EOF )

      IF ( EOF ) THEN
C
C        No more data lines are to be found.
C
         CALL CHKOUT ( 'PRSDR' ) 
         RETURN

      END IF

      INITLN = LINE


      IF ( TYPE .NE. 'DATA_LINE' ) THEN
C
C        We've hit the end of a data block.  Put back the line
C        we just read.
C
         CALL PUSHLN

         CALL CHKOUT ( 'PRSDR' ) 
         RETURN

      END IF

C
C     Record this line number.
C     
      LSTART = LNUM

C
C     The file is supposed to be positioned at the first line of 
C     a logical data record.  Some initial token sequence constitutes
C     a time tag---normally, the tag is supposed to be the first
C     token.  Make sure we have the tag.
C
      CALL FNDTAG ( LINE, TAG, ELOC, FOUND )

      IF ( .NOT. FOUND ) THEN

         CALL SETMSG ( 'Data record starting at line #* in '  //
     .                 'file <*> did not start with a '       //
     .                 'recognized time tag.  Line was *'      )
         CALL ERRINT ( '*',  LNUM                              )
         CALL ERRCH  ( '*',  INPUT                             ) 
         CALL ERRCH  ( '*',  LINE                              ) 
         CALL SIGERR ( 'SPICE(TIMETAGNOTFOUND)'                )
         CALL CHKOUT ( 'PRSDR'                                 )
         RETURN

      END IF

      CALL TIMCVT ( INPUT, LINE, LNUM, TIMSYS, TAG, ET )

C
C     Since the original line may have wrapped onto successive lines,
C     keep reading lines until we hit the next time tag, non-data line,
C     end-of-file, or until we've read the max allowed number of lines.
C
      BIGLIN = LINE

C
C     Read the next non-comment line.
C
      NLINES = 1
      NXTFND = .FALSE.

      CALL GETNCL ( INPUT, LINE,   LNUM,   TYPE,  EOF )
      CALL FNDTAG ( LINE,  NXTTAG, NXTLOC, NXTFND     )

      DO WHILE (       ( .NOT. EOF             )
     .           .AND. ( .NOT. NXTFND          )
     .           .AND. ( TYPE .EQ. 'DATA_LINE' )  )

C
C        This line should be a continuation of the current 
C        record.
C
         CALL SUFFIX ( LINE, 1, BIGLIN )

         NLINES = NLINES + 1

         IF ( NLINES .EQ. MAXDL ) THEN
            
            CALL SETMSG ( 'Maximum (*) number of data lines were ' //
     .                    'read without finding the end of the '   //
     .                    'current logical record.  Input file '   //
     .                    'is <*> Last line number read was #*. '  //
     .                    'Data record started on line #*. '       //
     .                    'Initial line of record was <*>'          )
            CALL ERRINT ( '*',  MAXDL                               )
            CALL ERRCH  ( '*',  INPUT                               ) 
            CALL ERRINT ( '*',  LNUM                                )
            CALL ERRINT ( '*',  LSTART                              )
            CALL ERRCH  ( '*',  INITLN                              ) 
            CALL SIGERR ( 'SPICE(ENDNOTFOUND)'                      )
            CALL CHKOUT ( 'PRSDR'                                   )
            RETURN

         END IF

C
C        Get the next line and see whether we're looking at the 
C        start of a new data record.
C         
         CALL GETNCL ( INPUT, LINE,   LNUM,   TYPE,  EOF )
         CALL FNDTAG ( LINE,  NXTTAG, NXTLOC, NXTFND     )

      END DO


      IF ( .NOT. EOF ) THEN
C
C        We've hit the start of the next data record, or else we've 
C        encountered a metadata block start marker.  Put back the line
C        we just read.
C
         CALL PUSHLN

      END IF

C
C     Time to collect the d.p. data.
C
C     Fetch the individual words from the data line.  Strip out commas
C     to simplify the parsing.
C
      DSTART = ELOC + 1

      CALL REPLCH ( BIGLIN(DSTART:), ',', ' ', BIGLIN(DSTART:) )

      CALL LPARSM ( BIGLIN(DSTART:), DELIMS, MAXWRD, N, WRDBUF )

      IF ( N .EQ. PACKSZ ) THEN
C
C        Hopefully this easy case is normal, so we handle it
C        separately, for the sake of speed.
C
         DO I = 1, PACKSZ
 
            MSG = ' '


            READ ( WRDBUF(I), '(D 24.1)' ) PACKET(I)


            MSG = ' '

C            CALL NPARSD ( WRDBUF(I), PACKET(I), MSG, PTR )
 
            IF ( MSG .NE. ' ' ) THEN
C 
C              The current state component didn't parse as a d.p.
C              number.
C             
               CALL SETMSG ( 'Could not parse input as a double '  //
     .                       'precision number while attempting '  //
     .                       'to read a state vector.  File is '   //
     .                       '<*> Last line number read was #*. '  //
     .                       'Data record started on line #*. '    //
     .                       'Initial line of record was <*> '     //
     .                       'Unparsed token was *'                 )
               CALL ERRCH  ( '*',  INPUT                            ) 
               CALL ERRINT ( '*',  LNUM                             )
               CALL ERRINT ( '*',  LSTART                           )
               CALL ERRCH  ( '*',  LINE                             ) 
               CALL ERRCH  ( '*',  WRDBUF(I)                        ) 
               CALL SIGERR ( 'SPICE(INVALIDDPNUMBER)'               )
               CALL CHKOUT ( 'PRSDR'                                )
               RETURN
 
            END IF
 
         END DO


         IF (          EQSTR(FILTYP, 'ORBIT FILE')  
     .        .AND.  ( SUBTYP .EQ. S18TP0        )  ) THEN
C  
C           Scale velocity data to km/sec.
C
            CALL VSCLG ( 1.0D0/SPD(), 
     .                   PACKET( (PACKSZ/2) + 1 ), 
     .                   PACKSZ/2,
     .                   TMPBUF                    )

            CALL MOVED ( TMPBUF, PACKSZ/2, PACKET( (PACKSZ/2) + 1 )  )
  
C
C           Swap position derivative and velocity for incorporation
C           into type 18 packets.
C
            CALL SWAPAD ( 3, 4, 3, 7, PACKET )

         END IF



      ELSE IF ( N .LT. PACKSZ ) THEN
C 
C        Something important is missing.
C 
         CALL SETMSG ( 'Could not find * numeric tokens '    //
     .                 'while attempting to read a data '    //
     .                 'record.  File is <*>  Last line '    //
     .                 'read was #*. Data record started '   //
     .                 'on line #*. Initial line of record ' //
     .                 'was <*> '                             )
         CALL ERRINT ( '*',  PACKSZ                           )
         CALL ERRCH  ( '*',  INPUT                            ) 
         CALL ERRINT ( '*',  LNUM                             )
         CALL ERRINT ( '*',  LSTART                           )
         CALL ERRCH  ( '*',  LINE                             ) 
         CALL SIGERR ( 'SPICE(TOOFEWTOKENS)'                  )
         CALL CHKOUT ( 'PRSDR'                                )
         RETURN
 


      ELSE
C
C        We have extra tokens.  Normally, this is due to one or more
C        numbers wrapping across lines.  We may try to handle this
C        case later, but for now it's considered an error.
C
         CALL SETMSG ( 'Found more than * numeric tokens '   //
     .                 'while attempting to read a data '    //
     .                 'record.  File is <*>  Last line '    //
     .                 'read was #*. Data record started '   //
     .                 'on line #*. Initial line of record ' //
     .                 'was <*> '                             )
         CALL ERRINT ( '*',  PACKSZ                           )
         CALL ERRCH  ( '*',  INPUT                            ) 
         CALL ERRINT ( '*',  LNUM                             )
         CALL ERRINT ( '*',  LSTART                           )
         CALL ERRCH  ( '*',  LINE                             ) 
         CALL SIGERR ( 'SPICE(TOOMANYTOKENS)'                 )
         CALL CHKOUT ( 'PRSDR'                                )
         RETURN
 
      END IF

      CALL CHKOUT ( 'PRSDR' )
      RETURN
      END
