import spiceypy
import os
import shutil
import fileinput
import glob
import difflib
import logging

from arcgen.classes.label import KernelPDSLabel
from arcgen.classes.label import MetaKernelPDS4Label
from arcgen.classes.label import InventoryPDS3Label
from arcgen.classes.label import InventoryPDS4Label
from arcgen.classes.label import DocumentPDS4Label
from arcgen.classes.label import BundlePDS4Label
from arcgen.utils.files import md5
from arcgen.utils.time import creation_time
from arcgen.utils.time import PDS3_label_gen_date
from arcgen.utils.time import current_time
from arcgen.utils.files import extension2type
from arcgen.utils.files import type2extension
from arcgen.utils.files import extension2PDS3type
from arcgen.utils.files import read_errata
from arcgen.utils.files import get_latest_kernel
from arcgen.utils.files import safe_make_directory
from arcgen.utils.files import mk2list
from arcgen.utils.files import add_carriage_return

class Product(object):
    
    def __init__(self):

        statinfo = os.stat(self.path)
        self.size = str(statinfo.st_size)
        self.checksum = str(md5(self.path))
        self.creation_time = creation_time(self.path)

class ReadmeProduct(Product):

    def __init__(self, mission, bundle):

        self.name = 'readme.txt'
        self.bundle = bundle
        self.path = mission.bundle_directory + os.sep+ self.name
        self.mission = mission
        self.vid = bundle.vid

        self.write_product()
        Product.__init__(self)

        #
        # Now we change the path for the difference of the name in the label
        #
        self.path = mission.bundle_directory + os.sep + bundle.name

        self.label = BundlePDS4Label(mission, self)

        return

    def write_product(self):

        if not os.path.isfile(self.path):
            with open(self.path, "w+") as f:

                for line in fileinput.input(self.mission.root_dir+'/etc/template_readme.txt'):
                    if '$PDS4_MISSION_NAME' in line:
                        line = line.replace('$PDS4_MISSION_NAME', self.mission.name)

                    line = add_carriage_return(line)

                    f.write(line)


        return

class InventoryProduct(Product):

    def __init__(self, mission, collection):

        self.mission = mission
        self.collection = collection

        if mission.pds == '3':
            self.path = mission.bundle_directory + os.sep + 'INDEX' \
                        + os.sep + 'INDEX.TAB'

        elif mission.pds == '4':
            self.name = f'collection_{collection.name}_inventory_v{mission.version}.csv'
            self.path = mission.bundle_directory + os.sep + collection.name \
                        + os.sep + self.name
            self.path_current = mission.increment + os.sep + collection.name \
                        + os.sep + self.name

        self.lid = self.product_lid()
        self.vid = self.product_vid()

        #
        # Kernels are already generated products but Inventories are not.
        #
        self.write_product()
        Product.__init__(self)


        if mission.pds == '3':
            self.label = InventoryPDS3Label(mission, collection, self)
        elif mission.pds == '4':
            self.label = InventoryPDS4Label(mission, collection, self)

        return


    def product_lid(self):

        product_lid = \
            'urn:esa:psa:{}_spice:document:spiceds'.format(
                    self.mission.accronym)

        return product_lid


    def product_vid(self):

        return '{}.0'.format(int(self.mission.version))


    def write_product(self):

        #
        # PDS4 collection file generation
        #
        if self.mission.pds == '4':

            with open(self.path, "w+") as f:
                #
                # If there is an existing version we need to add the items from
                # the previous version as SECONDARY members
                #
                if self.mission.increment:

                    with open(self.path_current.replace(self.mission.version+'.csv',
                                                self.mission.current_version+'.csv'),
                              "r") as r:
                        for line in r:
                            if 'P,urn' in line:
                                line = line.replace('P,urn', 'S,urn')
                                line = add_carriage_return(line)
                                f.write(line)

                for file in self.collection.files:
                    if file.new_product:
                        line = '{},{}::{}\r\n'.format('P',
                            file.label.PRODUCT_LID, file.label.PRODUCT_VID)

                        line = add_carriage_return(line)
                        f.write(line)

        #
        # PDS3 INDEX file generation
        #
        if self.mission.pds == '3':

            current_index = list()
            kernel_list = list()
            kernel_directory_list = ['IK', 'FK', 'SCLK', 'LSK', 'PCK', 'CK',
                                     'SPK', 'DSK']
        #  Note that PCK was doubled here. This accounted for a spurious extra line
        #  in the INDEX.TAB.

            if self.mission.increment:
                existing_index = self.mission.increment + '/INDEX/INDEX.TAB'

                with open(existing_index, 'r') as f:

                    for line in f:

                        if line.strip() == '': break

                        current_index.append(
                                [line.split(',')[0].replace(' ', ''),
                                 line.split(',')[1].replace(' ', ''),
                                 line.split(',')[2],
                        # A DVal message is that the times in INDEX.TAB are invalid,
                        # which is because they contain only date, not time. Just adding noon
                        # time to all dates removes this error, however, for the time being, we decided
                        # to leave it as is.
                                 line.split(',')[3].replace('\n', '\r\n') ]
                                )
                        line = line.split(',')[1]
                        line = line[1:-1].rstrip()
                        kernel_list.append(line)


            new_index = []

            for directory in kernel_directory_list:

                data_files = self.mission.bundle_directory + '/DATA/' + directory

                for file in os.listdir(data_files):

                    if file.split('.')[1] != 'LBL' and file not in kernel_list:
                        new_label_element = '"DATA/' + directory + '/' + \
                                            file.split('.')[0] + '.LBL"'
                        new_kernel_element = '"' + file + '"'

                        generation_date = PDS3_label_gen_date(
                            data_files + '/' + file.split('.')[0] + '.LBL')
                        if not 'T' in generation_date:
                            generation_date = generation_date + '         '
                        # See above remark about DVal message.

                        new_index.append([new_label_element, new_kernel_element,
                                          generation_date,
                                          '"' + self.mission.dataset + '"\r\n'])

            #
            # We merge both lists
            #
            index = current_index + new_index

            #
            # We sort out which is the kernel that has the most characters
            # and we add blank spaces to the rest
            #
            lab_filenam_list = list()
            ker_filenam_list = list()

            for element in index:
                lab_filenam_list.append(element[0])
                ker_filenam_list.append(element[1])

            longest_lab_name = (max(lab_filenam_list, key=len))
            max_lab_name_len = len(longest_lab_name)

            longest_ker_name = (max(ker_filenam_list, key=len))
            max_ker_name_len = len(longest_ker_name)

            index_list = list()
            dates = []   # used to sort out according to generation date the index_list
            for element in index:
                blanks = max_lab_name_len - (len(element[0]))
                label = element[0][:-1] + ' ' * blanks + element[0][-1]

                blanks = max_ker_name_len - (len(element[1]))
                kernel = element[1][:-1] + ' ' * blanks + element[1][-1]
                if '\n' in element[-1]:
                    index_list.append(label + ',' + kernel + ',' + element[2] + ',' + element[3])
                else:
                    index_list.append(label + ',' + kernel + ',' + element[2] + ',' + element[3] + '\n')
                dates.append(element[2])
            with open(self.mission.bundle_directory + '/INDEX/INDEX.TAB', 'w+') as f:
                for element in [x for _,x in sorted(zip(dates, index_list))]: f.write(element)

        return


class SpicedsProduct(object):

    def __init__(self, mission, collection):

        self.new_product = True
        self.mission = mission
        #
        # We obtain the previous spiceds file if it exists
        #
        path = mission.bundle_directory + os.sep + collection.name
        if self.mission.increment:
            spiceds_files = glob.glob(path + os.sep + 'spiceds_v*.html')
            spiceds_files.sort()
            latest_spiceds = spiceds_files[-1]
            latest_version = latest_spiceds.split('_v')[-1].split('.')[0]
            self.latest_spiceds = latest_spiceds
            self.latest_version = latest_version
            version = int(latest_version) + 1
        else:
            version = 1
            self.latest_spiceds = ''

        self.name = 'spiceds_v{0:0=3d}.html'.format(version)
        self.path = mission.bundle_directory + os.sep + collection.name \
                    + os.sep + self.name
        self.template = mission.root_dir + '/etc/template_spiceds.html'
        self.mission = mission

        self.lid = self.product_lid()
        self.vid = self.product_vid()

        #
        # We provide the values as if it was a label
        #
        self.PRODUCT_CREATION_TIME = current_time()
        self.PDS4_MISSION_NAME = mission.name
        self.PDS4_SPACECRAFT_NAME = mission.spacecraft
        self.ERRATA = read_errata(mission.errata)
        self.MISSION_ACCRONYM = mission.accronym
        self.PRODUCER_NAME = mission.producer_name
        self.PRODUCER_EMAIL = mission.producer_email
        self.PRODUCER_PHONE = mission.producer_phone

        self.SPK_AAREADME = self.read_readme(
            mission.kernels_directory + '/spk/aareadme.txt')
        self.PCK_AAREADME = self.read_readme(
            mission.kernels_directory + '/pck/aareadme.txt')
        self.CK_AAREADME = self.read_readme(
            mission.kernels_directory + '/ck/aareadme.txt')
        self.DSK_AAREADME = self.read_readme(
            mission.kernels_directory + '/dsk/aareadme.txt')
        self.LSK_AAREADME = self.read_readme(
            mission.kernels_directory + '/lsk/aareadme.txt')
        self.FK_AAREADME = self.read_readme(
            mission.kernels_directory + '/fk/aareadme.txt')
        self.IK_AAREADME = self.read_readme(
            mission.kernels_directory + '/ik/aareadme.txt')
        self.SCLK_AAREADME = self.read_readme(
            mission.kernels_directory + '/sclk/aareadme.txt')


        self.generated = self.write_product()
        if not self.generated:
            return

        #
        # Kernels are already generated products but Inventories are not.
        #
        Product.__init__(self)

        self.label = DocumentPDS4Label(mission, collection, self)

        return


    def product_lid(self):

        product_lid = \
            'urn:esa:psa:{}_spice:document:spiceds'.format(
                    self.mission.accronym)

        return product_lid

    def product_vid(self):

        return '{}.0'.format(int(self.mission.version))


    def write_product(self):

        with open(self.path, "w+") as f:

            spiceds_dictionary = vars(self)

            for line in fileinput.input(self.template):
                for key, value in spiceds_dictionary.items():
                    if isinstance(value, str) and key in line and '$' in line:
                        line = line.replace(key, value)
                        line = line.replace('$', '')

                line = add_carriage_return(line)

                f.write(line)

        #
        # If the previous spiceds document is the same then it does not
        # need to be generated.
        #
        generate_spiceds = True
        if self.latest_spiceds:
            with open(self.path) as f:
                spiceds_current = f.readlines()
            with open(self.latest_spiceds) as f:
                spiceds_latest = f.readlines()

            differ = difflib.Differ(charjunk=difflib.IS_CHARACTER_JUNK)
            diff = list(differ.compare(spiceds_current, spiceds_latest))

            generate_spiceds = False
            for line in diff:
                if line[0] == '-':
                    if 'Last update' not in line and line.strip() != '-' and line.strip() != '-\n':
                        generate_spiceds = True

            if not generate_spiceds:
                os.remove(self.path)
                logging.warning('spiceds document does not need to be updated')
                logging.warning('')

        return generate_spiceds


    def read_readme(self, path):

        with open(path, "r") as f:

            readme = ''
            write_flag = False
            for line in f:
                if 'Other directory contents' in line or \
                        'Current {} Kernels Set'.format(path.split('/')[-2].upper()) in line:
                    write_flag = False
                if write_flag and \
                        not '-------------------------------------------------' in line and \
                        not 'File naming conventions' in line:

                    readme += line
                if 'Brief Summary' in line:
                    write_flag = True
                    readme += '{} Files \n'.format(path.split('/')[-2].upper())

        return readme


class KernelProduct(Product):

    def __init__(self, path, mission, label_only=False):

        self.path = path
        self.mission = mission
        self.name = path.split(os.sep)[-1]
        self.extension = path.split('.')[-1] # '-' added by BG.

        self.type = extension2type(self)
        self.coverage()


        if self.extension[0].lower() == 'b':
            if self.mission.pds == '3':
                self.file_format = 'BINARY'
                self.record_type = 'FIXED_LENGTH\r\nRECORD_BYTES                 = 1024'
#                self.record_type = 'UNDEFINED'
            else:
                self.file_format = 'Binary'
        else:
            if self.mission.pds == '3':
                self.file_format = 'ASCII'
                self.record_type = 'STREAM'
            else:
                self.file_format = 'Character'

        if mission.pds == '3':
            self.description = self.PDS3description()
            self.kernel_mission_phase()
#            check = open( 'desc_pattern_check.txt', 'a' )
#            check.write( self.name + '\n' )
#            check.write( self.mission_phases + '\n\n' )
            self.pds3type = extension2PDS3type(self)
            self.collection_path = mission.bundle_directory + os.sep + \
                                   'DATA' + os.sep
            self.producer_id = self.producer()
            self.product_version_type = 'ACTUAL'

            # Previously, PDSLabel took its PDS4_SPACECRAFT_NAME directly from
            # mission.spacecraft. Now it takes it from
            # product.PDS4_SPACECRAFT_NAME. So we can change it here for PDS3
            # CATT and DSK file labels, and otheres where needed.
            if mission.spacecraft == 'ROSETTA-ORBITER' and \
            ( self.name[0:4] == 'CATT' or self.extension[0:3] == 'BDS' or \
              ( self.name[0:5] == 'ROS_V' or
                self.name[0:16] == 'ROS_DSK_SURFACES' )
              and self.extension[0:2] == 'TF' or
              self.name[0:6] == 'EARTH_' and self.extension[0:3] == 'BPC' or
              self.name[0:4] == 'CORB' or self.name[0:12] == 'OUTERPLANETS' ) :
                self.PDS4_SPACECRAFT_NAME = \
                '{ "ROSETTA-ORBITER", "ROSETTA-LANDER" }'
            elif mission.spacecraft == 'ROSETTA-ORBITER' and \
            ( self.name[0:4] == 'LATT' or self.name[0:9] == 'ROS_DIM_V' or
              self.name[0:6] == 'LANDER' or self.name[0:4] == 'LORB' or
              'PHILAE' in self.name ):
                self.PDS4_SPACECRAFT_NAME = '"ROSETTA-LANDER"'
            else:
                self.PDS4_SPACECRAFT_NAME = '"' + mission.spacecraft + '"'

            # TARGET_NAME was first set above together with
            # PDS4_SPACECRAFT_NAME (which ends up as INSTRUMENT_HOST_NAME),
            # but we need to treat its conditions seperately
            if mission.spacecraft == 'ROSETTA-ORBITER':
                if 'DEIMOS' in self.name and self.extension[0:3] == 'BDS' :
                    self.target_name = 'M2 DEIMOS'
                elif 'PHOBOS' in self.name and self.extension[0:3] == 'BDS' :
                    self.target_name = 'M1 PHOBOS'
                elif 'TEMPEL1' in self.name and self.extension[0:3] == 'BDS' :
                    self.target_name = '9P/TEMPEL 1 (1867 G1)'
                elif 'T7_LINEAR' in self.name and self.extension[0:3] == 'BSP' :
                    self.target_name = 'C/LINEAR (2002 T7)'
                elif 'A2_LINEAR' in self.name and self.extension[0:3] == 'BSP' :
                    self.target_name = '354P/LINEAR 58 (2010 A2)'
                elif 'MACHHOLZ' in self.name and self.extension[0:3] == 'BSP' :
                    self.target_name = 'C/MACHHOLZ (2004 Q2)'
                elif 'VESTA' in self.name and self.extension[0:3] == 'BSP' :
                    self.target_name = '4 VESTA'
                elif self.name[0:4] == 'CATT' or \
                     self.extension[0:3] == 'BDS' or \
                     self.name[0:4] == 'CORB' :
#                     self.name[0:5] == 'ROS_V' and \
#                     self.extension[0:2] == 'TF' or \
#  Boris wanted "N/A"in frames kernels.
                    self.target_name = '67P/CHURYUMOV-GERASIMENKO 1 (1969 R1)'
                else :
                    self.target_name = 'N/A'
            else :
                self.target_name = 'N/A'

            #
            # PDS3 CK specific fields.
            #
            self.naif_instrument_id = ''

            if self.type.upper() == 'CK':
                ids = spiceypy.ckobj(path)

                naif_instrument_id = list()
                for id in ids:
                    naif_instrument_id.append(str(id))

            elif self.type.upper() == 'DSK':
                self.start_time = '"N/A"'
                self.stop_time = '"N/A"'
                self.naif_instrument_id = '"N/A"'

            elif self.type.upper() ==  "FK":
                self.start_time = '"N/A"'
                self.stop_time = '"N/A"'
                self.naif_instrument_id = '"N/A"'

            # elif self.type.upper() ==  "LSK":
            # Not required for Rosetta for the time being

            elif self.type.upper() == "PCK" and self.extension.upper() == 'TPC':

                self.start_time = '"N/A"'
                self.stop_time = '"N/A"'
                self.naif_instrument_id = '"N/A"'

            elif self.type.upper() == "IK":

                ids = self.ik_kernel_ids(path)
                naif_instrument_id = ids[0]
                if mission.spacecraft == 'ROSETTA-ORBITER':
                    self.start_time = '2004-03-02T07:17:51'
                else:
                    self.start_time = '"N/A"'
                self.stop_time = '"N/A"'
#                self.description = "SPICE kernel containing instrument information\n" \
#                                   "                                for {}".format(self.name.split('_')[-2])
#  This does not put a appropriate line ending, therefore it caught Boris'
#  attention. Moreorver, it overwrites the description obtained from the
#  aareadme.txt. As we get here only in the PDS3 case, I can safely just
#  take it out.

            elif self.type.upper() == "SCLK":

                self.stop_time = '"N/A"'
                self.naif_instrument_id = '"N/A"'

#  Getting the initial epoch of an SCLK by sclk_coverage() goes wrong if
#  the first partition does not start with zero. For SCLK, we set the
#  START_TIME to a value suggested by Boris Semenov:
                if self.name == 'LANDER_170904_STEP.TSC' or \
                   self.name == 'ROS_160929_STEP.TSC' :
                    self.start_time = '2004-03-02T07:17:51'

            elif self.type.upper() == "SPK":

                self.naif_instrument_id = '"N/A"'

            else:
                # Here we get only in the cases of an LSK or a binary PCK.
                # For all these cases, times were overwritten with N/A, but
                # we don't want this for a binary PCK, thus:
                if self.extension.upper() !=  "BPC":
                    self.start_time = '"N/A"'
                    self.stop_time = '"N/A"'
                self.naif_instrument_id = '"N/A"'


            if not self.naif_instrument_id:

                if len(naif_instrument_id) == 1:
                    self.naif_instrument_id = naif_instrument_id[0]

                else:
                    naif_instrument_id_str = '{\r\n'

                    nline = len( naif_instrument_id )
                    iline = 0
                    for line in naif_instrument_id:
                        iline = iline + 1
                        if iline < nline:
                            naif_instrument_id_str += '                                {},\r\n'.format(line)
                        else:
                            naif_instrument_id_str += '                                {}\r\n'.format(line)
                    naif_instrument_id_str += '                               }\r\n'

                    self.naif_instrument_id = naif_instrument_id_str

            #
            # The version type of data contained in the kernel is extracted
            # from the kernel description. If it contains the keyword
            # 'predicted' then it is PREDICT, otherwise is considered as ACTUAL
            for line in self.description.splitlines():
                if 'predicted' in line.lower():
                    self.product_version_type = 'PREDICT'
                    break
                else:
                    self.product_version_type = 'ACTUAL'

        elif mission.pds == '4':
            self.description = self.PDS4description()
            self.collection_path = mission.bundle_directory + os.sep + \
                                   'spice_kernels' + os.sep

        if label_only:
            product_path = self.path
        else:
            product_path = self.collection_path + self.type + os.sep

        self.lid = self.product_lid()
        self.vid = self.product_vid()

        #
        # We generate the kernel directory if not present
        #
        if not label_only:
            if mission.pds == '3':
                if 'orb' in product_path:
                    product_path = self.mission.bundle_directory + '/EXTRAS/ORBNUM/'
                else:
                    product_path = product_path.split('/')
                    product_path[-2] = product_path[-2].upper()
                    product_path = '/'.join(product_path)
            safe_make_directory(product_path)

        #
        # We copy the kernel to the directory. If the kernel is already
        # present we skip
        #
        if not label_only:
            if not os.path.isfile(product_path + self.name):
                shutil.copy2(self.path, product_path + self.name)
                self.new_product = True
            else:
                logging.warning('{} already present in Bundle'.format(self.name))
                self.new_product = False
                return

            #
            # We update the path after having copied the kernel
            #
            self.path = product_path + self.name

        Product.__init__(self)

        self.label = KernelPDSLabel(mission, self)

        #
        # Insert PDS3 label inside the kernel
        #
        if self.mission.accronym == 'MEX' or self.mission.accronym == 'VEX':
            #
            # insert label at the comment of binary kernels
            #
            if self.name[-2:] == 'BC' or self.name[-3:] == 'BPC' or self.name[-3:] == 'BSP' or self.name[-3:] == 'BDS':
                print('inserting label in binary kernel comment')
                commntlog = os.popen('commnt -e ' + product_path + self.name + ' ' + product_path + 'cmt.txt').readlines()
                if not commntlog:
                    commentexist = True
                else:
                    os.system('commnt -d ' + product_path + self.name)
                    commentexist = False   # this implies the kernel did not have a comment
                file = open(product_path + self.name.split('.')[0] + '.LBL', "r+")   # open label
                label_lines = file.readlines()
                file = open(product_path + 'cmt.txt', "r+")   # open comment
                comment_lines = file.readlines()
                addlabel = True
                for line in comment_lines:
                    if 'beginlabel' in line:
                        addlabel = False
                        print('kernel comment already contained a label!')
                if addlabel:
                    with open(product_path + 'cmt.txt', 'w+') as f:  # write label and comment together
                        f.write("%s" % '\n')
                        f.write("%s" % '\\beginlabel \n')
                        f.write("%s" % '\n')
                        for item in label_lines:
                            f.write("%s" % item)
                        f.write("%s" % '\n')
                        f.write("%s" % '\\endlabel \n')
                        f.write("%s" % '\n')
                        if commentexist:
                            for item in comment_lines:
                                if not 'KPL' in item:
                                    f.write("%s" % item)
                    #
                    # transform LTL-IEEE binary kernels to BIG-IEEE format
                    #
                    commntlog = os.popen('commnt -d ' + product_path + self.name).readlines()
                    if commntlog and 'UNSUPPORTEDBFF' in commntlog[5]:   # check binary kernel BIG-IEEE format
                        os.system('bingo -ieee2pc ' + product_path + self.name + ' ' + product_path + self.name.split('.')[
                            0] + 'bin.' + self.name.split('.')[1])   # transform from ieee to pc binary to be modified
                        os.system('commnt -d ' + product_path + self.name.split('.')[0] +
                                  'bin.' + self.name.split('.')[1])   # delete comment of transformed file
                        os.system('commnt -a ' + product_path + self.name.split('.')[0] +
                                  'bin.' + self.name.split('.')[1] + ' ' + product_path + 'cmt.txt')   # add new comment
                        os.remove(product_path + self.name)   # remove original file
                        os.system('bingo -pc2ieee ' + product_path + self.name.split('.')[0] +
                                  'bin.' + self.name.split('.')[1] + ' ' + product_path + self.name)   # create final file
                        os.remove(product_path + self.name.split('.')[0] + 'bin.' + self.name.split('.')[1])
                    elif not commntlog:   # for LTL-IEEE binary format
                        os.system('commnt -a ' + product_path + self.name + ' ' + product_path + 'cmt.txt')
                        os.system('bingo -pc2ieee ' + product_path + self.name + ' ' + product_path + self.name.split('.')[
                            0] + 'bin.' + self.name.split('.')[1])  # transform from pc binary to ieee
                        os.remove(product_path + self.name)  # remove original file
                        os.rename(product_path + self.name.split('.')[0] +
                                  'bin.' + self.name.split('.')[1], product_path + self.name)   # rename final file
                os.remove(product_path + 'cmt.txt')

            #
            # insert label at the top of text kernels
            #
            elif self.name[-2:] == 'TF' or self.name[-2:] == 'TI' or self.name[-3:] == 'TLS' \
                    or self.name[-3:] == 'ORB' or self.name[-3:] == 'TPC' or self.name[-3:] == 'TSC':
                print('inserting label in text kernel')
                file = open(product_path + self.name.split('.')[0] + '.LBL', "r+")   # open label
                label_lines = file.readlines()
                file = open(product_path + self.name, "r+")   # open kernel
                kernel_lines = file.readlines()
                with open(product_path + self.name, 'w+') as f:   # write label and kernel together
                    f.write("%s" % 'KPL/' + self.type.upper() + '\n')
                    f.write("%s" % '\n')
                    f.write("%s" % '\\beginlabel \n')
                    f.write("%s" % '\n')
                    for item in label_lines:
                        f.write("%s" % item)
                    f.write("%s" % '\n')
                    f.write("%s" % '\\endlabel \n')
                    f.write("%s" % '\n')
                    for item in kernel_lines:
                        if not 'KPL' in item:
                            f.write("%s" % item)
            return


    def product_lid(self):

        product_lid = \
            'urn:esa:psa:{}_spice:spice_kernels:{}_{}'.format(
                    self.mission.accronym,
                    self.type,
                    self.name)

        return product_lid

    def product_vid(self):

        product_vid = '1.0'


        return product_vid


    # PDS3 specific
    def producer(self):

        # if self.extension.upper() != 'PCK' and self.extension.upper() != 'DSK':
        if self.extension.upper() != 'TPC' and (self.name[0:6] != 'EARTH_' or self.extension[0:3] != 'BPC' ) \
        or self.name == 'PHOBOS_M157_GAS_V01.BDS':
            producer = 'ESA-ESAC'
        else:
            producer = 'JPL'


        return producer


    def coverage(self):
        if self.type.lower() == 'spk':
            (self.start_time, self.stop_time) = self.spk_coverage()
        elif self.type.lower() == 'ck':
            (self.start_time, self.stop_time) = self.ck_coverage()
        elif self.extension.lower() == 'bpc':
            (self.start_time, self.stop_time) = self.pck_coverage()
        elif self.type.lower() == 'sclk':
            (self.start_time, self.stop_time) = self.sclk_coverage()
#            print( self.name )
#            print( self.start_time )
#            print( self.stop_time )
#            exit()
        else:
            self.start_time = self.mission.start
            self.stop_time = self.mission.stop


    def spk_coverage(self):

        ids = spiceypy.spkobj(self.path)

        MAXIV = 1000
        WINSIZ = 2 * MAXIV
        TIMLEN = 62

        coverage = spiceypy.support_types.SPICEDOUBLE_CELL(WINSIZ)

        start_points_list = list()
        end_points_list = list()

        for id in ids:

            spiceypy.scard, 0, coverage
            spiceypy.spkcov(spk=self.path, idcode=id, cover=coverage)

            num_inter = spiceypy.wncard(coverage)

            for i in range(0, num_inter):
                endpoints = spiceypy.wnfetd(coverage, i)

                start_points_list.append(endpoints[0])
                end_points_list.append(endpoints[1])

                time_str = spiceypy.timout(endpoints,
                                           "YYYY-MM-DDTHR:MN:SC.###::UTC",
                                           TIMLEN)
        start_time = min(start_points_list)
        stop_time = max(end_points_list)

        start_time_cal = spiceypy.timout(start_time,
                                         "YYYY-MM-DDTHR:MN:SC.###::UTC", TIMLEN)
        stop_time_cal = spiceypy.timout(stop_time,
                                        "YYYY-MM-DDTHR:MN:SC.###::UTC", TIMLEN)

        dt_start, micro = start_time_cal.split('.')
        start_time_cal = "%s.%03d" % (dt_start, int(micro) / 1000) + 'Z'

        dt_stop, micro = stop_time_cal.split('.')
        stop_time_cal = "%s.%03d" % (dt_stop, int(micro) / 1000) + 'Z'

        return [start_time_cal, stop_time_cal]


    def ck_coverage(self):

        start_points_list = list()
        end_points_list = list()

        MAXIV = 10000
        WINSIZ = 2 * MAXIV
        TIMLEN = 500
        MAXOBJ = 10000

        ids = spiceypy.support_types.SPICEINT_CELL(MAXOBJ)
        coverage = spiceypy.support_types.SPICEDOUBLE_CELL(WINSIZ)

        ids = spiceypy.ckobj(ck=self.path, outCell=ids)

        num_ids = spiceypy.card(ids)
        for id in ids:

            coverage = spiceypy.support_types.SPICEDOUBLE_CELL(WINSIZ)
            spiceypy.scard, 0, coverage
            coverage = spiceypy.ckcov(ck=self.path, idcode=id, needav=False,
                                    level='SEGMENT', tol=0.0, timsys='TDB',
                                    cover=coverage)

            num_inter = spiceypy.wncard(coverage)

            for j in range(0, num_inter):
                endpoints = spiceypy.wnfetd(coverage, j)

                start_points_list.append(endpoints[0])
                end_points_list.append(endpoints[1])

                time_str = spiceypy.timout(endpoints,
                                         "YYYY-MM-DDTHR:MN:SC.###::UTC", TIMLEN)

        start_time = min(start_points_list)
        stop_time = max(end_points_list)

        start_time_cal = spiceypy.timout(start_time,
                                       "YYYY-MM-DDTHR:MN:SC.###::UTC", TIMLEN) + 'Z'
        stop_time_cal = spiceypy.timout(stop_time, "YYYY-MM-DDTHR:MN:SC.###::UTC",
                                      TIMLEN) + 'Z'

        return [start_time_cal, stop_time_cal]

    #
    # PCK kernel processing
    #

    def pck_coverage(self):

        MAXIV = 1000
        WINSIZ = 2 * MAXIV
        TIMLEN = 62
        MAXOBJ = 1000

        ids = spiceypy.support_types.SPICEINT_CELL(MAXOBJ)

        spiceypy.pckfrm(self.path, ids)

        coverage = spiceypy.support_types.SPICEDOUBLE_CELL(WINSIZ)

        start_points_list = list()
        end_points_list = list()

        for id in ids:

            spiceypy.scard, 0, coverage
            spiceypy.pckcov(pck=self.path, idcode=id, cover=coverage)

            num_inter = spiceypy.wncard(coverage)

            for i in range(0, num_inter):
                endpoints = spiceypy.wnfetd(coverage, i)

                start_points_list.append(endpoints[0])
                end_points_list.append(endpoints[1])

                time_str = spiceypy.timout(endpoints,
                                         "YYYY-MM-DDTHR:MN:SC.###::UTC", TIMLEN)

        start_time_tbd = min(start_points_list)
        stop_time_tbd = max(end_points_list)

        start_time_cal = spiceypy.timout(start_time_tbd,
                                       "YYYY-MM-DDTHR:MN:SC.###::UTC", TIMLEN) + 'Z'
        stop_time_cal = spiceypy.timout(stop_time_tbd,
                                      "YYYY-MM-DDTHR:MN:SC.###::UTC", TIMLEN) + 'Z'


        return [start_time_cal, stop_time_cal]

    #
    # SCLK kernel processing
    #

    def sclk_coverage(self):

        TIMLEN = 62

        spiceypy.furnsh(self.path)

        with open(self.path, "r") as f:

            for line in f:

                if 'SCLK_DATA_TYPE_' in line:
                    line = line.lstrip()
                    line = line.split("SCLK_DATA_TYPE_")
                    line = line[1].split("=")[0]
                    id = int('-' + line.rstrip())

        partitions = spiceypy.scpart(id)

        #
        # We need to take into account clocks that have more than one partition
        #
        if len(partitions[0]) > 1.0:
            partitions = partitions[0]
        start_time_tdb = spiceypy.sct2e(id, partitions[0])
        if isinstance(start_time_tdb, list):
            start_time_tdb = start_time_tdb[0]
        start_time_cal = spiceypy.timout(start_time_tdb,
                                       "YYYY-MM-DDTHR:MN:SC.###::UTC", TIMLEN) + 'Z'

        stop_time_cal = self.mission.stop

        return [start_time_cal, stop_time_cal]

    #
    # IK kernel processing
    #
    def ik_kernel_ids(self, path):

        with open(path, "r") as f:

            id_list = list()
            parse_bool = False

            for line in f:

                if 'begindata' in line:
                    parse_bool = True

                if 'begintext' in line:
                    parse_bool = False

                if 'INS-' in line and parse_bool == True:
                    line = line.lstrip()
                    line = line.split("_")
                    id_list.append(line[0][3:])

        id_list = list(set(id_list))

        return [id_list]

    #
    # kernel Info processing
    #
    def PDS4description(self):

        aareadme_file = self.path.split(self.name)[0] + 'aareadme.txt'
        description = ''
        description_bool = False
        description_section_bool = False
        previous_line = 'Initial assignment added by BG.'

        for line in fileinput.input(aareadme_file):
            line = line.rstrip()

            #
            # We need to have reached the Current KER_TYPE kernels Type
            #
            if 'Current' in line:
                description_section_bool = True

            if '.'+self.extension.lower() in line.lower() and description_section_bool:

                schema = ''
                for letter in line.strip():
                    if letter.islower() or letter.isdecimal() or letter == '_':
                        schema += letter
                    else:
                        break

                self.schema = schema

                if self.schema in self.name:
                    description_bool = True

            if 'Other directory contents' in line:
                description_section_bool = False
            if line.strip() == '' and previous_line.strip() == '':
                description_bool = False

            if description_bool and description_section_bool:
                if not self.schema in line:
                    description += line.strip() + ' '

            previous_line = line

        return description.rstrip().lstrip()


    def PDS3description(self):

        #  We construct the full path to the aareadme similarly to the old version,
        #  cf. below. Note that the aareadme should ideally be read only once when
        #  looping through the kernels directories.
        aareadme_file = self.path.split(self.name)[0] + 'aareadme.txt'
        with open(aareadme_file, "r") as f:

        #  We want to start reading the descriptions when we have reached their
        #  section to reduce the risk of misinterpreting some other text. The
        #  underlining of `Name' and `Comments' seem to be a unique indicator
        #  of the description section. So we overread everything until we find it:

            if self.mission.accronym == 'ROSETTA':
                iline = 0
                for line in f:
                    iline = iline + 1
                    try:
                        if '-----------------' in line.split()[0] and \
                           '------------------------------' in line.split()[1]:
                            break
                    except:
                        pass
#                print( line )
            if self.mission.accronym == 'VEX' or self.mission.accronym == 'MEX':
                iline = 0
                for line in f:
                    iline = iline + 1
                    try:
                        if 'Directory Structure' in line:
                            break
                    except:
                        pass
                        #                print( line )

            #  Now we read the descriptions. There can be empty lines and also extra lines
            #  with running text in between. The actual description lines can contain
            #  only a file name (there can be multiple kernels for one description, e.g.,
            #  for DSKs) only descrition text (text is usually more than one line), or
            #  both (file name and first line of description). So, we resume reading:
            names = list()
            name  = ''
            descs = list()
            desc  = ''
            for line in f:

                #  We start with checking for the case that there is only a file name.
                if ' ' not in line.strip() and '.' in line.strip() and line[0] == ' ' and line[0:24] != '                        ' :
                    if desc != '':
                        descs.append( desc )
                        desc = ''
                    name = name + line.strip() + ' '

                #  Now we treat the case that we have file name and first line of description.
                elif len( line.split() ) > 0 and '.' in line.split()[0] and line[0] == ' ' and line[0:24] != '                        ' :
                    if desc != '':
                        descs.append( desc )
                        desc = ''
                    name = line.split()[0] + ' '
                    names.append( name )
                    name = ''
                    desc = desc + line.strip().split(' ',1)[1].strip() + '\r\n'

                #  Finally the case with only a continuation like of description.
                elif line.strip() != '' and line[0:24] == '                        ' :
                    desc = desc + '                               ' + line.strip() + '\r\n'
                    if name != '' and len(names) and names[-1] != name:
                        names.append(name)
            descs.append( desc )

#  This seems to work so far, at least for CK and DSK. Now we loop through the
#  list of file names and count the matching letters.

        iline = -1
        nmax = 0
        maxline = iline
        maxfn = iline
        for string in names:
            iline = iline + 1
            ifn = -1
            for fname in string.split():
                ifn += 1
#                    print( self.name )
#                    print( fname )
                iletter = -1
                nletter = 0
                for letter in self.name:
                    iletter = iletter + 1
                    if iletter < len( fname ) and letter == fname[iletter]:
                        nletter = nletter + 1
#                    print( nletter )
                if nletter > nmax:
                    nmax = nletter
                    maxline = iline
                    maxfn   = ifn

#            if '.BSP' in self.name :
#                print( string )
#                print( descs[iline] )
#        if '.BSP' in self.name :
#            print( len( names ), len( descs ) )
#            exit()

#  Now we know the best match.
        check = open( 'desc_pattern_check.txt', 'a' )
        check.write( self.name + '\n' )
        check.write( names[maxline].split()[maxfn] + '\n\n' )
#        check.write( names[maxline].split()[maxfn] + '\n' )
#        check.write( descs[maxline] + '\n\n' )
#        print( self.name )
#        print( names[maxline].split()[maxfn] )

#                print( descs[iline] )

#        exit()

#  So we copy the respective description to the return value.
        kernel_info = descs[maxline]

        return kernel_info[0:-2]


    def PDS3descriptionOFF(self):

#  We switch this off by changing the name (some of the reasons explained
#  below) and start from scratch above.

        kernel_name_map = ''

        aareadme_file = self.path.split(self.name)[0] + 'aareadme.txt'

        char_bool = False
        for char in self.name:

            if char == '_':
                char_bool = True

            if char.isdigit() or char == ".":
                if char_bool:
                    break
                else:
                    kernel_name_map += char
            else:
                kernel_name_map += char
#  This stops the kernel_name_map at the first digit that appears somewhere
#  after a _ has appeared previously. This is nonsense and goes definitely
#  wrong for DSK names like ROS_CG_K006_OSPCLPS_N_V2.LBL. What we really
#  want is to exclude the version number. Moreover, for CKs, there are
#  placeholders in the aareadme, which is not accounted for here, so all
#  the CK descriptions are also most probable wrong.

#  It is inefficient to re-read the aareadme for every single kernel file,
#  however, we have to get it right in the first place, so, for the time
#  being, we leave it here.
        with open(aareadme_file, "r") as f:

            kernel_info_list = list()
            parsing_bool = 'False'
            last_line_bool = 'False'
            blank_line = 'False'
            some_line_parsed_bool = 'False'

            for line in f:

                if parsing_bool == True and some_line_parsed_bool == True and line.strip() != '':
                    description_text = line
                    description_text = description_text.lstrip()
                    description_text = description_text.rstrip()

                    kernel_info_list.append(description_text)

                    some_line_parsed_bool = True

                if 'Current ' + self.type.upper() + ' Kernels Set' in line:
                    parsing_bool = True

                if kernel_name_map and kernel_name_map in line and parsing_bool == True:
                    description_text = line.split(self.extension.upper())[1]
                    description_text = description_text.lstrip()
                    description_text = description_text.rstrip()

                    kernel_info_list.append(description_text)

                    some_line_parsed_bool = True

                # This needs to be put before the next if statement

                if some_line_parsed_bool == True and blank_line == True and line.strip() == '':
                    break

                if line.strip() == '':
                    blank_line = True

        kernel_info = ''
        first = True
        for line in kernel_info_list:
            if line.strip() != '' \
            and ( not '.'+self.extension in line \
            or ' ' in line.strip() ):
                if not first:
                    kernel_info += '                               '
                    kernel_info += line+'\r\n'
                else:
                    separating_blanks_bool = False
                    in_description_text_bool = False
                    for letter in line:
                        if letter == ' ':
                            separating_blanks_bool = True
                        if separating_blanks_bool and letter != ' ':
                            in_description_text_bool = True
                        if in_description_text_bool:
                            kernel_info += letter
                    kernel_info += '\r\n'
                first = False
        kernel_info = kernel_info[:-2]

#        print( kernel_info[0:-1] )
        return kernel_info[0:-1]


    def kernel_mission_phase(self):

        start_time = self.start_time[0:-1]   # avoid reading the Z at the end of UTC tag
        stop_time = self.stop_time[0:-1]

        mission_phase_map = self.mission.root_dir + \
                            '/etc/mission_phases_{}.txt'.format(self.mission.accronym.lower())


        with open(mission_phase_map, 'r') as f:
            mission_phases = list()

            if start_time == 'N/A' and stop_time == 'N/A':

                for line in f:
                    mission_phases.append(line.split('   ')[0])

            if start_time != 'N/A' and stop_time == 'N/A':

                next_phases_bool = False
                start_tdb = spiceypy.str2et(start_time)  # + ' TDB')

                for line in f:
                    start_phase_date = line.rstrip().split(' ')[-2]
                    start_phase_tdb = spiceypy.str2et(start_phase_date)

                    stop_phase_date = line.rstrip().split(' ')[-1]
                    stop_phase_tdb = spiceypy.str2et(stop_phase_date)

                    if next_phases_bool == True:
                        mission_phases.append(line.split('   ')[0])

                    elif start_tdb >= start_phase_tdb:
                        mission_phases.append(line.split('   ')[0])
                        next_phases_bool = True

            if start_time != 'N/A' and stop_time != 'N/A':

                next_phases_bool = False
                start_tdb = spiceypy.str2et(start_time)  # + ' TDB')
                stop_tdb = spiceypy.str2et(stop_time)  # + ' TDB')

                for line in f:

                    start_phase_date = line.rstrip().split(' ')[-2]
                    start_phase_tdb = spiceypy.str2et(
                        start_phase_date)  # + ' TDB')

                    stop_phase_date = line.rstrip().split(' ')[-1]
                    stop_phase_tdb = spiceypy.str2et(stop_phase_date)
                    stop_phase_tdb += 86400
                    #  One day added to account for gap between phases.

                    if next_phases_bool == True and start_phase_tdb >= stop_tdb:
                        break

                    if next_phases_bool == True:
                        mission_phases.append(line.split('   ')[0])

                    if start_phase_tdb <= start_tdb <= stop_phase_tdb:
                        mission_phases.append(line.split('   ')[0])
                        next_phases_bool = True

        mission_phases_str = ''
        for phase in mission_phases:
            mission_phases_str += '                                 ' + phase + ',\r\n'

        mission_phases_str = mission_phases_str[0:-3]

        self.mission_phases = mission_phases_str

        return


class MetaKernelProduct(Product):

    def __init__(self, mission, spice_kernels_collection='', product=False):
        '''

        :param mission:
        :param spice_kernels_collection:
        :param product: We can input a meta-kernel such that the meta-kernel does not have to be generated
        '''
        self.new_product = True
        self.template = mission.root_dir + '/etc/template_metakernel.tm'
        self.path = self.template
        self.mission = mission
        self.name = self.path.split(os.sep)[-1]
        self.extension = self.path.split('.')[1]
        if spice_kernels_collection:
            self.spice_kernels_collection = spice_kernels_collection

        self.type = extension2type(self)

        if int( str( mission.pds ) ) == 3:
            self.collection_path = mission.bundle_directory + os.sep + \
                                   'EXTRAS' + os.sep
        elif int( str( mission.pds ) ) == 4:
            self.collection_path = mission.bundle_directory + os.sep + \
                                   'spice_kernels' + os.sep

        if self.mission.pds == '3':
            product_path = self.collection_path + self.type.upper() + os.sep
        else:
            product_path = self.collection_path + self.type + os.sep
        self.start_time = self.mission.start
        self.stop_time = self.mission.increment_stop
        self.description = 'SPICE MK file listing kernels for the complete mission, created by the ESA SPICE Service.'

        self.PDS4_MISSION_NAME = self.mission.name
        self.CURRENT_DATE = current_time().split('T')[0]

        if self.mission.pds == '3':
            self.file_format = 'ASCII'
            self.kernel_mission_phase()
            self.pds3type = extension2PDS3type(self)
            self.producer_id = self.producer()
            self.product_version_type = 'ACTUAL'
            self.naif_instrument_id = ''
            self.record_type = 'STREAM'
            self.target_name = self.mission.spacecraft

        #
        # We generate the kernel directory if not present
        #
        safe_make_directory(product_path)

        #
        # We name the metakernel
        #
        if product:
            self.name = product.split(os.sep)[-1]
            self.path = product
        else:
            if self.mission.pds == '3':
                #
                # check metakernels in previous increment
                #
                mk_version = 0
                if mission.increment:
                    for file in os.listdir(mission.increment + '/EXTRAS/MK'):
                        if '.TM' in file:
                            if int(file.split('_V')[1][0:2]) > mk_version: mk_version = int(file.split('_V')[1][0:2])
                            shutil.copyfile(mission.increment + '/EXTRAS/MK/' + file, product_path + file)
                    if mk_version < 9: mk_version = '0' + str(mk_version + 1)
                    else: mk_version = str(mk_version + 1)
                    self.name = '{}_V{}.TM'.format(mission.accronym, mk_version)
                else:
                    self.name = '{}_V{}.TM'.format(mission.accronym, mission.version)
            else:
                self.name = '{}_v{}.tm'.format(mission.accronym, mission.version)
            self.path = product_path + self.name
        self.FILE_NAME = self.name

        self.lid = self.product_lid()
        self.vid = self.product_vid()

        #
        # We generate the metakernel
        #
        if not product:
            self.write_product()
        else:
            #
            # If we don't generate the product then we need to read the kernels
            #
            self.spice_kernels_metakernel = mk2list(product)


        Product.__init__(self)

        if self.mission.pds == '4':
            self.label = MetaKernelPDS4Label(mission, self)

        return

    def product_lid(self):

        if self.type == 'mk':
            name = self.name.split('_v')[0]
        else:
            name = self.name

        product_lid = 'urn:esa:psa:{}_spice:spice_kernels:{}_{}'.format(
                        self.mission.accronym,
                        self.type,
                        name)


        return product_lid

    def product_vid(self):

        try:
            product_vid = str(int(self.name.split('.')[0][-1])) + '.0'
        except:
            logging.warning(f'{self.name} No vid explicit in kernel name: set to 1.0')
            product_vid = '1.0'

        return product_vid


    def write_product(self):

        from collections import OrderedDict

        #
        # We parse the meta-kernel grammar to obtain the kernel grammar.
        #
        with open(self.mission.mk_grammar, 'r') as kg:
            kernel_grammar_list = []

            for line in kg:

                if line.strip() != '':
                    line = line.split("\n")[0]
                    kernel_grammar_list.append(line)

        #
        # We scan the kernel directory to obtain the list of available kernels
        #
        if self.mission.pds == '3':
            kernel_type_list = ['CK', 'FK', 'IK', 'LSK', 'PCK', 'SCLK', 'SPK', 'DSK']
        else:
            kernel_type_list = ['ck', 'fk', 'ik', 'lsk', 'pck', 'sclk', 'spk', 'dsk']

        #
        # All the files of the directory are read into a list
        #
        mkgen_kernels = []
        excluded_kernels = []

        for kernel_type in kernel_type_list:

            #
            # First we build the list of excluded kernels
            #
            for kernel_grammar in kernel_grammar_list:
                if 'exclude:' in kernel_grammar:
                    excluded_kernels.append(
                            kernel_grammar.split('exclude:')[-1])

            for kernel_grammar in kernel_grammar_list:

                if 'date:' in kernel_grammar:
                    kernel_grammar = kernel_grammar.split('date:')[-1]
                    dates = True
                else:
                    dates = False

                if kernel_grammar.split('.')[-1].lower() in type2extension(kernel_type):
                    try:
                        if self.mission.pds == '3':
                            path = self.mission.bundle_directory + '/DATA'
                        else:
                            path = self.mission.bundle_directory+'/spice_kernels'
                        latest_kernel = get_latest_kernel(kernel_type,
                                                          path,
                                                          kernel_grammar,
                                                          dates=dates,
                                                          excluded_kernels=excluded_kernels,
                                                          mkgen=True)
                    except:
                        latest_kernel = []

                    if latest_kernel:
                        if not isinstance(latest_kernel, list):
                            latest_kernel = [latest_kernel]

                        for kernel in latest_kernel:
                            mkgen_kernels.append(kernel_type + '/' + kernel)


        #
        # We remove duplicate entries from the kernel list (possible depending on
        # the grammar)
        #
        mkgen_kernels = list(OrderedDict.fromkeys(mkgen_kernels))

        #
        # We subset the spice_kernels collection with the kernels only
        # in the Meta-Kernel
        #
        spice_kernels_metakernel = []
        for spice_kernel in self.spice_kernels_collection.files:
            for name in mkgen_kernels:
                if spice_kernel.name in name:
                    spice_kernels_metakernel.append(spice_kernel)

        self.spice_kernels_metakernel = spice_kernels_metakernel

        self.spice_kernels_metakernel = mkgen_kernels

        #
        # The kernel list for the new mk is formatted accordingly
        #
        kernels = ''
        kernel_dir_name = None
        for kernel in mkgen_kernels:

            if kernel_dir_name:
                if kernel_dir_name != kernel.split('.')[1]:
                    kernels += '\n'

            kernel_dir_name = kernel.split('.')[1]

            kernels += "'$KERNELS/{}'\n".format(kernel)


        self.KERNELS_IN_METAKERNEL = kernels[:-1]

        metakernel_dictionary = vars(self)

        with open(self.path, "w+") as f:

            for line in fileinput.input(self.template):
                line = line.rstrip()
                for key, value in metakernel_dictionary.items():
                    if isinstance(value, str) and key.isupper() and key in \
                            line and '$' in line:
                        line = line.replace('$', '')
                        line = line.replace(key, value)
                f.write(line + '\n')

        return

    def kernel_mission_phase(self):

        start_time = self.start_time[0:-1]   # avoid reading the Z at the end of UTC tag
        stop_time = self.stop_time[0:-1]

        mission_phase_map = self.mission.root_dir + \
                            '/etc/mission_phases_{}.txt'.format(self.mission.accronym.lower())


        with open(mission_phase_map, 'r') as f:
            mission_phases = list()

            if start_time == 'N/A' and stop_time == 'N/A':

                for line in f:
                    mission_phases.append(line.split('   ')[0])

            if start_time != 'N/A' and stop_time == 'N/A':

                next_phases_bool = False
                start_tdb = spiceypy.str2et(start_time)  # + ' TDB')

                for line in f:
                    start_phase_date = line.rstrip().split(' ')[-2]
                    start_phase_tdb = spiceypy.str2et(start_phase_date)

                    stop_phase_date = line.rstrip().split(' ')[-1]
                    stop_phase_tdb = spiceypy.str2et(stop_phase_date)

                    if next_phases_bool == True:
                        mission_phases.append(line.split('   ')[0])

                    elif start_tdb >= start_phase_tdb:
                        mission_phases.append(line.split('   ')[0])
                        next_phases_bool = True

            if start_time != 'N/A' and stop_time != 'N/A':

                next_phases_bool = False
                start_tdb = spiceypy.str2et(start_time)  # + ' TDB')
                stop_tdb = spiceypy.str2et(stop_time)  # + ' TDB')

                for line in f:

                    start_phase_date = line.rstrip().split(' ')[-2]
                    start_phase_tdb = spiceypy.str2et(
                        start_phase_date)  # + ' TDB')

                    stop_phase_date = line.rstrip().split(' ')[-1]
                    stop_phase_tdb = spiceypy.str2et(stop_phase_date)
                    stop_phase_tdb += 86400
                    #  One day added to account for gap between phases.

                    if next_phases_bool == True and start_phase_tdb >= stop_tdb:
                        break

                    if next_phases_bool == True:
                        mission_phases.append(line.split('   ')[0])

                    if start_phase_tdb <= start_tdb <= stop_phase_tdb:
                        mission_phases.append(line.split('   ')[0])
                        next_phases_bool = True

        mission_phases_str = ''
        for phase in mission_phases:
            mission_phases_str += '                                 ' + phase + ',\r\n'

        mission_phases_str = mission_phases_str[0:-3]

        self.mission_phases = mission_phases_str

        return

    def producer(self):

        # if self.extension.upper() != 'PCK' and self.extension.upper() != 'DSK':
        if self.extension.upper() != 'TPC' and (self.name[0:6] != 'EARTH_' or self.extension[0:3] != 'BPC' ) \
        or self.name == 'PHOBOS_M157_GAS_V01.BDS':
            producer = 'ESA-ESAC'
        else:
            producer = 'JPL'

        return producer


