import os
import glob
import fileinput

from arcgen.utils.time import current_time
from arcgen.utils.files import extension2type
from arcgen.utils.files import add_carriage_return

class PDSLabel(object):

    def __init__(self, mission, product):

        self.root_dir = mission.root_dir
        self.product = product
        self.mission_accronym = mission.accronym
        self.mission = mission

        #
        # Fields from Mission
        #
        if mission.pds == '3':
            self.PDS3_DATASET           = mission.dataset
        self.XML_MODEL                  = mission.xml_model
        self.SCHEMA_LOCATION            = mission.schema_location
        self.INFORMATION_MODEL_VERSION  = mission.information_model_version
        self.PDS4_MISSION_NAME          = mission.name
        self.PDS4_MISSION_LID           = mission.lid
        self.CURRENT_YEAR               = current_time().split('-')[0]
        self.BUNDLE_DESCRIPTION_LID     = \
            'urn:esa:psa:{}_spice:document:spiceds'.format(self.mission_accronym)
        self.PRODUCT_CREATION_TIME      = product.creation_time
        self.PDS3_PRODUCT_CREATION_TIME = product.creation_time.split('T')[0]
        self.FILE_SIZE                  = product.size
        self.FILE_CHECKSUM              = product.checksum

        self.PDS4_SPACECRAFT_NAME       = mission.spacecraft
        self.PDS4_SPACECRAFT_TYPE       = 'Spacecraft'
        self.PDS4_SPACECRAFT_LID        = \
            'urn:esa:psa:context:instrument_host:spacecraft.{}'.format(
                    mission.spacecraft.lower())

        #
        # For Bundles with multiple s/c
        #
        if mission.secondary_spacecrafts:
            #
            # We sort out how many S/C we have
            #
            sec_scs = mission.secondary_spacecrafts.split('/')

            for sc in sec_scs:
                if sc:
                    if sc.lower() in product.name.lower():
                        self.PDS4_SPACECRAFT_NAME = sc
                        self.PDS4_SPACECRAFT_TYPE = 'Spacecraft'
                        self.PDS4_SPACECRAFT_LID = \
                        'urn:esa:psa:context:instrument_host:spacecraft.{}'.format(sc.lower())


        self.PDS4_TARGET_NAME = mission.target
        self.PDS4_TARGET_TYPE = mission.target_type
        self.PDS4_TARGET_LID = \
            'urn:nasa:pds:context:target:{}.{}'.format(
                    mission.target_type.lower(),
                    mission.target.lower())

        #
        # For Bundles with multiple targets
        #
        if mission.secondary_targets:
            #
            # We sort out how many S/C we have
            #
            sec_tars = mission.secondary_targets.split('/')

            for tar in sec_tars:
                if tar:
                    tar_name = tar.split(',')[0]
                    if tar_name.lower() in product.name.lower():
                        self.PDS4_TARGET_NAME = tar.split(',')[0]
                        self.PDS4_TARGET_TYPE = tar.split(',')[1]
                        self.PDS4_TARGET_LID = \
                        'urn:nasa:pds:context:target:{}.{}'.format(
                                tar.split(',')[1].lower(),
                                tar.split(',')[0].lower())

        return


    def write_label(self):

        label_dictionary = vars(self)

        if self.mission.pds == '3':
            label_extension = '.LBL'
        else:
            label_extension = '.xml'

        label_name = self.product.path.split('.')[0] + label_extension

        with open(label_name, "w+") as f:

            for line in fileinput.input(self.template):
                line = line.rstrip()
                for key, value in label_dictionary.items():
                    if isinstance(value, str) and key in line and '$' in line:
                        line = line.replace('$'+key, value)

                line = add_carriage_return(line)

                f.write(line)

        return


class KernelPDSLabel(PDSLabel):

    def __init__(self, mission, product):

        PDSLabel.__init__(self, mission, product)

        if mission.pds == '3':
            self.template = self.root_dir + '/etc/template_product_spice_kernel.LBL'
        else:
            self.template = self.root_dir + '/etc/template_product_spice_kernel.xml'

        #
        # Fields from Kernels
        #
        self.FILE_NAME = product.name
        self.PRODUCT_LID = self.product.lid
        self.FILE_FORMAT = product.file_format
        self.START_TIME = product.start_time
        self.STOP_TIME = product.stop_time
        self.KERNEL_TYPE_ID = product.type.upper()
        self.PRODUCT_VID = self.product.vid
        self.SPICE_KERNEL_DESCRIPTION = product.description

        #
        # PDS3 specific:
        #
        if mission.pds == '3':
            self.MISSION_PHASES = product.mission_phases
            self.INTERCHANGE_FORMAT = product.file_format
            self.PDS3_KERNEL_TYPE = self.product.pds3type
            self.PRODUCER_ID = product.producer_id
            self.PRODUCT_VERSION_TYPE = product.product_version_type
            self.NAIF_INSTRUMENT_ID = product.naif_instrument_id
            self.RECORD_TYPE = product.record_type
            self.TARGET_NAME = product.target_name

        if self.mission.pds == '4':
            self.PDS4_SPACECRAFT_NAME = product.mission.spacecraft

            extension = '.LBL'
        else:
            extension = '.xml'
        #
        #    We try to extract it form the filename, otherwise it goes to the
        #    default one.
        #
        #try:
        #    spacecraft_name = product.name.split('_')[1].upper()
        #    if 'EDM' in spacecraft_name:
        #        spacecraft_name = 'edm'
        #except:
        #    spacecraft_name = mission.spacecraft


        self.name = product.name.split('.')[0] + extension

        self.write_label()

        return


class MetaKernelPDS4Label(PDSLabel):

    def __init__(self, mission, product):

        PDSLabel.__init__(self, mission, product)

        self.template = self.root_dir + '/etc/template_product_spice_kernel_mk.xml'

        #
        # Fields from Kernels
        #
        self.FILE_NAME = product.name
        self.PRODUCT_LID = self.product.lid
        self.FILE_FORMAT = 'Character'
        self.START_TIME = product.start_time
        self.STOP_TIME = product.stop_time
        self.KERNEL_TYPE_ID = product.type.upper()
        self.PRODUCT_VID = self.product.vid
        self.SPICE_KERNEL_DESCRIPTION = product.description

        self.KERNEL_INTERNAL_REFERENCES = self.get_kernel_internal_references()
        self.SKD_SPACECRAFTS = self.get_skd_spacecrafts()

        #
        #    We try to extract it form the filename, otherwise it goes to the
        #    default one.
        #
        # try:
        #    spacecraft_name = product.name.split('_')[1].upper()
        #    if not spacecraft_name in ['TGO', 'EDM']:
        #        spacecraft_name = mission.spacecraft
        # except:
        #    spacecraft_name = mission.spacecraft

        self.name = product.name.split('.')[0] + '.xml'

        self.write_label()

        return

    def get_kernel_internal_references(self):

        #
        # From the collection we only use kernels in the MK
        #
        kernel_list_for_label = ''
        for kernel in self.product.spice_kernels_metakernel:
            try:
                kernel_lid = kernel.lid
            except:
                #
                # When we generate the label from an already existing MK
                # or when we generate an increment of the archive.
                #
                kernel_lid = f"urn:esa:psa:" \
                             f"{self.mission.accronym}" \
                             f"_spice:spice_kernels:" \
                             f"{extension2type(kernel.split('.')[-1])}_" \
                             f"{kernel.split('/')[-1]}"

            kernel_list_for_label += \
            '        <Internal_Reference>\r\n' + \
            '            <lid_reference>{}\r\n'.format(kernel_lid) + \
            '            </lid_reference>\r\n' + \
            '            <reference_type>data_to_associate</reference_type>\r\n' +\
            '        </Internal_Reference>\r\n'

        kernel_list_for_label = kernel_list_for_label.rstrip() + '\r\n'

        return kernel_list_for_label

    def get_skd_spacecrafts(self):

        sc = ['{}'.format(self.mission.spacecraft)]

        sec_scs = self.mission.secondary_spacecrafts.split('/')
        if not isinstance(sec_scs, list):
            sec_scs = [sec_scs]

        scs = sc + sec_scs

        sc_list_for_label = ''

        for sc in scs:
            if sc:
                sc_list_for_label += \
                '            <Observing_System_Component>\r\n' + \
                '                <name>{}</name>\r\n'.format(sc.split(',')[0]) + \
                '                <type>Spacecraft</type>\r\n' + \
                '                <Internal_Reference>\r\n' + \
                '                    <lid_reference>urn:esa:psa:context:instrument_host:spacecraft.{}</lid_reference>\r\n'.format(sc.split(',')[0].lower()) + \
                '                    <reference_type>is_instrument_host</reference_type>\r\n' + \
                '                </Internal_Reference>\r\n' + \
                '            </Observing_System_Component>\r\n'

        sc_list_for_label = sc_list_for_label.rstrip() + '\r\n'

        return sc_list_for_label


class InventoryPDS4Label(PDSLabel):

    def __init__(self, mission, collection, inventory):

        PDSLabel.__init__(self, mission, inventory)

        self.collection = collection
        self.template = self.root_dir + \
                        '/etc/template_collection_{}.xml'.format(collection.type)


        self.COLLECTION_LID = self.collection.lid
        self.COLLECTION_VID = self.collection.vid
        self.START_TIME = mission.start
        self.STOP_TIME = mission.increment_stop
        self.FILE_NAME = inventory.name
        self.DOI =  mission.doi

        #
        # We need to account for the files present in the last increment.
        #
        if self.mission.increment:
            path = os.sep.join(self.product.path.split(os.sep)[0:-1])
            collections = glob.glob(path + '/collection*.xml')
            collections.sort()
            last_collection_label = collections[-1]

            with open(last_collection_label) as xml:
                for line in xml:
                    if '<records>' in line:
                        n_records_increment = line.split('<records>')[-1].split('<')[0]
        else:
            n_records_increment = 0

        self.N_RECORDS = str(len(collection.files) + int(n_records_increment))


        self.name = collection.name.split('.')[0] + '.xml'

        self.write_label()

        return


    def write_label(self):

        label_dictionary = vars(self)
        label_name = self.product.path.split('_inventory')[0] + self.product.path.split('_inventory')[-1].split('.')[0] + '.xml'

        with open(label_name, "w+") as f:

            for line in fileinput.input(self.template):
                line = line.rstrip()
                for key, value in label_dictionary.items():
                    if isinstance(value, str) and key in line and '$' in line:
                        line = line.replace('$'+key, value)

                line = add_carriage_return(line)

                f.write(line)

        return


class InventoryPDS3Label(PDSLabel):

    def __init__(self, mission, collection, inventory):

        PDSLabel.__init__(self, mission, inventory)

        self.collection = collection
        self.template = self.root_dir + \
                        '/etc/template_collection_{}.LBL'.format(
                            collection.type)

        self.PRODUCT_CREATION_TIME = current_time()
        self.DATASET = mission.dataset

        self.N_RECORDS = str(len(collection.files))

        self.write_label()

        return


    def write_label(self):

        label_dictionary = vars(self)


        with open(self.mission.bundle_directory + '/INDEX/INDEX.LBL', "w+") as f:

            for line in fileinput.input(self.template):
                line = line.rstrip()
                for key, value in label_dictionary.items():
                    if isinstance(value, str) and key in line and '$' in line:
                        line = line.replace('$' + key, value)

                line = add_carriage_return(line)

                f.write(line)

        return


class DocumentPDS4Label(PDSLabel):

    def __init__(self, mission, collection, inventory):

        PDSLabel.__init__(self, mission, inventory)

        self.mission = mission
        self.collection = collection
        self.template = self.root_dir + \
                        '/etc/template_product_html_document.xml'

        self.PRODUCT_LID = inventory.lid
        self.PRODUCT_VID = inventory.vid
        self.START_TIME = mission.start
        self.STOP_TIME = mission.stop
        self.FILE_NAME = inventory.name
        self.CURRENT_TIME = current_time()
        self.CURRENT_DATE =self.CURRENT_TIME.split('T')[0]

        self.name = collection.name.split('.')[0] + '.xml'

        self.write_label()

        return


class BundlePDS4Label(PDSLabel):

    def __init__(self, mission, readme):

        PDSLabel.__init__(self, mission, readme)

        self.template = self.root_dir + \
                        '/etc/template_bundle.xml'

        self.BUNDLE_LID = self.product.bundle.lid
        self.BUNDLE_VID = self.product.bundle.vid

        self.START_TIME = mission.start
        self.STOP_TIME = mission.increment_stop
        self.FILE_NAME = readme.name
        self.CURRENT_TIME = current_time()
        self.CURRENT_DATE =self.CURRENT_TIME.split('T')[0]
        self.DOI =  mission.doi


        for collection in self.product.bundle.collections:
            if collection.name == 'spice_kernels':
                self.SPICE_COLL_LIDVID =  collection.lid + '::' + collection.vid
                self.SPICE_COLL_STATUS = 'Primary'
            if collection.name == 'document':
                self.DOC_COLL_LIDVID =  collection.lid + '::' + collection.vid
                self.DOC_COLL_STATUS = 'Primary'

        self.write_label()

        return
