import os
import re
import glob
import shutil
import filecmp
import tarfile
import subprocess
import numpy.testing as npt

from adcsng.utils.files import get_exe_dir
from adcsng.command_line import main as adcsng


def runADCSng(version, debug=False, log=False, untar=True):

    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    if untar:
        tar = tarfile.open('incoming.tar.gz')
        tar.extractall()
        tar.close()

    print(f"Executing ADCSng v{version} for Mars-Express")
    adcsng('mars_express.json', debug=debug, log=log)

def cleanup():

    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    try:
        shutil.rmtree('incoming')
        os.remove('._incoming')
    except:
        pass

    for CleanUp in glob.glob('ker_gen/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('ancdr/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('log/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('temp/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt') or not CleanUp.endswith('temp.log'):
            try:
                shutil.rmtree(CleanUp)
            except:
                if 'aareadme.txt' not in CleanUp:
                    os.remove(CleanUp)


def SPKcompare():

    spks = glob.glob('ker_gen/spk/*.BSP')
    for spk in spks:

        exe_dir = get_exe_dir()
        spk_1 = spk
        spk_2 = f'ker_val/spk/{spk.split(os.sep)[-1]}'

        if not os.path.exists(spk_1): raise NameError(f"SPK Kernel {spk_1} does not exist")
        if not os.path.exists(spk_2): raise NameError(f"SPK Kernel {spk_2} does not exist")

        command = f'../..{exe_dir}/spkdiff  {spk_1}  {spk_2}'
        print(command)


        command_process = subprocess.Popen(command, shell=True,
                                           stdout=subprocess.PIPE,stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print(text)
        if 'Toolkit version:' in text:
            raise NameError(f"SPKDIFF did not work with SPK Kernels {spk}")

        try:
            text = text.split('\n')
            for line in text:
                if 'Position:' in line or 'Velocity:' in line:
                    npt.assert_almost_equal(float(line.split()[1]), 0.0, decimal=10)
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=10)
                if 'Position (km)' in line or 'Velocity (km/s)' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=10)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=10)
        except AssertionError as error:
            print(error)
            raise NameError(f"SPK Kernels {spk} are different")


def ORBNUMcompare():

    orbnums = glob.glob('ker_gen/orbnum/*.ORB')
    for orb in orbnums:

        reldir = f"/orbnum/{orb.split(os.sep)[-1]}"

        if not filecmp.cmp('ker_gen'+reldir, 'ker_val'+reldir):
            raise NameError(f"ORBNUM Files {orb} are different")
        else:
            with open('ker_gen' + reldir, 'r') as fil:
                print(fil.read())

    return


def MKcompare():

    mks = [f for f in os.listdir('ker_gen/mk/') if
           re.search(r'.*[^0-9].TM', f)]

    if not mks:
        raise NameError("No MK Kernels available")

    for mk in mks:
        data_gen = []
        record = False
        with open(f'ker_gen/mk/{mk}', 'r') as m:
            for line in m:
                if record: data_gen.append(line)
                if '\\begindata' in line: record = True
                if '\\begintext' in line: break

        print(data_gen)

        with open(f'ker_val/mk/{mk}', 'r') as m: data_val = m.readlines()

        if not set(data_gen).issubset(data_val):
            raise NameError("MK Kernels are different")

    return


if __name__ == '__main__':
    runADCSng('1.0.0', debug=True, log=True)
    #SPKcompare()
    #CKcompare()