Program tcpescan_emrsp
   Implicit None

   Integer :: I, ios
   Integer(Kind=4) :: N, P
   Character(Len=100) :: tmfile, lskfile

   Byte, Dimension(1:18) :: DDS
   Byte, Dimension(1:16) :: SPARE
   Byte, Dimension(1:50) :: TCP

   Integer(Kind=8) :: DDS_seconds, DDS_microseconds, DDS_Packet_Length
   Integer         :: Ground_Station, Virtual_Channel
   Byte            :: SLE_Type, Time_Quality

   Byte            :: Version, SynchroCheck, Validity, Accuracy, Synchro
   Integer(Kind=4) :: Algorithm
   Real(Kind=8)    :: Gradient, Offset, OBT_N, UTC_N, Display_Offset, StdDev, StdErr

   Real(Kind=8)    :: DDS_Time

   Real(Kind=8)    :: TDB, TDT, DT
   Character(Len=26)  :: UTCSTR
   Character(Len=26) :: DT_STR

   ! get the name of the telemetry file from the command line argument
   Call Getarg(1, tmfile)
   If ( Len(Trim(tmfile)) .EQ. 0 ) Stop 'No input file specified.'

   ! get the name of the telemetry file from the command line argument
   Call Getarg(2, lskfile)
   If ( Len(Trim(lskfile)) .EQ. 0 ) Stop 'No input LSK specified.'

   Call FURNSH( lskfile )

   ! initialise packet number and file position pointer
   N = 1
   P = 0

   ! open the time correlation packet file
   Open (UNIT=1, IOSTAT=ios, STATUS='Old', ACTION='Read', FILE=tmfile, FORM='Unformatted', ACCESS='Direct', RECL=1)
   If (ios .NE. 0) Stop 'File not found.'

   Do

      ! read DDS Header
      Do I = 1, 18
         Read (UNIT=1, IOSTAT=ios, REC=P+I) DDS(I)
         If (ios .NE. 0 .AND. I .EQ. 1) THEN
            DT_STR = 'DELTET='
            Print "(1x, a, f20.15)", DT_STR, DT
            Stop 'End of File.'
         End If
         If (ios .NE. 0 .AND. I .GT. 1) Stop 'Unexpected end of File.'
      End Do
      ! read TM Packet Header
      Do I = 1, 16
         Read (UNIT=1, IOSTAT=ios, REC=P+18+I) SPARE(I)
         If (ios .NE. 0) Stop 'Unexpected end of File.'
      End Do
      ! read Time Correlation Packet
      Do I = 1, 50
         Read (UNIT=1, IOSTAT=ios, REC=P+18+16+I) TCP(I)
         If (ios .NE. 0) Stop 'Unexpected end of File.'
      End Do

      ! analyse DDS header
      DDS_seconds       = Int(16777216,8)*Unsign(DDS(1)) + 65536*Unsign(DDS(2)) + 256*Unsign(DDS(3)) + Unsign(DDS(4))
      DDS_microseconds  = Int(16777216,8)*Unsign(DDS(5)) + 65536*Unsign(DDS(6)) + 256*Unsign(DDS(7)) + Unsign(DDS(8))
      DDS_Packet_Length = Int(16777216,8)*Unsign(DDS(9)) + 65536*Unsign(DDS(10)) + 256*Unsign(DDS(11)) + Unsign(DDS(12))
      Ground_Station    = 256*Unsign(DDS(13)) + Unsign(DDS(14))
      Virtual_Channel   = 256*Unsign(DDS(15)) + Unsign(DDS(16))
      SLE_Type          = DDS(17)
      Time_Quality      = DDS(18)

      ! analyse time correlation packet
      ! Version, Algorithm, SynchroCheck, Synchro and Display_Offset not available in TCP packets; values are fixed
      Version        = 1
      Algorithm      = 1
      SynchroCheck   = 1
      Validity       = TCP(1)
      Accuracy       = TCP(2)
      Synchro        = 1
      Gradient       = Int82Real( TCP(3:10) )
      Offset         = Int82Real( TCP(11:18) )
      OBT_N          = Int82Real( TCP(19:26) )
      UTC_N          = Int82Real( TCP(27:34) )
      Display_Offset = OBT_N - UTC_N + Offset
      StdDev         = Int82Real( TCP(35:42) )
      StdErr         = Int82Real( TCP(43:50) )

      DDS_Time  = DDS_seconds + DDS_microseconds/1000000.

      ! transform unix time to UTC time and ephemeris time
      UTCSTR = Unix2UTC(DDS_Time)
      Call UTC2ET(UTCSTR, TDB)
      CALL DELTET(TDB, 'ET', DT)
      ! compensate for the difference of max approx 2ms between TDB and TDT
      TDT = TDB + 0.184 + ( Int(DT)-DT )

      ! output line displaying DDS header and TCP information
!      Print "(I6, X, F20.6, X, I3, X, I1, X, I1, X, I1, X, I3, X, &
!        & F20.18, X, E20.13, X, F20.6, X, F20.6, X, F20.6, X, F20.18, X, F20.18)", &
!        & N, DDS_Time, DDS_Packet_Length, Ground_Station, Virtual_Channel, SLE_Type, Time_Quality, &
!        & Gradient, Offset, OBT_N, UTC_N, Display_Offset, StdDev, StdErr
     Print "(I6, X, F20.6, X, A, X, I1, X, I1, X, I1, X, I1, X, I1, X, I1, X, &
       & F20.18, X, E20.13, X, F20.6, X, F20.6, X, F20.6, X, E20.13, X, E20.13)", &
       & N, DDS_Time, UTCSTR, Version, Algorithm, SynchroCheck, Validity, Accuracy, Synchro, &
       & Gradient, Offset, OBT_N, UTC_N, Display_Offset, StdDev, StdErr
!      Print "(I6, X, F20.6, X, A, X, I1, X, I1, X, &
!        & F20.18, X, E20.13, X, F20.6, X, F20.6, X, F20.6, X, F20.18, X, F20.18)", &
!        & N, DDS_Time, UTCSTR, Validity, Accuracy, &
!        & Gradient, Offset, OBT_N, UTC_N, Display_Offset, StdDev, StdErr
!       Print "(I6, X, F15.3, X, A, X, F14.3)", N, DDS_Time, UTCSTR, TDT

      ! increase the packet number and file position pointer
      N = N + 1
      P = P + 18 + DDS_Packet_Length

   End Do

   Close (UNIT=1)

   Call UNLOAD( lskfile )

   Contains

   ! function to transform signed 1-byte integers into "unsigned integers"
   Function Unsign(B)
      Implicit None
      Byte, Intent(In) :: B
      Integer :: Unsign

      Unsign = B

      If (B .LT. 0) Unsign = B + 256

   End Function Unsign

   ! function for converting Byte(1:8) to Real(Kind=8)
   Function Int82Real(Int8)
      Implicit None
      Byte, Dimension(1:8), Intent(In) :: Int8
      Real(Kind=8)    :: Int82Real
      Real(Kind=8)    :: Mantissa
      Integer(Kind=4) :: Exponent
      Integer         :: I, E, B

      Mantissa = 1.
      E = -1
      Do I = 3, 0, -1
         If ( BTest(Int8(2),I) ) Mantissa = Mantissa + 2.**E
         E = E - 1
      End Do
      Do B = 3, 8
         Do I = 7, 0, -1
            If ( BTest(Int8(B),I) ) Mantissa = Mantissa + 2.**E
            E = E - 1
         End Do
      End Do
      Exponent = 16*IAnd(Int8(1),B'01111111') + IShft(Int8(2),-4) - 1023
      Int82Real = Mantissa * 2.**Exponent
      If ( BTest(Int8(1),7) ) Int82Real = - Int82Real

   End Function Int82Real

   ! function for converting UNIX time to UTC
   Function Unix2UTC(unix)
      Implicit None
      Real(Kind=8), Intent(In) :: unix
      Character(Len=26)        :: Unix2UTC
      Integer(Kind=4)          :: julday
      Integer                  :: I, year, month, day
      Real(Kind=8)             :: hours, minutes, seconds
      Character(Len=9)         :: sec_string

      ! determine the julian day (plus 0.5) from unix time
      julday  = Int( unix / (60*60*24) + 2440587.5 + 0.5 )

      ! determine UTC hours, minutes, and seconds from unix time
      hours   = Mod(unix,86400.) / 3600.
      minutes = Mod(hours,1.) * 60.
      seconds = Mod(minutes,1.) * 60.

      Write (sec_string, "(F9.6)") seconds
      ! check if the F9.6-format rounding for the seconds leads to a value of 60.000000
      ! if this is the case, set the seconds to zero and increment the minutes etc.
      If ( sec_string .EQ. "60.000000" ) Then
         sec_string = "00.000000"
         minutes = minutes + 1.
         If ( minutes .GE. 60. ) Then
            minutes = 0.
            hours = hours + 1.
            If ( hours .GE. 24. ) Then
               hours = 0.
               julday = julday + 1
            End If
         End If
      End If

      ! get the calendar date
      Call caldat( julday, month, day, year )

      ! generate the UTC time string
      Write (Unix2UTC, "(I4,A,I2,A,I2,A,I2,A,I2,A,A)") year, '-', month, '-', day, 'T', &
        & Int(hours), ':', Int(minutes), ':', sec_string

      ! replace spaces with zeroes in the time string
      Do
         I = Index(Unix2UTC,' ')
         If ( I .EQ. 0 ) Exit
         Unix2UTC(I:I) = '0'
      End Do


   End Function Unix2UTC

End Program tcpescan_emrsp






