C$Procedure      DYNMFR ( Dynamic Frame )
 
      SUBROUTINE DYNMFR( AXIS, DESCRP,
     .                   ET,   FROM, TO,  XFORM,  ROTATE )
 
C$ Abstract
C
C     This is an umbrella routine that provides the capability
C     of generating frame transformations based upon the dynamics
C     of bodies and other reference frames.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAMES
C
C$ Declarations
 
      IMPLICIT NONE
 
      INCLUDE              'zzdynenm.inc'
      CHARACTER*(*)         AXIS
      CHARACTER*(*)         DESCRP
      DOUBLE PRECISION      ET
      CHARACTER*(*)         FROM
      CHARACTER*(*)         TO
      DOUBLE PRECISION      XFORM ( 6, 6 )
      DOUBLE PRECISION      ROTATE( 3, 3 )
 
C$ Brief_I/O
C
C     VARIABLE  I/O  Entry Point
C     --------  ---  --------------------------------------------------
C     AXIS       I
C     DESCRP     I
C     ET         I
C     FROM       I
C     TO         I
C     XFORM      O
C     ROTATE     O
C
C$ Detailed_Input
C
C     AXIS       Specifies which axis of the frame a direction
C                is tied to.  This is either the +X, +Y, +Z or
C                -X, -Y, or -Z axis of the dynamic frame
C
C     DESCRP     Describes the direction that is aligned with AXIS.
C
C     ET         Epoch at which to return a state transformation
C                matrix
C
C     FROM       The frame that the state transformation matrix
C                transforms from.
C
C
C     TO         The frame that the state transformation matrix
C                transforms to.
C
C                FROM and TO may be the word 'DYNAMIC' (to indicate
C                the frame defined by calls to DYNMPR and DYNMSC)
C                or may have the form 'FRAME <frame>' where <frame>
C                is the string for any frame recognized by the SPICE
C                frame system.
C
C$ Detailed_Output
C
C     XFORM      A state transformation matrix that converts states
C                relative to the dynamic frame to states in the RELTO
C                frame.
C
C     ROTATE     A rotation matrix that converts positions relative to
C                the dynamic frame to positions relative to the RELTO
C                frame.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1)
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This is an umbrella routine for three entry points that
C     allow you to define a special frame called 'DYNAMIC' that
C     is oriented relative to the directions associated with various
C     physical directions.
C
C     The DYNAMIC frame is defined using calls to DYNMPR (defines the
C     primary axis of the frame) and DYNMSC (defines the secondary
C     axis of the frame).
C
C     Once the axes of the frames have been defined, the entry points
C     DYNMXF and DYNMRT are available to return state and position
C     transformation matrices, respectively.
C
C     More details as they become available.
C
C$ Examples
C
C     Suppose you wanted to construct a frame whose X-axis is pointed
C     at the sun (as seen from the spacecraft PHOENIX) and whose Y-axis
C     is pointed in the direction of the earth.
C
C     The following set calls will define the frame.
C
C     First set the primary axis.
C
C     CALL DYNMPR ('+X',
C                  'POSITION OF SUN FROM PHOENIX CORRECTION LT+S' )
C
C     Next set the secondary axis:
C
C     CALL DYNMSC ( '+Y',
C                   'POSITION OF EARTH FROM PHOENIX CORRECTION LT+S' )
C
C
C     To get the state transformation from the dynamic frame to
C     J2000, make the following call:
C
C     CALL DYNMXF ( ET, 'FRAME J2000', 'DYNAMIC', XFORM )
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C     B.V. Semenov    (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.1.0, 29-APR-2009 (BVS)
C
C        Bug fix: modified DYNFPR and DYNFSC entry points to call
C        ZZONEWRD, ZZDOFLIP, and ZZWHCHAX as subroutines because some
C        compilers cannot handle function entry points with different
C        return types (LOGICAL vs. INTEGER)
C
C        Updated DYNMRT and DYNMXF entry points to remove non-standard
C        use of duplicate arguments in DVHAT, VHAT, VSCLG, VSCL,
C        INVSTM, and XPOSE calls.
C
C-    SPICELIB Version 1.0.0, 15-MAR-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Construct a dynamic frame
C
C-&
C
C     SPICELIB Functions
C
      LOGICAL               RETURN
      LOGICAL               FAILED

C
C     Local parameters.
C 
      INTEGER               WDSIZE
      PARAMETER           ( WDSIZE = 32 )

C
C     The following variables are set via calls to DYNFPR and DYNFSC
C
      CHARACTER*(WDSIZE)    DRNAME(2)
      INTEGER               PRMSEC(2)
      LOGICAL               FLIP  (2)
      LOGICAL               ONEWRD
      LOGICAL               HAVPRM
      LOGICAL               HAVSEC
 
      SAVE                  DRNAME
      SAVE                  PRMSEC
      SAVE                  FLIP
      SAVE                  HAVPRM
      SAVE                  HAVSEC
C
C     Variables that need to be initialized or that control the
C     initialization
C
      LOGICAL               FIRST
      DOUBLE PRECISION      SCALE ( 3, 3 )
 
      SAVE                  FIRST
      SAVE                  SCALE
 
C
C     Utility Variables.
C
      CHARACTER*(WDSIZE)    RELTO
 
      DOUBLE PRECISION      DDIR  ( 6 )

      DOUBLE PRECISION      TVEC3 ( 3 )
      DOUBLE PRECISION      TVEC6 ( 6 )
      DOUBLE PRECISION      TMAT3 ( 3, 3 )
      DOUBLE PRECISION      TMAT6 ( 6, 6 )
 
      INTEGER               DEP
      INTEGER               HANDLE
      INTEGER               I
      INTEGER               J
      INTEGER               PRIMRY
      INTEGER               SCNDRY
 
      LOGICAL               INVERT
      LOGICAL               OK
      LOGICAL               UNARY
      LOGICAL               ZEROV
 
 
      DATA                  FIRST  / .TRUE.  /
      DATA                  HAVPRM / .FALSE. /
      DATA                  HAVSEC / .FALSE. /
 
 
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'DYNMFR')
      CALL SETMSG ( 'The umbrella routine DYNMFR has been called '
     .//            'directly. This routine exists only to '
     .//            'include its entry points. Calling it '
     .//            'directly is a programming error. ' )
      CALL SIGERR ( 'SPICE(BOGUSENTRY)'  )
      CALL CHKOUT ( 'DYNMFR' )
      RETURN
 
 
C$Procedure DYNFPR ( Dynamic Frame Primary Axis Definition )
 
      ENTRY DYNFPR ( AXIS, DESCRP )
 
C$ Abstract
C
C     Define the primary axis of a dynamic frame.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME
C
C$ Declarations
C
C     CHARACTER*(*)         AXIS
C     CHARACTER*(*)         DESCRP
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     AXIS       I   Name of the primary axis
C     DESCRP     I   Description of the direction to attach to axis.
C
C$ Detailed_Input
C
C     AXIS       is the name of the primary axis.  Allowed values are
C                '+X','+Y','+Z','-X','-Y',and '-Z'.  The value of
C                axis is case insensitive.
C
C     DESCRP     is a description of the the direction to associate
C                with the primary axis.  Templates for valid
C                descriptions are:
C
C                POSITION OF @body FROM @body CORRECTION @word
C
C                VELOCITY OF @body FROM @body CORRECTION @word
C
C                   where @body is the name or Id-code of a body, and
C                   @word is the one of the corrections allowed by
C                   SPKEZ.
C
C                FIXED @name XYZ @number @number @number'
C
C                FIXED @name RADEC RA @number DEC @number
C
C                    where @name is the name of some reference frame
C                    and @number represents the "obvious" coordinate.
C
C                SUBOBSERVER ON @body FROM @body
C                CORRECTION @word
C
C                   where @body is the name or Id-code of a body, and
C                   @word is the one of the corrections allowed by
C                   SPKEZ.
C
C                NORMAL TO @body AT (2:2){ LATITUDE @number
C                                        | LONGITUDE @number }
C
C                NORMAL TO @body AT
C                (2:2){ LATITUDE @number
C                     | CARTOGRAPHIC LONGITUDE @number }
C
C                NORMAL TO @body AT (3:3){ X @number
C                                        | Y @number
C                                        | Z @number }
C
C                   where @body is the name of some object and @number
C                   is the obvious coordinate.
C
C
C                INCREASING (1:1){ LATITUDE
C                                | LONGTIDUTE
C                                | CARTOGRAPHIC LONGITUDE
C                                }
C                        AT (2:2){ LATITUDE @number
C                                | @option
C                                | LONGITUDE @number
C                                | CARTOGRAPHIC LONGITUDE @number
C                                } ON @body
C
C
C
C                INCREASING (1:1){ LATITUDE
C                                | LONGTIDUTE
C                                | CARTOGRAPHIC LONGITUDE
C                                }
C                        AT (3:3){ X @number
C                                | Y @number
C                                | Z @number
C                                } ON @body
C
C
C                DECREASING (1:1){ LATITUDE
C                                | LONGTIDUTE
C                                | CARTOGRAPHIC LONGITUDE
C                                }
C                        AT (2:2){ LATITUDE @number
C                                | @option
C                                | LONGITUDE @number
C                                | CARTOGRAPHIC LONGITUDE @number
C                                } ON @body
C
C                DECREASING (1:1){ LATITUDE
C                                | LONGTIDUTE
C                                | CARTOGRAPHIC LONGITUDE
C                                }
C                        AT (3:3){ X @number
C                                | Y @number
C                                | Z @number
C                                } ON @body
C
C
C                   where @body is the name of some object and @number
C                   is the obvious coordinate.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C      1) Errors are diagnosed by the private routine ZZPRSDIR.  See
C         that routine for a complete descriptions.  Potential problems
C         include improperly specified descriptions, improperly named
C         Axis, inadequate date in the kernel pool to determine
C         particulars of a direction
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This entry point and the entry point DYNMSC are used to describe
C     the directions associated with the axes of the dynamic frame.
C
C     By using the DYNMPR entry point, you specify exact direction the
C     axis specified by AXIS.  The companion entry point DYNMSC defines
C     the direction of its axis to be as close to the specified
C     direction description as is possible given the constraint imposed
C     by the primary axis definition.
C
C$ Examples
C
C     See the main umbrella.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C     B.V. Semenov    (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.1.0, 29-APR-2009 (BVS)
C
C        Bug fix: modified to call ZZONEWRD, ZZDOFLIP, and ZZWHCHAX as
C        subroutines because some compilers cannot handle function
C        entry points with different return types (LOGICAL vs. INTEGER)
C
C-    SPICELIB Version 1.0.0, 15-MAY-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Define the primary axis of a dynamic frame.
C
C-&
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'DYNFPR')
C
C     The utility ZZPRSDIR does all of the work of parsing the
C     description.
C
      CALL ZZWHCHAX ( AXIS, PRMSEC(1) )
      CALL ZZDOFLIP ( AXIS, FLIP  (1) )
C
C     We make a nod to backward compatibility. If DESCRP is
C     a single word, we regard it as being the name of the
C     principle direction.  Otherwise we up a local name and
C     define a local direction.
C
      CALL ZZONEWRD( DESCRP, ONEWRD )
      IF ( ONEWRD ) THEN
 
         DRNAME(1) = DESCRP
         CALL GETDRM ( DRNAME(1), UNARY, HANDLE, HAVPRM )
 
         HAVPRM = HAVPRM .AND. .NOT. FAILED()
 
      ELSE
 
         DRNAME(1) = '$LOCAL_PRINCIPLE'
         CALL DYNMPR ( DRNAME(1), DESCRP )
         HAVPRM  = .NOT. FAILED()
 
      END IF
 
 
      CALL CHKOUT ( 'DYNFPR' )
      RETURN
 
 
 
 
 
C$Procedure      DYNMSC ( Dynamic Frame Secondary Axis Definition )
 
      ENTRY DYNFSC ( AXIS, DESCRP )
 
C$ Abstract
C
C     Define the secondary axis of a dynamic frame.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME
C
C$ Declarations
C
C     CHARACTER*(*)         AXIS
C     CHARACTER*(*)         DESCRP
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     AXIS       I   Name of the secondary axis
C     DESCRP     I   Description of the direction to attach to axis.
C
C$ Detailed_Input
C
C     AXIS       is the name of the secondary axis.  Allowed values are
C                '+X','+Y','+Z','-X','-Y',and '-Z'.  The value of
C                axis is case insensitive.
C
C     DESCRP     is a description of the the direction to associate
C                with the secondary axis.  Templates for valid
C                descriptions are described in DYNMPR.
C
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C      1) Errors are diagnosed by the private routine ZZPRSDIR.  See
C         that routine for a complete descriptions.  Potential problems
C         include improperly specified descriptions, improperly named
C         Axis, inadequate date in the kernel pool to determine
C         particulars of a direction
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This entry point and the entry point DYNMPR are used to describe
C     the directions associated with the axes of the dynamic frame.
C
C     By using the DYNMSC entry point, you specify the direction of of
C     the secondary axis to be as close to the specified direction
C     description as is possible given the constraint imposed by the
C     primary axis definition.
C
C$ Examples
C
C     See the main umbrella.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C     B.V. Semenov    (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.1.0, 29-APR-2009 (BVS)
C
C        Bug fix: modified to call ZZONEWRD, ZZDOFLIP, and ZZWHCHAX as
C        subroutines because some compilers cannot handle function
C        entry points with different return types (LOGICAL vs. INTEGER)
C
C-    SPICELIB Version 1.0.0, 15-MAY-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Define the secondary axis of a dynamic frame.
C
C-&
 
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN  ( 'DYNFSC' )
 
 
      CALL ZZWHCHAX ( AXIS, PRMSEC(2) )
      CALL ZZDOFLIP ( AXIS, FLIP  (2) )
C
C     We make a nod to backward compatibility. If DESCRP is
C     a single word, we regard it as being the name of the
C     principle direction.  Otherwise we up a local name and
C     define a local direction.
C
      CALL ZZONEWRD( DESCRP, ONEWRD )
      IF ( ONEWRD ) THEN
 
         DRNAME(2) = DESCRP
         CALL GETDRM ( DRNAME(2), UNARY, HANDLE, HAVSEC )
         HAVSEC = HAVSEC .AND. .NOT. FAILED()
 
      ELSE
 
         DRNAME(2) = '$LOCAL_SECONDARY'
         CALL DYNMPR ( DRNAME(2), DESCRP )
         HAVSEC  = .NOT. FAILED()
 
      END IF
 
 
 
      CALL CHKOUT ( 'DYNFSC' )
      RETURN
 
 
 
 
C$Procedure      DYNMXF ( Dynamic Frame State Transformation )
 
      ENTRY DYNMXF ( ET, FROM, TO, XFORM )
 
C$ Abstract
C
C     Return the state transformation matrix that transforms from/to
C     the DYNAMIC frame to/from any other frame recognized by the
C     frame system.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME
C
C$ Declarations
C
C     DOUBLE PRECISION      ET
C     CHARACTER*(*)         FROM
C     CHARACTER*(*)         TO
C     DOUBLE PRECISION      XFORM ( 6, 6 )
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     ET         I   Epoch at which to return the transformation
C     FROM       I   Frame to transform from
C     TO         I   Frame to transform to
C     XFORM      O   State transformation matrix.
C
C$ Detailed_Input
C
C     ET          is the epoch in seconds past the J2000 epoch at
C                 which a state transformation matrix is desired.
C
C     FROM        are names of frames to transform from and to
C     TO          respectively.  One of these must be 'DYNAMIC'
C                 which specifies the frame defined by calls to
C                 DYNMPR and DYNMSC.  The other frame must be
C                 of the form 'FRAME <frame_name>' where <frame_name>
C                 is the name of some frame recognized by FRAMEX
C                 (if necessary, after appropriate frame definitions
C                 have been loaded).  The error 'SPICE(BADXFORMREQUEST)'
C                 will be signaled if the two frames do not satisfy
C                 the requirements spelled out above.
C
C$ Detailed_Output
C
C     XFORM       is a state transformation matrix that transforms
C                 states relative to the FROM frame to states relative
C                 to the TO frame.
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If FROM and TO do not satisfy the requirement laid out above
C        the error 'SPICE(BADXFORMREQUEST)' will be signaled.
C
C     2) If the frame specified by the 'FRAME <frame_name>' string
C        is not recognized, the error 'SPICE(UNRECOGNIZEDFRAME)' will
C        be signaled.
C
C     3) Singularities in the definition of the frame transformation
C        are NOT detected.  Maybe eventually they will be, but for
C        now try to make up definitions that don't suffer from this
C        problem.
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This entry point allows you to retrieve the state transformation
C     that maps from or to any other reference frame recognized by
C     the frame subsystem.
C
C$ Examples
C
C     See the main umbrella.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C     B.V. Semenov    (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.1.0, 29-APR-2009 (BVS)
C
C        Updated to remove non-standard use of duplicate arguments
C        in DVHAT, VSCLG, and INVSTM calls.
C
C-    SPICELIB Version 1.0.0, 15-MAR-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Frame transformation between a dynamic and standard frame
C
C-&
 
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'DYNMXF')
C
C     Initializations on the first pass.
C
      IF ( FIRST ) THEN
         FIRST = .FALSE.
 
         SCALE ( X, Y ) =  1.0D0
         SCALE ( X, Z ) = -1.0D0
         SCALE ( X, X ) =  0.0D0
         SCALE ( Y, X ) = -1.0D0
         SCALE ( Y, Y ) =  0.0D0
         SCALE ( Y, Z ) =  1.0D0
         SCALE ( Z, X ) =  1.0D0
         SCALE ( Z, Y ) = -1.0D0
         SCALE ( Z, Z ) =  0.0D0
 
      END IF
C
C     Make sure that both the Primary and Secondary axes have
C     been defined.
C
      IF ( .NOT. HAVPRM ) THEN
         CALL SETMSG ( 'No specification for the primary axis of '
     .   //            'the dynamic frame has been supplied. ' )
         CALL SIGERR ( 'SPICE(PRIMARYUNDEFINED)' )
         CALL CHKOUT ( 'DYNMXF' )
         RETURN
      END IF
 
      IF ( .NOT. HAVSEC ) THEN
         CALL SETMSG ( 'No specification for the secondary axis of '
     .   //            'the dynamic frame has been supplied. ' )
         CALL SIGERR ( 'SPICE(SECONDARYUNDEFINED)' )
         CALL CHKOUT ( 'DYNMXF' )
         RETURN
      END IF
 
      RELTO = 'SPAM'
 
      CALL ZZDYNCHK ( FROM, TO, OK, RELTO, INVERT )
 
      IF ( .NOT. OK ) THEN
         CALL CHKOUT ( 'DYNMXF' )
         RETURN
      END IF
C
C     Get the two directions
C
      PRIMRY  = PRMSEC(1)
      SCNDRY  = PRMSEC(2)
      ZEROV   = .FALSE.
 
      DO I = 1, 2
 
         CALL EVALDR  ( DRNAME(I), ET,  RELTO, ZEROV, DDIR )
 
C
C        Flip the state if necessary (i.e. if it is along one of
C        the -X, -Y or -Z axes.
C
         IF ( FLIP(I) ) THEN
            CALL VMINUG ( DDIR, 6, XFORM(1,PRMSEC(I)) )
         ELSE
            CALL VEQUG  ( DDIR, 6, XFORM(1,PRMSEC(I)) )
         END IF
 
      END DO
C
C     We now have "two" columns of the state transformation matrix
C     filled in.  Fill in the dependent axis and polish up the
C     secondary axis.
C
 
      DEP = 6 - PRIMRY - SCNDRY
 
      CALL DVHAT ( XFORM(1, PRIMRY ), TVEC6 )
      CALL VEQUG ( TVEC6, 6, XFORM(1, PRIMRY ) )
 
      CALL DUCRSS( XFORM(1,PRIMRY), XFORM(1,SCNDRY), XFORM(1,DEP) )
      CALL VSCLG ( SCALE(PRIMRY,SCNDRY ),            XFORM(1,DEP),
     .                                 6,            TVEC6 )
      CALL VEQUG ( TVEC6, 6, XFORM(1,DEP) )

      CALL DUCRSS( XFORM(1,DEP), XFORM(1,PRIMRY),    XFORM(1,SCNDRY) )
      CALL VSCLG ( SCALE(DEP,PRIMRY ),               XFORM(1,SCNDRY),
     .                              6,               TVEC6 )
      CALL VEQUG ( TVEC6, 6, XFORM(1,SCNDRY) )
 
 
C
C     Now fill in the right side of the state transformation matrix
C
      DO I = 1, 3
         DO J = 1, 3
            XFORM(I,  J+3) = 0.0
            XFORM(I+3,J+3) = XFORM(I,J)
         END DO
      END DO
 
      IF ( INVERT ) THEN
         CALL INVSTM ( XFORM, TMAT6 )
         CALL MEQUG  ( TMAT6, 6, 6, XFORM )
      END IF
 
      CALL CHKOUT ( 'DYNMXF' )
      RETURN
 
 
C$Procedure      DYNMRT ( Dynamic Rotation )
 
      ENTRY DYNMRT ( ET, FROM, TO, ROTATE )
 
C$ Abstract
C
C     Return the rotation matrix that transforms positions from/to
C     the DYNAMIC frame to/from any other frame recognized by the
C     frame system.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME
C
C$ Declarations
C
C     DOUBLE PRECISION      ET
C     CHARACTER*(*)         FROM
C     CHARACTER*(*)         TO
C     DOUBLE PRECISION      ROTATE ( 3, 3 )
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     ET         I   Epoch at which to return the transformation
C     FROM       I   Frame to transform from
C     TO         I   Frame to transform to
C     ROTATE     O   Rotation matrix.
C
C$ Detailed_Input
C
C     ET          is the epoch in seconds past the J2000 epoch at
C                 which a state transformation matrix is desired.
C
C     FROM        are names of frames to transform from and to
C     TO          respectively.  One of these must be 'DYNAMIC'
C                 which specifies the frame defined by calls to
C                 DYNMPR and DYNMSC.  The other frame must be
C                 of the form 'FRAME <frame_name>' where <frame_name>
C                 is the name of some frame recognized by FRAMEX
C                 (if necessary, after appropriate frame definitions
C                 have been loaded).  The error 'SPICE(BADXFORMREQUEST)'
C                 will be signaled if the two frames do not satisfy
C                 the requirements spelled out above.
C
C$ Detailed_Output
C
C     ROTATE      is a rotation matrix that transforms positions
C                 relative to the FROM frame to positions relative to
C                 the TO frame.
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If FROM and TO do not satisfy the requirement laid out above
C        the error 'SPICE(BADXFORMREQUEST)' will be signaled.
C
C     2) If the frame specified by the 'FRAME <frame_name>' string
C        is not recognized, the error 'SPICE(UNRECOGNIZEDFRAME)' will
C        be signaled.
C
C     3) Singularities in the definition of the frame transformation
C        are NOT detected.  Maybe eventually they will be, but for
C        now try to make up definitions that don't suffer from this
C        problem.
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This entry point allows you to retrieve the rotation
C     that maps from or to any other reference frame recognized by
C     the frame subsystem.
C
C$ Examples
C
C     See the main umbrella.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C     B.V. Semenov    (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.1.0, 29-APR-2009 (BVS)
C
C        Updated to remove non-standard use of duplicate arguments
C        in VHAT, VSCL, and XPOSE calls.
C
C-    SPICELIB Version 1.0.0, 15-MAR-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Rotation between a dynamic and standard frame
C
C-&
 
 
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'DYNMRT')
C
C     Initializations on the first pass.
C
      IF ( FIRST ) THEN
         FIRST = .FALSE.
 
         SCALE ( X, Y ) =  1.0D0
         SCALE ( X, Z ) = -1.0D0
         SCALE ( X, X ) =  0.0D0
         SCALE ( Y, X ) = -1.0D0
         SCALE ( Y, Y ) =  0.0D0
         SCALE ( Y, Z ) =  1.0D0
         SCALE ( Z, X ) =  1.0D0
         SCALE ( Z, Y ) = -1.0D0
         SCALE ( Z, Z ) =  0.0D0
 
      END IF
C
C     Make sure that both the Primary and Secondary axes have
C     been defined.
C
      IF ( .NOT. HAVPRM ) THEN
         CALL SETMSG ( 'No specification for the primary axis of '
     .   //            'the dynamic frame has been supplied. ' )
         CALL SIGERR ( 'SPICE(PRIMARYUNDEFINED)' )
         CALL CHKOUT ( 'DYNMRT' )
         RETURN
      END IF
 
      IF ( .NOT. HAVSEC ) THEN
         CALL SETMSG ( 'No specification for the secondary axis of '
     .   //            'the dynamic frame has been supplied. ' )
         CALL SIGERR ( 'SPICE(SECONDARYUNDEFINED)' )
         CALL CHKOUT ( 'DYNMRT' )
         RETURN
      END IF
 
C
C     Make sure we have a legitimate request
C
      CALL ZZDYNCHK ( FROM, TO, OK, RELTO, INVERT )
 
      IF ( .NOT. OK ) THEN
         CALL CHKOUT ( 'DYNMRT' )
         RETURN
      END IF
C
C     Get the two directions
C
      PRIMRY  = PRMSEC(1)
      SCNDRY  = PRMSEC(2)
      ZEROV   = .TRUE.
 
      DO I = 1, 2
 
         CALL EVALDR ( DRNAME(I), ET,  RELTO, ZEROV, DDIR )
C
C        Flip the state if necessary (i.e. if it is along one of
C        the -X, -Y or -Z axes.
C
         IF ( FLIP(I) ) THEN
            CALL VMINUS ( DDIR, ROTATE(1,PRMSEC(I)) )
         ELSE
            CALL VEQU   ( DDIR, ROTATE(1,PRMSEC(I)) )
         END IF
 
 
      END DO
C
C     We now have "two" columns of the state transformation matrix
C     filled in.  Fill in the dependent axis and polish up the
C     secondary axis.
C
 
      DEP = 6 - PRIMRY - SCNDRY
 
      CALL VHAT ( ROTATE(1, PRIMRY ), TVEC3 )
      CALL VEQU ( TVEC3, ROTATE(1, PRIMRY ) )
 
      CALL UCRSS( ROTATE(1,PRIMRY), ROTATE(1,SCNDRY), ROTATE(1,DEP) )
      CALL VSCL ( SCALE(PRIMRY,SCNDRY ),              ROTATE(1,DEP),
     .                                                TVEC3 )
      CALL VEQU ( TVEC3, ROTATE(1,DEP) )

      CALL UCRSS( ROTATE(1,DEP), ROTATE(1,PRIMRY), ROTATE(1,SCNDRY) )
      CALL VSCL ( SCALE(DEP,PRIMRY ),              ROTATE(1,SCNDRY),
     .                                             TVEC3 )
      CALL VEQU ( TVEC3, ROTATE(1,SCNDRY) )
 
C
C     Now fill in the right side of the state transformation matrix
C
      IF ( INVERT ) THEN
         CALL XPOSE ( ROTATE, TMAT3 )
         CALL MEQU  ( TMAT3, ROTATE )
      END IF
 
      CALL CHKOUT ( 'DYNMRT' )
      RETURN
 
 
 
      END
