C$Procedure RDOEM ( Read OEM file --- umbrella routine )
 
      SUBROUTINE RDOEM ( FILE, LINE, LNUM, TYPE, EOF )
 
C$ Abstract
C
C     Umbrella routine for OEM low-level readers.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     OEM2SPK
C
C$ Declarations
 
      IMPLICIT NONE
 
      CHARACTER*(*)         FILE
      CHARACTER*(*)         LINE
      INTEGER               LNUM
      CHARACTER*(*)         TYPE
      LOGICAL               EOF

C$ Brief_I/O
C
C     VARIABLE  I/O  Entry points
C     --------  ---  --------------------------------------------------
C     FILE       I   RDLIN
C     LINE       O   RDLIN
C     LNUM       O   RDLIN
C     TYPE       O   RDLIN
C     EOF        O   RDLIN
C
C$ Detailed_Input
C
C     See the entry points for a description of their arguments.
C                     
C$ Detailed_Output
C
C     See the entry points for a description of their arguments.
C                     
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If this routine is called directly, the error 
C        SPICE(BOGUSENTRY) is signaled.
C
C     See the entry points for a description of exceptions specific
C     to those routines.
C
C$ Files
C
C     This suite of routines supports sequential reading of lines
C     of text from an OEM file.  The OEM format is defined in 
C     reference [1].
C
C$ Particulars
C
C     This routine simplifies the logic required in order for 
C     the program OEM2SPK to parse an OEM file.  Lines obtained
C     from this routine are always non-blank, and a line 
C     classification string and line number are returned with
C     each line.
C
C     The entry point PUSHLN enables a caller to "put back" a line
C     so that the line will be returned on the next read operation.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-1,
C          September, 2004.
C
C     [2]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-2,
C          November, 2009.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C     M.C. Kim    (JPL)
C
C$ Version
C
C-    OEM2SPK Version 2.0.0, 30-AUG-2013 (MCK)
C
C        The COVARIANCE_START_LINE and COVARIANCE_STOP_LINE keywords are
C        added to the list of TYPE.
C
C-    OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&

C
C     SPICELIB  functions
C     
      INTEGER               RTRIM
      INTEGER               LTRIM

      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C
      CHARACTER*(*)         VERMRK
      PARAMETER           ( VERMRK = 'CCSDS_OEM_VERS' )

      CHARACTER*(*)         CREMRK
      PARAMETER           ( CREMRK = 'CREATION_DATE' )
      
      CHARACTER*(*)         COMMRK
      PARAMETER           ( COMMRK = 'COMMENT' )
      
      CHARACTER*(*)         MTBMRK
      PARAMETER           ( MTBMRK = 'META_START' )

      CHARACTER*(*)         MTEMRK
      PARAMETER           ( MTEMRK = 'META_STOP' )

      CHARACTER*(*)         CVBMRK
      PARAMETER           ( CVBMRK = 'COVARIANCE_START' )

      CHARACTER*(*)         CVEMRK
      PARAMETER           ( CVEMRK = 'COVARIANCE_STOP' )

      CHARACTER*(*)         ORGMRK
      PARAMETER           ( ORGMRK = 'ORIGINATOR' )

      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 256 )

      INTEGER               NBUFF
      PARAMETER           ( NBUFF  = 1 )

      INTEGER               FILSIZ
      PARAMETER           ( FILSIZ = 255 )

      



C
C     Local variables
C
      CHARACTER*(LNSIZE)    LINBUF
      CHARACTER*(LNSIZE)    MAPCOM
      CHARACTER*(LNSIZE)    MAPCRE
      CHARACTER*(LNSIZE)    MAPMTB
      CHARACTER*(LNSIZE)    MAPMTE
      CHARACTER*(LNSIZE)    MAPCVB
      CHARACTER*(LNSIZE)    MAPCVE
      CHARACTER*(LNSIZE)    MAPORG
      CHARACTER*(LNSIZE)    MAPVER
      CHARACTER*(LNSIZE)    SAVTYP

      INTEGER               LEVEL
      INTEGER               LNO

      LOGICAL               FIRST
      LOGICAL               INDATA
      LOGICAL               INMETA
      LOGICAL               SAVEOF

C
C     Saved variables
C
      SAVE                  FIRST
      SAVE                  INDATA
      SAVE                  INMETA
      SAVE                  LEVEL
      SAVE                  LINBUF
      SAVE                  LNO
      SAVE                  MAPCOM
      SAVE                  MAPCRE
      SAVE                  MAPMTB
      SAVE                  MAPMTE
      SAVE                  MAPCVB
      SAVE                  MAPCVE
      SAVE                  MAPORG
      SAVE                  MAPVER
      SAVE                  SAVEOF
      SAVE                  SAVTYP
    
C
C     Initial values
C
      DATA                  FIRST  / .TRUE.  /
      DATA                  INDATA / .FALSE. /
      DATA                  INMETA / .FALSE. /
      DATA                  LEVEL  /  0      /
      DATA                  LNO    /  0      /
      DATA                  SAVEOF / .FALSE. /



      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'RDOEM' )

      CALL SIGERR ( 'SPICE(BOGUSENTRY)' )

      CALL CHKOUT ( 'RDOEM' )
      RETURN





C$Procedure RDLIN ( Read non-blank text line from OEM file )

      ENTRY RDLIN ( FILE, LINE, LNUM, TYPE, EOF ) 
 
C$ Abstract
C
C     Read a non-blank line of text from an OEM file.  Classify
C     the line.  Return the line number. 
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     OEM2SPK
C
C$ Declarations
C 
C     CHARACTER*(*)         FILE
C     CHARACTER*(*)         LINE
C     INTEGER               LNUM
C     CHARACTER*(*)         TYPE
C     LOGICAL               EOF
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     FILE       I   Name of OEM file.
C     LINE       O   Current non-blank text line read from OEM.
C     LNUM       O   Line number.
C     TYPE       O   Line classification.
C     EOF        O   End-of-file flag.
C
C$ Detailed_Input
C
C     FILE           is the name of an OEM file to be read.
C                     
C$ Detailed_Output
C
C     LINE           is the "next" non-blank line in the OEM file.
C                    Lines are read sequentially from the file.
C
C                    LINE contains no more than 255 characters.
C
C     LNUM           is the ordinal position of LINE in the file.
C                    Line numbers start at 1.
C
C     TYPE           is a string indicating the classification of the
C                    output line.  Possible values are:
C
C                       'COMMENT'                
C                       'CREATION_DATE_LINE'     
C                       'ORIGINATOR_LINE'     
C                       'DATA_LINE'    
C                       'METADATA_LINE'    
C                       'METADATA_START_LINE'    
C                       'METADATA_STOP_LINE'      
C                       'COVARIANCE_START_LINE'                
C                       'COVARIANCE_STOP_LINE'                
C                       'VERSION_LINE'            
C                       
C     EOF            is a logical flag indicating the end of the input
C                    file has been reached.  When EOF is .TRUE., the 
C                    other output arguments are undefined.  Subsequent
C                    read attempts will continue to return EOF = .TRUE.
C                     
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If the current line cannot be classified, the error 
C        SPICE(FORMATERROR) is signaled.
C
C     2) If a read error occurs, the problem will be diagnosed by
C        routines called by this routine.
C
C$ Files
C
C     This suite of routines supports sequential reading of lines
C     of text from an OEM file.  The OEM format is defined in 
C     reference [1].
C
C$ Particulars
C
C     This routine simplifies the parsing logic in the program 
C     OEM2SPK by centralizing line classification functions.  
C
C     Note that in some circumstances, after reading a line, it 
C     is convenient to push the line back on the input stream. 
C     The entry point PUSHLN is provided for this purpose.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-1,
C          September, 2004.
C     [2]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-2,
C          November, 2009.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C     M.C. Kim    (JPL)
C
C$ Version
C
C-    OEM2SPK Version 2.0.0, 29-AUG-2013 (MCK)
C
C        The COVARIANCE_START_LINE and COVARIANCE_STOP_LINE keywords
C        are added to the list of TYPE.
C
C-    OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&

      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'RDLIN' )

C
C     On the first pass, fetch the marker values to look for.
C     These are obtained from the inverse of the string mapping
C     defined in the kernel pool.
C
      IF ( FIRST ) THEN

         CALL CMAPI ( COMMRK, MAPCOM )
         CALL CMAPI ( VERMRK, MAPVER )
         CALL CMAPI ( CREMRK, MAPCRE )
         CALL CMAPI ( ORGMRK, MAPORG )
         CALL CMAPI ( MTBMRK, MAPMTB )
         CALL CMAPI ( MTEMRK, MAPMTE )
         CALL CMAPI ( CVBMRK, MAPCVB )
         CALL CMAPI ( CVEMRK, MAPCVE )
         
         FIRST = .FALSE.

      END IF

C
C     If we're already at EOF, just return.
C
      IF ( SAVEOF ) THEN
         
         EOF = .TRUE.

         CALL CHKOUT ( 'RDLIN' )
         RETURN

      END IF

C
C     If there's a stacked line, return it.  The saved line
C     number has already been set by the push operation that
C     stacked the line.
C
      IF ( LEVEL .GE. 1 ) THEN

         LINE  = LINBUF
         LEVEL = LEVEL - 1
         TYPE  = SAVTYP
         EOF   = SAVEOF

         CALL CHKOUT ( 'RDLIN' )
         RETURN

      ELSE
C
C        Read a line from the file.  Keep going until we get a 
C        non-blank line.
C
         CALL RDTEXT ( FILE, LINE, EOF )

         SAVEOF = EOF
         LNO    = LNO + 1

         DO WHILE (       ( .NOT. EOF      ) 
     .              .AND. ( .NOT. FAILED() )
     .              .AND. ( LINE .EQ. ' '  )  )

            CALL RDTEXT ( FILE, LINE, EOF )

            SAVEOF = EOF
            LNO    = LNO + 1

         END DO

         IF ( EOF .OR. FAILED() ) THEN
            CALL CHKOUT ( 'RDLIN' )
            RETURN
         END IF

      END IF

C
C     We found a non-blank line.  Classify it.
C
C     Note that we order our checks so that we don't look 
C     for special flags, other than comment markers and
C     the metadata end flag, within a metadata block.
C
      CALL LJUST ( LINE, LINE )

      IF ( LINE(:RTRIM(MAPCOM)) .EQ. MAPCOM ) THEN

         TYPE = 'COMMENT'

      ELSE IF (LINE(LTRIM(LINE):RTRIM(LINE)) .EQ. MAPCVB ) THEN

         TYPE = 'COVARIANCE_START_LINE'

      ELSE IF (LINE(LTRIM(LINE):RTRIM(LINE)) .EQ. MAPCVE ) THEN

         TYPE = 'COVARIANCE_STOP_LINE'

      ELSE IF ( LINE .EQ. MAPMTE ) THEN

         TYPE = 'METADATA_STOP_LINE'

C
C        Change our state to indicate we're no longer reading 
C        a metadata block and that we are reading data lines.
C
         INMETA = .FALSE.
         INDATA = .TRUE.


      ELSE IF ( INMETA ) THEN

         TYPE = 'METADATA_LINE'


      ELSE IF ( LINE(:RTRIM(MAPVER)) .EQ. MAPVER ) THEN

         TYPE = 'VERSION_LINE'


      ELSE IF (  LINE(:RTRIM(MAPCRE)) .EQ. MAPCRE ) THEN

         TYPE = 'CREATION_DATE_LINE'


      ELSE IF (  LINE(:RTRIM(MAPORG)) .EQ. MAPORG ) THEN

         TYPE = 'ORIGINATOR_LINE'


      ELSE IF ( LINE .EQ. MAPMTB ) THEN

         TYPE = 'METADATA_START_LINE'

C
C        Change our state to indicate we're reading a metadata 
C        block.  If we were reading data previously, we're no
C        longer doing so.
C
         INMETA = .TRUE.
         INDATA = .FALSE.


      ELSE IF ( INDATA ) THEN

         TYPE = 'DATA_LINE'

      ELSE
C
C        We really oughtn't get to this point.
C
         CALL SETMSG ( 'Line #* is unclassified: *' )
         CALL ERRINT ( '*',  LNO                    )
         CALL ERRCH  ( '*',  LINE                   )
         CALL SIGERR ( 'SPICE(FORMATERROR)'         )
         CALL CHKOUT ( 'RDLIN'                      )
         RETURN

      END IF

      
      LINBUF = LINE
      LNUM   = LNO
      SAVTYP = TYPE
      
      CALL CHKOUT ( 'RDLIN' )
      RETURN




C$Procedure PUSHLN ( Push OEM line )

      ENTRY PUSHLN
 
C$ Abstract
C
C     Push a line returned from RDLIN back on the virtual OEM 
C     input stream.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     OEM2SPK
C
C$ Declarations
C 
C     None.
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     None.
C
C$ Detailed_Input
C
C     None.
C                     
C$ Detailed_Output
C
C     None.
C                     
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If two push operations are requested with no intervening
C        read operation, the error SPICE(BUFFERFULL) is signaled.
C
C$ Files
C
C     This suite of routines supports sequential reading of lines
C     of text from an OEM file.  The OEM format is defined in 
C     reference [1].
C
C$ Particulars
C
C     Note that in some circumstances, after reading a line, it 
C     is convenient to push the line back on the input stream. 
C     This entry point is provided for this purpose.  After this
C     routine is called, the next call to RDLIN will return the
C     same outputs as the previous call to RDLIN.  Subsequent
C     calls to RDLIN will return new lines from the input OEM file.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     1) Multiple lines cannot be pushed without intervening RDLIN
C        calls.
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-1,
C          September, 2004.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&

      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'PUSHLN' )

      IF ( LEVEL .LT. NBUFF ) THEN

         LEVEL  = 1

C
C        The line number remains unchanged.
C
      ELSE

         CALL SETMSG ( 'Buffer full; level = 1.' )
         CALL SIGERR ( 'SPICE(BUFFERFULL)'       )
         CALL CHKOUT ( 'PUSHLN'                  )
         RETURN

      END IF


      CALL CHKOUT ( 'PUSHLN' )
      RETURN
      END



 
      
