import os
import shutil
import re

from adcsng.utils.files import fill_template
from adcsng.core.emrsp import  get_rm_site

def make_setup_file(directories, kernel, naif_util):
    """
    Creates a configuration file named ``name`` with the data
    provided in the ``config_data``.

    :param name:
    :param config_data:
    :return: NoneType
    """

    configuration = kernel.config
    configuration['kernels_to_load'] = kernel.get_kernels2load()

    #
    # If we are running AEM2CK with a planning CK we need to load the
    # specific configuration
    #
    if 'sclk' in configuration and '_fict_' in configuration['sclk'] and naif_util != 'prediCkt':
        naif_util += '_plan'


    if kernel.input_type == 'SPECS' and naif_util != 'msopck_plan':
        template = directories.templates + '/' + naif_util + '_' + \
                   kernel.config['default_attitude'] + '.setup'
    elif kernel.input_type == 'EVENTS':
        template = directories.templates + '/' + naif_util + '_' + \
                   kernel.config['default_attitude'] + '.setup'
    else:
        template = directories.templates + '/' + naif_util + '.setup'

    name = naif_util +  '.setup'

    #
    # We need to modify the template to include the angular value (1 or 2)
    # That will be used. We also include the maximum interpolation interval
    # option.
    #
    if kernel.input_type == 'TM_DAFCAT':
        with open(template, 'r') as f:
            with open('msopck_dafcat_{}.setup'.format(str(kernel.dafcat)), 'w+') as i:
                for line in f:
                    if '{object_frame}' in line:
                        line = line.replace('object_frame',
                                            'object_frame_{}'.format(str(kernel.dafcat)))
                    if '{object_id}' in line:
                        line = line.replace('object_id',
                                            'object_id_{}'.format(str(kernel.dafcat)))
                    if '{target_frame}' in line:
                        line = line.replace('target_frame',
                                            'target_frame_{}'.format(str(kernel.dafcat)))

                    if not '{interpolation_interval}' in line:
                        i.write(line)
                    else:
                        if kernel.config['fill_tm_gaps'] == "False":
                            i.write(line)

        template = 'msopck_dafcat_{}.setup'.format(str(kernel.dafcat))

        #
        # We need to modify the template to the maximum interpolation interval
        # if need be.
        #
    if kernel.input_type == 'TM':
        with open(template, 'r') as f:
            with open('msopck.setup','w+') as i:
                for line in f:
                    if not '{interpolation_interval}' in line:
                        i.write(line)
                    else:
                        if kernel.config['fill_tm_gaps'] == 'False':
                            i.write(line)

        template = 'msopck.setup'.format(str(kernel.dafcat))

    fill_template(template, name, kernel.config)

    #
    # This is a particular case for ExoMarsRSP in which the mapping of the
    # setup file needs to be updated due to the different sites.
    #
    if 'OEM' in kernel.input_type and kernel.config['mission_accr'] == 'RM':
        get_rm_site(name, kernel)


    if kernel.input_type == 'TM_DAFCAT':
        shutil.copy('msopck.setup','msopck_dafcat_{}.setup'.format(str(kernel.dafcat)))


def setup_production(kernel, directories, input_directory=None):

    """
    Function that creates a temporary directory for a given adcsng conversion.
    One is created for each kernel to be created. The name of the directory
    is the source file name without extension. If the process fails the
    directory remains in order to analise the error.

    :param directories: Object that provides all the relevant directories for
       the adcsng execution
    :type directories: object
    :param input_file: Source file to be converted by adcsng
    :type input_file: str
    :param input_directory: Directory where the Source file is
    :type input_directory: str
    :return: Current Working directory and Execution Directory
    :rtype: (str, str)
    """

    input_files = kernel.source

    if not isinstance(input_files, list):
       input_files = [input_files]

    #
    # We determine if multiple input files need to use the same temporary
    # directory
    #
    if 'merge_kernel' in kernel.config.keys():
        if kernel.config['merge_kernel'] == 'True':
        #
        # We use a regular expression:
        #  - We group with parenthesis
        #  - First we only want the _ char
        #  - Then multiple decimal numbers that we indicate with 0-9 and
        #    the repetition with *
        #  - Then we want a . char
        #  - We close the group
        #
            temp_dir = re.compile("([_][0-9]*[.])").split(input_files[0])[0]
        else:
            if 'AEM_QUAT_MERGED' in kernel.config['input_type']:
                temp_dir = input_files[0] + '_interp'
            else:
                temp_dir = input_files[0]
    elif 'input_type' in kernel.config.keys():
        if 'AEM_QUAT_MERGED' in kernel.config['input_type']:
            temp_dir = input_files[0] + '_interp'
        else:
            temp_dir = input_files[0]
    else:
        temp_dir = input_files[0]

    # input_directory is optional for we do not need to have an input file to setup the production
    working_directory = directories.temp + '/' + temp_dir

    cwd = os.getcwd()

    #
    # Create a directory using the input filename without the extension and
    # move there the source file.
    #
    #
    # If an input file is repeated in an ADCSng execution it is because is
    # a SPECS file from an OEM or a CK to be merged,
    # therefore we do not create the directory.
    #
    if os.path.isdir(working_directory):
        if kernel.input_type != 'TM_DAFCAT' \
                and kernel.input_type != 'SPECS' \
                and kernel.input_type != 'OEM_MERGED':
            if 'merge_kernel' in kernel.config.keys():
                if kernel.config['merge_kernel'] != 'True':
                    shutil.rmtree(working_directory)
            else:
                shutil.rmtree(working_directory)

    if kernel.input_type != 'SPECS' and \
            kernel.input_type != 'TM_DAFCAT' and \
             kernel.input_type != 'OEM_MERGED':
        if 'merge_kernel' in kernel.config.keys():
            if kernel.config['merge_kernel'] != 'True':
                os.mkdir(working_directory)
        else:
            os.mkdir(working_directory)
    if kernel.input_type == 'TM_DAFCAT':
        if not os.path.exists(working_directory):
            os.mkdir(working_directory)
    if 'merge_kernel' in kernel.config.keys():
        if kernel.config['merge_kernel'] == 'True':
            if not os.path.exists(working_directory):
                os.mkdir(working_directory)


    if input_directory:

        for input_file in input_files:
            #
            # copy2 method is used instead of copy method for it preserves the
            # metadata of the file and hence the generation time, used to determine
            # the delivery date of the input file.
            #

            #
            # Default Attitude CKs do not have an input file.
            #
            if input_file and kernel.input_type != 'SPECS' and kernel.input_type != 'TM_DAFCAT':

                #
                # If we are merging multiple CKs we need to move the generated inputs instead of copying them only
                #
                if 'merge_kernel' in kernel.config.keys():
                    if kernel.config['merge_kernel'] == 'True':
                        try:
                            shutil.move(input_directory + '/' + input_file,working_directory+'/'+input_file)
                        except:
                            if not os.path.exists(working_directory+'/'+input_file):
                                raise  NameError(f'{input_file} not present in incoming directory')
                    else:
                        shutil.copy2(input_directory + '/' + input_file,working_directory + '/' + input_file)
                else:
                    shutil.copy2(input_directory + '/' + input_file,working_directory+'/'+input_file)
            if kernel.input_type == 'TM_DAFCAT':
                shutil.copy2(input_directory + '/' + input_file, working_directory + '/' + input_file)

    kernel.working_directory = working_directory

    os.chdir(kernel.working_directory)

    return (cwd, working_directory)