#!/usr/bin/env python3

import shutil
import os
import datetime
import json
import textwrap
import logging
import traceback
import glob


from pathlib import Path
from argparse import ArgumentParser, RawDescriptionHelpFormatter

from .core.director import director
from .core import email
from .core.config import configure_execution

from .utils.files import clean_temp_input

def main(config=False, debug=False, log=False, show_version=False):

    execution_dir = os.getcwd()

    with open(os.path.dirname(__file__) + '/config/version',
              'r') as f:
        for line in f:
            version = line

    if not config:
        parser = ArgumentParser(formatter_class=RawDescriptionHelpFormatter,
                                description = textwrap.dedent('''\
                                
   ___   ___  _________       Auxiliary Data Conversion System next generation
  / _ | / _ \/ ___/ __/__  ___ _          
 / __ |/ // / /___\ \/ _ \/ _ `/         a command-line utility  program  that                      
/_/ |_/____/\___/___/_//_/\_, /          converts auxiliary  data  into  SPICE
                         /___/           Kernels for the European Space Agency
                              v{}


'''.format(version)),
        epilog = '''

   Source and documentation is available here: 
      https://repos.cosmos.esa.int/socci/projects/SPICE/repos/adcsng

   developed and maintained by the
    __   __   __      __   __     __   ___     __   ___  __          __   ___
   /__\ /__` '__\    /__` |__) | /  ` |__     /__` |__  |__) \  / | /  ` |__
   \__, .__/ \__/    .__/ |    | \__, |___    .__/ |___ |  \  \/  | \__, |___

   esa_spice@sciops.esa.int     for  Mars-Express // ExoMars2016 // ExoMarsRSP 
   http://spice.esac.esa.int         BepiColombo // Solar Orbiter

''')
        parser.add_argument('config', metavar='CONFIG', type=str, nargs='+',
                            help='Mission specific JSON configuration file')
        parser.add_argument('-v', '--version',
                        help='Display the version of ADCSng',
                        action='store_true')
        parser.add_argument('-m', '--metakernel',
                        help='Execute ADCSng to generate meta-kernel only',
                        action='store_true')
        parser.add_argument('-l', '--log',
                        help='Prompt log during execution',
                        action='store_true')
        parser.add_argument('-d', '--debug',
                        help='Run in debug mode; run files not cleaned-up, etc.',
                        action='store_true')
        args = parser.parse_args()

        if args.version or show_version:
            return version

        if args.config:
            config = args.config[0]

        if args.log:
            log = True

        if args.debug:
            debug = True

    #
    # We load the configuration
    #
    try:
        with open(config) as f:
            try:
                config = json.load(f)
            except:
                error_message = str(traceback.format_exc())
                print("Error: The ADCSng JSON configuration file has syntactical errors.")
                print(error_message)
                raise
    except:
        #
        # This needs to be completed by a validation of the JSON file
        #
        print('Error: The ADCSng configuration file has not been provided.')
        raise

    #
    # Lock file generation & check
    #
    if not 'lock_file' in config:
        lock_file = 'adcsng_{}.lock'.format(config['mission'].lower())
    else:
        lock_file = config['lock_file']
        path = os.sep.join(config['lock_file'].split(os.sep)[:-2])
        if not os.path.exists(path):
            lock_message = 'Error: Lock file directory: {} does not exist.'.format(path)
            print(lock_message)
            try:
                if 'email' in config.keys():
                    if config['email'][0]['send_email'] == "True":
                        email_body = lock_message

                        email.send_status_email(config, body_text=email_body,
                                                error=True)
            except:
                pass
                raise

    if os.path.isfile(lock_file):
        lock_message = 'Warning: ADCSng being executed? Lock file {} present.'.format(lock_file)
        print(lock_message)

        try:
            if 'email' in config.keys():
                if config['email'][0]['send_email'] == "True":
                    email_body = lock_message

                email.send_status_email(config, body_text=email_body,
                                        error=True)
        except:
            pass

        raise NameError(lock_message)
    elif not debug:
        Path(lock_file).touch()

    try:
        if args.metakernel:
            config['mkgen'] = 'Only'
    except:
        pass


    #
    # We first provide the generic configuration for the execution and
    # check the directory structure
    #
    try:
        setup, adcsng_setup, directories = configure_execution(config)
    except ValueError as e:
        print(e)
        try:
            os.remove(lock_file)
        except:
            pass
        raise

    #
    # Create the log for the run
    # We first need to check log file rotation
    #
    cwd = os.getcwd()
    os.chdir(config['log_dir'])
    log_files = glob.glob('adcsng_*')
    log_files.sort(reverse=True)
    os.chdir(cwd)

    try:
        rotation = config['log_rotation'].lower()
    except:
        rotation = 'monthly'

    today = datetime.date.today()
    try:
        for file in log_files:
            if rotation in file:
                latest_log = log_files[-1]
                break

        latest_log_date = latest_log.split('_')[-2]

        today_datetime = datetime.date.today()
        latest_log_datetime = datetime.datetime.strptime(latest_log_date,"%Y%m%d")

        current_year = today_datetime.isocalendar()[0]
        log_year = latest_log_datetime.isocalendar()[0]

        if rotation == 'weekly':
            current_week = today_datetime.isocalendar()[1]
            log_week = latest_log_datetime.isocalendar()[1]
            if current_week == log_week and log_year == current_year:
                date = latest_log_date
            else:
                raise
        elif rotation == 'daily':
            today = datetime.date.today()
            if today.strftime("%Y%m%d") == latest_log_date and log_year == current_year:
                date = latest_log_date
            else:
                raise
        elif rotation == 'monthly':
            log_month = latest_log_date[4:6]
            if today.strftime("%m") == log_month and log_year == current_year:
                date = latest_log_date
            else:
                raise
    except:
        date = today.strftime("%Y%m%d")


    log_file = config['log_dir'] + os.sep + f'adcsng_{date}_{rotation}.log'

    logger = logging.getLogger()
    logger.setLevel(logging.INFO)

    log_format = '%(asctime)s.%(msecs)03d %(levelname)-8s %(module)-16s %(message)s'
    date_format = '%Y-%m-%dT%H:%M:%S'

    fh = logging.FileHandler(log_file)
    fh.setLevel(logging.INFO)
    formatter = logging.Formatter(log_format,date_format)
    fh.setFormatter(formatter)
    logger.addHandler(fh)

    th = logging.FileHandler(config['temp_dir'] + '/temp.log')
    th.setLevel(logging.INFO)
    formatter = logging.Formatter(log_format,date_format)
    th.setFormatter(formatter)
    logger.addHandler(th)

    if log:
        ch = logging.StreamHandler()
        ch.setLevel(logging.INFO)
        formatter = logging.Formatter('%(message)s')
        ch.setFormatter(formatter)
        logger.addHandler(ch)

    try:
        #
        # To Execution
        #
        tcp_present = director(config, setup, adcsng_setup, directories, debug, log_file)

    except:
        email_body = ''
        error_message = str(traceback.format_exc())
        for line in error_message.split('\n'):
            logging.fatal(line)

        try:
            if 'email' in config.keys():
                if config['email'][0]['send_email'] == "True":
                    with open(directories.temp + '/temp.log', 'r') as l:
                        for line in l:
                            email_body += line

                    email.send_status_email(config, body_text=email_body, error=True)
        except:
            pass

        #
        # Cleaned up a failed execution
        #
        os.chdir(execution_dir)
        if not debug:
            os.remove(lock_file)
        
            for the_file in os.listdir(config['temp_dir']):
                file_path = os.path.join(config['temp_dir'], the_file)
                try:
                    if os.path.isfile(file_path):
                        os.unlink(file_path)
                    elif os.path.isdir(file_path): shutil.rmtree(file_path)
                except Exception as e:
                    print(e)

        clean_temp_input(config['source_dir'],debug=debug)

        print('ADCSng v{} for {} FAILED TO EXECUTE'.format(version, config[
            'mission']))
        raise

    #
    # The daily log has to distinguish in between a SCLK execution and a non
    # SCLK execution.
    #
    if directories.exec_log:
        with open(directories.temp + '/temp.log', 'r') as f:
            log_text = f.read()
        if log_text:
            if tcp_present: sclk_exec = 'tcp_'
            else: sclk_exec = 'aux_'

            shutil.copy2(directories.temp + '/temp.log',
                     directories.exec_log+'/adcsng_{}{}.eop'.format(sclk_exec,
                     datetime.datetime.now().strftime('%Y%m%dT%H%M%S')))

    os.remove(directories.temp + '/temp.log')

    #
    # Lock file removal
    #
    if not debug:
        os.chdir(execution_dir)
        os.remove(lock_file)

    return

