import spiceypy
from spiceypy import support_types as stypes

def gfroi(roi, start, finish, debug=False):

    MAXIVL = 10000
    MAXWIN = 2 * MAXIVL

    interval_start = spiceypy.utc2et(start)
    interval_finish = spiceypy.utc2et(finish)

    intervals = []


    # Initialize the "confinement" window with the interval
    # over which we'll conduct the search.
    cnfine = stypes.SPICEDOUBLE_CELL(2)
    spiceypy.wninsd(interval_start, interval_finish, cnfine)

    #
    # In the call below, the maximum number of window
    # intervals gfposc can store internally is set to MAXIVL.
    # We set the cell size to MAXWIN to achieve this.
    #
    riswin = stypes.SPICEDOUBLE_CELL(MAXWIN)

    #
    # First we look for the intervals in which the ROI is occulted by
    # Ganymede
    #
    #   VARIABLE        I/O  DESCRIPTION
    #   --------------- ---  -------------------------------------------------
    #   SPICE_GF_CNVTOL  P   Convergence tolerance.
    #   occtyp           I   Type of occultation.
    #   front            I   Name of body occulting the other.
    #   fshape           I   Type of shape model used for front body.
    #   fframe           I   Body-fixed, body-centered frame for front body.
    #   back             I   Name of body occulted by the other.
    #   bshape           I   Type of shape model used for back body.
    #   bframe           I   Body-fixed, body-centered frame for back body.
    #   abcorr           I   Aberration correction flag.
    #   obsrvr           I   Name of the observing body.
    #   step             I   Step size in seconds for finding occultation
    #                        events.
    #   cnfine          I-O  SPICE window to which the search is restricted.
    #   result           O   SPICE window containing results.
    #
    spiceypy.gfoclt('ANY', 'GANYMEDE', 'ELLIPSOID', 'IAU_GANYMEDE',
                    roi, 'ELLIPSOID', roi, 'NONE',
                    'JUICE', 60, cnfine, riswin)

    #
    # We need to obtain the difference in between the original window
    # and the resulting occutltations
    #
    difwin = spiceypy.wndifd(cnfine, riswin)

    if debug:
        difwin = cnfine

    riswin = stypes.SPICEDOUBLE_CELL(MAXWIN)
    #   VARIABLE  I/O  DESCRIPTION
    #   --------  ---  --------------------------------------------------
    #   MARGIN     P   Minimum complement of FOV cone angle.
    #   LBCELL     P   SPICE Cell lower bound.
    #   CNVTOL     P   Convergence tolerance.
    #   MAXVRT     P   Maximum number of FOV boundary vertices.
    #   INST       I   Name of the instrument.
    #   TARGET     I   Name of the target body.
    #   TSHAPE     I   Type of shape model used for target body.
    #   TFRAME     I   Body-fixed, body-centered frame for target body.
    #   ABCORR     I   Aberration correction flag.
    #   OBSRVR     I   Name of the observing body.
    #   STEP       I   Step size in seconds for finding FOV events.
    #   CNFINE     I   SPICE window to which the search is restricted.
    #   RESULT     O   SPICE window containing results.
    spiceypy.gftfov('JUICE_JANUS', roi, 'ELLIPSOID', roi, 'NONE', 'JUICE', 60,
                    difwin,  riswin)

    #
    # The function wncard returns the number of intervals
    # in a SPICE window.
    #
    winsiz = spiceypy.wncard(riswin)


    if winsiz == 0:
        print('No events were found.')

    else:

        #
        # Display the visibility time periods.
        #
        print(f'{roi} on JANUS FOV times of JUICE')

        for i in range(winsiz):
            #
            # Fetch the start and stop times of
            # the ith interval from the search result
            # window riswin.
            #
            [intbeg, intend] = spiceypy.wnfetd(riswin, i)

            #
            # Convert the rise time to a TDB calendar string.
            #
            et_rise = intbeg
            et_set = intend

            #
            # Generate a Time Window with the rise and set times
            #
            utc_rise = spiceypy.et2utc(et_rise, 'ISOC', 3)
            utc_set = spiceypy.et2utc(et_set, 'ISOC', 3)
            intervals.append([utc_rise, utc_set])

            print(utc_rise, utc_set)

    return intervals


start_utc = '2033-01-23T09:00:00'
finish_utc = '2033-01-23T10:00:00'
spiceypy.furnsh(
    '/Users/mcosta/SPICE/JUICE/kernels/mk/juice_crema_3_0_ops_local.tm')

for i in range(1,110,1):
    gfroi(f'JUICE_ROI_{i}', start_utc, finish_utc)